/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 */

#include "cis.h"
#include "cis_clnt.h"
#include "cis_rec.h"

#define progname "xcis"

#define DEF_CONFIG_FILENAME         "./.xcisrc"
#define DEF_INTERVAL                1.
#define DEF_CPU_SCALE               80
#define DEF_NET_SCALE               800
#define DEF_TIME_SCALE              100
#define SOCK_INTERVAL               5
#define DELAY                       150

struct user_info {

        char           *name;
        unsigned short  id;
        char           *fullname;
        struct user_info *next;
	
};

struct cisinfo {
        char *name;
        CLIENT *handle;
        struct cis_hostinfo **htab;
	int nhosts;
	struct user_info *users;

	struct cis_hostinfo *hlist;
	int hlist_length;

	struct cisinfo *next;

};

struct host_info {
        int flag;
        int links_available;
        
        struct cisinfo *cis;

        float interval;
        unsigned short uid;
        
        struct cis_hostinfo *hostinfo, *old_hostinfo;
        float maxspeed;
	char *color;
	unsigned long pixel;
	
        unsigned long send_rate;
        unsigned long recv_rate;

        int hostlist_row;
        
        int proc_number;
        int proc_offset;
        struct cis_procinfo **proclist;
        
        int link_number;
        int link_offset;
        struct link_info **linklist;

        int dev_number;
        int dev_offset;

        int nproc;
        struct cis_procinfo *proctab;
        
        int nsock;
        struct cis_sockinfo *socktab;
        
        int nsock_old;
        struct cis_sockinfo *socktab_old;

        int ndev;
        struct cis_netdevinfo *devtab;

        void *CPU_curve;

        char  hostinfo_updated;
        char  proc_updated;
        char  sock_updated;
        char  dev_updated;

	struct host_info *next;
};

struct link_info {

        char flag;
        char is_updated;

        short             type;
        unsigned int      src_pid;
        struct host_info *dst_host;
        unsigned int      dst_pid;
        struct in_addr    dst_addr;
        unsigned short    dst_port;
        
        unsigned long sent;
        unsigned long rcvd;
};

struct rec_info {
        char               *filename;
	FILE               *file;
	char               *path;
	struct cis_header  *header;
        struct timeval     start_time;
	unsigned long      length, offset, start, end;
        int                counter;
};

enum {
	REC_CURVE_LEN = CIS_REC_END+1,
	REC_CURVE,
};

struct proc_info {
	struct cis_procinfo *prev, *next, *children;
	struct user_info *user;
	unsigned char flags;
};

struct sock_info {
	struct link_info *link;
	unsigned char flags;
};

struct dev_info {
	void *name;
	unsigned char flags;
};

extern struct host_info *hostlist;
extern struct cisinfo  *cislist;

void fatal_err (const char *fmt, ...);
void warning (const char *fmt, ...);
int bprintf (char *buff, char *fmt, ...);
int kprintf (char *buff, char *fmt, ...);
int tprintf (char *buff, char *fmt, ...);
void getparams (int argc, char **argv);
void parse_configfile (void);
struct cisinfo *find_cis (char *name);
struct host_info *find_host(struct in_addr caddr, struct in_addr addr);
void update_time (void);
void proc_tree_sort (struct host_info *host);
void clear_host_data (struct host_info *host);
void clear_data (void);
void allarm_handler (int sig);

#ifndef _WITHOUT_X

#include <Xm/Xm.h>
#if XmVersion >= 1002
#include <Xm/PrimitiveP.h>
#endif
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/DrawnB.h>
#include <Xm/Label.h>
#include <Xm/Scale.h>
#include <Xm/TextF.h>
#include <Xm/FileSB.h>

#include <XmL/Folder.h>
#include <XmL/Tree.h>
#include <XmL/Progress.h>

#include <Xm/DrawnB.h>

#include <X11/Intrinsic.h>
#include "XY.h"

enum {
        HOSTLIST,
        PROCLIST,
        LINKLIST,
        DEVLIST,
        CPUPLOT,
        NETPLOT,
        NFOLDERS,
};

enum {
        ONLINE,
        RECORD,
        REPLAY,
};

enum {
        PROC_SORT_PID,
        PROC_SORT_TREE,
        PROC_SORT_PCPU,
        PROC_SORT_RSS,
        PROC_SORT_VM,
        PROC_SORT_MAJFLT,
};

enum {
        HOST_SORT_NONE,
        HOST_SORT_PERF_MAX,
        HOST_SORT_PERF_AVAIL,
};

enum {
        HOST_COL_NAME = 0,
        HOST_COL_CHECK,
        HOST_COL_PERF_MAX,
        HOST_COL_PERF_AVAIL,
        HOST_COL_RAM_TOTAL,
        HOST_COL_RAM_FREE,
        HOST_COL_SWAP_TOTAL,
        HOST_COL_SWAP_FREE,
        HOST_COL_NPROC,
	HOST_COL_CTX_SWTCH,
        HOST_COL_LOADAVG1,
        HOST_COL_LOADAVG5,
	HOST_COL_LOADAVG15,
	HOST_COL_SWAPRATE,
	HOST_COL_DISKRATE,
        HOST_COL_RELIABILITY,
        HOST_NCOLS,
};

enum {
        PROC_COL_CMD,
        PROC_COL_PID,
        PROC_COL_USER,
        PROC_COL_PRIO,
        PROC_COL_RSS,
        PROC_COL_VM,
        PROC_COL_PCPU,
        PROC_COL_MAJFLT,
        PROC_COL_READ,
        PROC_COL_WRITE,
        PROC_COL_UPTIME,
        PROC_COL_USEDTIME,
        PROC_NCOLS,
};

enum {
        LINK_COL_PID,
        LINK_COL_DSTHOST,
        LINK_COL_DSTPID,
        LINK_COL_TYPE,
        LINK_COL_SENT,
        LINK_COL_RCVD,
        LINK_NCOLS,
};

enum {
        DEV_COL_HOST,
        DEV_COL_IFACE,
        DEV_COL_STATUS,
        DEV_COL_TX,
        DEV_COL_RX,
        DEV_COL_COLL,
        DEV_NCOLS,
};

enum {
	DOT,
	CHECK,
	UNCHECK,
};

enum {
	PRINT,
	REC,
	REP,
	BEGIN,
	REW,
	STEP,
	FWD,
	END,
	NBUTTONS
};


#define PROC_EXPANDED 2
#define LINK_EXPANDED 4

extern Screen *screen;
extern XtAppContext app_context;
extern Display *display;
extern Widget Xcis;

extern Pixmap pixmaps[2];
extern Pixmap pixmasks[3];

extern Widget Folder;
extern Widget Time_field;
extern Widget Hostlist;
extern Widget Proclist;
extern Widget Linklist;
extern Widget Devlist;

extern Widget Buttons[NBUTTONS];
extern Widget File_selection;
extern Widget Progress_bar;
extern Widget CPU_plot;
extern Widget net_plot;

extern int nCPU_curves;
extern XYCurve *CPU_curves;

extern int nnet_curves;
extern XYCurve *net_curves;

extern struct timeval current_time, old_time;
extern char *CPU_units;
extern char *net_units;

extern int time_scale;
extern float CPU_scale;
extern float net_scale;

extern char myhostname[];
extern float interval;

extern int mode;
extern int active_folder;

extern int proclist_sort_type;
extern int hostlist_sort_type;

extern XtSignalId sigid;

char *string_to_text(XmString s);
void hostlist_sort (void);

void proctab_update (struct host_info *host, struct cis_procinfo *ptab, int len);
void socktab_update (struct host_info *host, struct cis_sockinfo *stab, int len);
void devtab_update (struct host_info *host, struct cis_netdevinfo *dtab, int len);

void hostlist_update (struct host_info *host);
void proclist_update (struct host_info *host);
void socklist_update (struct host_info *host);
void devlist_update  (struct host_info *host);
void linklist_update (void);

void CPU_curve_update (struct host_info *host);
void update_curves_time (int);

void monitor_error (struct host_info *host, int err);

XYCurve *find_curve (XYCurve *curves, int nCPU_curves, XmString name);

#define proc_priv(p) ((struct proc_info *) p->priv)
#define sock_priv(s) ((struct sock_info *) s->priv)
#define dev_priv(d)  ((struct dev_info *)  d->priv)


#define for_each_selected_host(h) \
	for (h = hostlist; h; h = h->next) \
		if (h->flag)

#define usectotval(A,B) { A.tv_sec  = (B) / 1000000L; \
        A.tv_usec = (B) % 1000000L; }

#define tvaltousec(A)   A.tv_sec * 1000000L + A.tv_usec

#define sectotval(A,B) { A.tv_sec  = (int) (B) ; \
        A.tv_usec = (int) (((B) - (int) (B)) * 1000000L); }

#define tvaltosec(A)    (A.tv_sec + (A.tv_usec / 1e6))

#define tvaludiff(A,B)    ((A.tv_sec  - B.tv_sec) * 1000000L + \
        A.tv_usec - B.tv_usec)


#endif


