/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * As a special exception you are allowed to link xcis executable with
 * Microline Widget Library and Plot Widgets library.
 */

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <locale.h>
#include <sys/types.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "xcis.h"
#include "replay.h"
#include "cb.h"

void create_window(int argc, char *argv[]);

struct {
	int hostlist:1;
	int proclist:1;
	int linklist:1;
	int devlist:1;
	int CPUplot :1;
} changed;

struct timeval current_time, old_time;

struct user_info *userlist = NULL;
struct cisinfo  *cislist = NULL;
struct host_info *hostlist = NULL, **hostlist_sorted = NULL;

int nhosts = 0;

char *perf_type = "";

char myhostname[MAXHOSTNAMELEN];

Display *display;
Widget Xcis;

Pixmap pixmaps[2];
Pixmap pixmasks[3];

float interval = DEF_INTERVAL;

int   time_scale = DEF_TIME_SCALE;
int   mypid;

int proclist_sort_type = PROC_SORT_TREE;
int hostlist_sort_type = HOST_SORT_NONE;

Screen *screen;
XtAppContext app_context;

Widget Folder;
Widget Time_field;
Widget Hostlist;
Widget Proclist;
Widget Linklist;
Widget Devlist;

Widget Buttons[NBUTTONS];

Widget File_selection;
Widget Progress_bar;
int proclist_size=0;
//Pixmap pixmap, pixmask;
XtSignalId sigid;

int mode = ONLINE;
int active_folder = 0;

int nCPU_curves = 0;

Widget CPU_plot;
XYCurve *CPU_curves = NULL;
float CPU_scale = DEF_CPU_SCALE;
char *CPU_units = NULL;

Widget net_plot;
int nnet_curves = 0;
XYCurve *net_curves = NULL;
float net_scale = DEF_NET_SCALE;
char *net_units = NULL;

unsigned char reliability = 0;

sigset_t sigblockset;

char *string_to_text(XmString s)
{
        XmStringContext ctxt;
        XmStringCharSet charset;
        XmStringDirection direction;
        Boolean sep, ret;
        char *text;

        XmStringInitContext(&ctxt, s);
        ret = XmStringGetNextSegment(ctxt, &text, &charset, &direction, &sep);
        XmStringFreeContext(ctxt);
        XtFree((char *)charset);

        if (!ret)
                return NULL;
        
        return text;
}

struct cisinfo *find_cis (char *name)
{
	struct cisinfo *cis;

	for (cis = cislist; cis; cis = cis->next)
		if (!strcmp (name, cis->name))
			return (cis);

	return (NULL);
}

struct host_info *find_host(struct in_addr caddr, struct in_addr addr)
{
	struct host_info *h;

	for (h = hostlist; h; h = h->next)
		if (h->hostinfo->caddr.s_addr == caddr.s_addr &&
		    h->hostinfo->addr.s_addr  == addr.s_addr)
			return h;

	for (h = hostlist; h; h = h->next)
		if (h->hostinfo->addr.s_addr == addr.s_addr)
			return h;

	return NULL;
}

struct user_info *find_user(struct host_info *host, unsigned short uid)
{
	struct user_info *u;

	if (host->cis)
		for (u = host->cis->users; u; u = u->next)
			if (u->id == uid)
				return u;

	for (u = userlist; u; u = u->next)
		if (u->id == uid)
			return u;

	return NULL;
}

XYCurve *find_curve (XYCurve *curves, int nCPU_curves, XmString name)
{
        for (; nCPU_curves; nCPU_curves--, curves++)
                if (curves->name == name)
                        return curves;
        return NULL;
}

struct cis_procinfo *proc_find (struct cis_procinfo *tab, int len, int pid)
{
        int i;
        for (i = 0; i < len; i++, tab++)
                if (tab->pid == pid)
                        return tab;
        return NULL;
}

struct cis_sockinfo *sock_find (struct cis_sockinfo *tab, int len, struct cis_sockinfo *s)
{
        int i;
        
        for (i = 0; i < len; i++, tab++)
                if (tab->type         == s->type &&
                    tab->saddr.s_addr == s->saddr.s_addr &&
                    tab->sport        == s->sport &&
                    tab->daddr.s_addr == s->daddr.s_addr &&
                    tab->dport        == s->dport)
                        return tab;
        return NULL;
}

struct cis_sockinfo *sock_find_dest (struct cis_sockinfo *tab, int len, struct cis_sockinfo *s)
{
        int i;
        
        for (i = 0; i < len; i++, tab++)
                if (tab->type         == s->type &&
                    tab->saddr.s_addr == s->daddr.s_addr &&
                    tab->sport        == s->dport /*&&
                    tab->daddr.s_addr == s->saddr.s_addr &&
                    tab->dport        == s->sport*/)
                        return tab;
        return NULL;
}

struct link_info *link_find (struct host_info *src_host, struct cis_sockinfo *s1, struct host_info *dst_host, struct cis_sockinfo *s2)
{
        struct link_info *link, **l;

        for (l = src_host->linklist; l < src_host->linklist + src_host->link_number; l++) {
                link = *l;
                if (link->src_pid != s1->pid)
                        continue;
                if (link->type != s1->type)
                        continue;
                if (dst_host) {
                        if (link->dst_host != dst_host)
                                continue;
                } else if (link->dst_addr.s_addr != s1->daddr.s_addr)
                        continue;

                if (s2) {
                        if (s2->pid != link->dst_pid)
                                continue;
                } else if (s1->dport != link->dst_port)
                        continue;

                return link;
        }
        return NULL;
}

struct cis_netdevinfo *dev_find (struct cis_netdevinfo *tab, int len, char *name)
{
        int i;
        for (i = 0; i < len; i++, tab++)
                if (!strncmp (tab->name, name, CIS_DEVNAMELEN))
                        return tab;
        return NULL;
}


void refresh_screen (void)
{
        while (XtAppPending (app_context))
                XtAppProcessEvent(app_context, XtIMAll);
}

void update_time (void)
{
        time_t t = current_time.tv_sec;
        struct tm *tmp;
        static char time_buff[128];
        XmString s;

	tmp = localtime (&t);
        strftime (time_buff, 128, "%x %X", tmp);
        s = XmStringCreateSimple (time_buff);
	XmUpdateDisplay (Xcis);
	refresh_screen();
	XtVaSetValues (Time_field, XmNlabelString, s, NULL);
	XmStringFree(s);
	XmUpdateDisplay (Xcis);
	refresh_screen();
}

void update_curve_time (XYCurve *curve, float tdiff)
{
        XYPoint *p;

        for (p = curve->points; p < curve->points + curve->nPoints; p++)
                p->x -= tdiff;

}

void add_value_to_curve (XYCurve *curve, float value, float time)
{
        XYPoint *p;

        if (!curve)
                return;

	if (curve->nPoints == time_scale)
		memmove (curve->points, curve->points+1, (time_scale-1)*sizeof(XYPoint));
	else
		curve->nPoints++;

	p = curve->points + curve->nPoints - 1;

        p->y = value;
        p->x = time;
}

void CPU_curve_update (struct host_info *host)
{
	float tdiff = tvaltosec (current_time) - tvaltosec(old_time);
        float value = (host->hostinfo->CPU_available*host->hostinfo->performance) / 1e4;

        if (!host->CPU_curve) return;

        add_value_to_curve (host->CPU_curve, value, tdiff);
}

void net_curve_update (XmString name, unsigned long value)
{
        float tdiff = tvaltosec (current_time) - tvaltosec(old_time);

        add_value_to_curve (find_curve (net_curves, nnet_curves, name),
                            value/1024., tdiff);
}

void update_curves_time (int show)
{
        struct host_info *h;
        struct cis_netdevinfo *dev;
	float tdiff = tvaltosec (current_time) - tvaltosec(old_time);
	XYCurve *c;

        if (tdiff > 0.)
		for_each_selected_host(h) {
			update_curve_time (h->CPU_curve, tdiff);

			for (dev = h->devtab; dev < h->devtab + h->ndev; dev++) {
				c = find_curve (net_curves, nnet_curves, dev_priv(dev)->name);
				update_curve_time (c, tdiff);
			}
		}
        old_time = current_time;

	if (show) {
		XYUpdateCurveData (CPU_plot, CPU_curves, XY_NO_RESCALE, True);
		XYUpdateCurveData (net_plot, net_curves, XY_NO_RESCALE, True);
	}
}



void hostlist_update (struct host_info *host)
{
        int row = host->hostlist_row;
        struct cis_hostinfo *hi = host->hostinfo;
	static char buf[128];

        if (!hi || hi->status != HOST_AVAILABLE)
                sprintf (buf, "%s|%5.2f |not|available||||||| ",
			 hi->name,
			 hi->performance);
	else
		/*                Performance      Ram     Swap  NProc Ctx_swtch  Avg.load        */
		sprintf (buf, "%s|%5.2f |%5.2f |%ld |%ld |%ld |%ld |%d |%ld |%5.3f |%5.3f |%5.3f |%s",
			 hi->name,
			 hi->performance,
			 (hi->CPU_available * hi->performance)/ 1e4,
			 hi->totalram,
			 hi->freeram + hi->bufferram + hi->cachedram,
			 hi->totalswap,
			 hi->freeswap,
                         hi->procnum,
                         hi->ctx_swtch,
			 hi->loads[0]/100., hi->loads[1]/100., hi->loads[2]/100.,
			 rel_types[hi->reliability]);

        XmLGridSetStringsPos (Hostlist, XmCONTENT, row, XmCONTENT, 1, buf);
        host->hostinfo_updated = FALSE;
}

int host_compare_fn (void *user_data, void *l,void *r)
{
        int type = *(int *)user_data;
        struct host_info *left  = *(struct host_info **) l;
        struct host_info *right = *(struct host_info **) r;

        switch (type) {
        case HOST_SORT_PERF_MAX:
		return (right->hostinfo->performance*100 - left->hostinfo->performance*100);
        case HOST_SORT_PERF_AVAIL:
                if (!left->hostinfo) return -1;
                if (!right->hostinfo) return 1;
                return ((right->hostinfo->CPU_available/1e4) * right->hostinfo->performance -
                        (left->hostinfo->CPU_available/1e4) * left->hostinfo->performance);
        }
        return 0;
}

void hostlist_sort (void)
{
        static int *newlist = NULL;
        static struct host_info *h, **oldlist = NULL;
        int i, sort_type = hostlist_sort_type;

        if (!newlist)
                newlist = malloc (nhosts*sizeof(int));
        if (!oldlist)
                oldlist = malloc (nhosts*sizeof(struct host_info *));

	memcpy (oldlist, hostlist_sorted, nhosts*sizeof(struct host_info *));

	if (sort_type == HOST_SORT_NONE)
		for (h = hostlist, i = 0; h; h = h->next, i++)
			hostlist_sorted[i] = h;

        XmLSort (hostlist_sorted, nhosts, sizeof (struct host_info *), host_compare_fn, &sort_type);

        for (i = 0; i < nhosts; i++)
                newlist[i] = hostlist_sorted[i]->hostlist_row;

        if (memcmp (oldlist, hostlist_sorted, nhosts*sizeof(struct host_info *)))
                XmLGridReorderRows (Hostlist, newlist, 0, nhosts);

        for (i = 0; i < nhosts; i++)
                hostlist_sorted[i]->hostlist_row = i;
}


int proc_compare_fn (void *user_data, void *l,void *r)
{
        int type = *(int *)user_data;
        struct cis_procinfo *left  = *(struct cis_procinfo **) l;
        struct cis_procinfo *right = *(struct cis_procinfo **) r;
        int result;

        switch (type) {
        case PROC_SORT_TREE:
                if ((result = left->ppid - right->ppid))
                        return (result);
        case PROC_SORT_PID:
                return (left->pid - right->pid);
                break;
        case PROC_SORT_PCPU:
                return (right->pCPU - left->pCPU);
                break;
        case PROC_SORT_RSS:
                return (right->rss - left->rss);
                break;
        case PROC_SORT_VM:
                return (right->vm - left->vm);
                break;
	case PROC_SORT_MAJFLT:
                return (right->majflt - left->majflt);
                break;
        }
        return 0;
}

int proc_search_fn (const void *key, const void *memb)
{
        int ppid = *(int *) key;
        struct cis_procinfo *proc = *(struct cis_procinfo **) memb;

        return (ppid - proc->pid);
}


void proc_tree_change(int row, int column, struct cis_procinfo *p)
{
        static char buf[128];

        
        switch (column) {
        case PROC_COL_CMD:
                sprintf (buf, "%s", p->cmd);
                break;
        case PROC_COL_USER:
                if (p->priv)
                        sprintf (buf, " %s", ((struct user_info *) p->priv)->name);
                else
                        sprintf (buf, " %d", p->uid);
                break;
        case PROC_COL_PRIO:
                sprintf (buf, "%ld ", p->priority);
                break;
        case PROC_COL_RSS:
                lprintf (buf, "%ld ", p->rss);
                break;
        case PROC_COL_VM:
                lprintf (buf, "%ld ", p->vm);
                break;
        case PROC_COL_PCPU:
                sprintf (buf, "%5.2f ", p->pCPU/100.);
                break;
        case PROC_COL_MAJFLT:
                lprintf (buf, "%ld ", p->majflt);
                break;
        case PROC_COL_READ:
                lprintf (buf, "%ld ", p->rd_bytes);
                break;
        case PROC_COL_WRITE:
                lprintf (buf, "%ld ", p->wr_bytes);
                break;
        }
        XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, column, buf);
}

void proc_tree_sort (struct host_info *host)
{
        static struct cis_procinfo **oldtree = NULL;
        static struct cis_procinfo **newtree = NULL;
        static int *array = NULL;
	static int *levels = NULL;
	struct proc_info *priv;

        struct cis_procinfo *p, **pp, *root, *last;
        int i,j;
        XmLGridRow rowp;

        if (!host->proc_number)
                return;
        
        oldtree  = realloc (oldtree, host->proc_number * sizeof (struct cis_procinfo *));
        newtree  = realloc (newtree, host->proc_number * sizeof (struct cis_procinfo *));
        array    = realloc (array, host->proc_number * sizeof (int));
        levels   = realloc (levels, host->proc_number * sizeof (int));

        /*
         * Get the last state.
         */
        for (i = 0; i < host->proc_number; i++) {
                rowp = XmLGridGetRow (Proclist, XmCONTENT, host->proc_offset+i+1);
                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNrowUserData, &oldtree[i],
                              XmNrowLevel, &levels[i],
                              NULL);

		proc_priv(oldtree[i])->flags = 1; /* level for all sorts except tree */
        }
        /*
         * Create new tree.
         */
        memcpy (newtree, oldtree, host->proc_number*sizeof (struct cis_procinfo *));
        XmLSort (newtree, host->proc_number, sizeof (struct cis_procinfo *), proc_compare_fn, &proclist_sort_type);

        if (proclist_sort_type == PROC_SORT_TREE) {
                int level;
                
                /*
                 * Clean the tree.
                 */
		for (i = 0; i < host->proc_number; i++) {
			priv = proc_priv(newtree[i]);

			priv->flags = FALSE;
                        priv->next = NULL;
                        priv->prev = NULL;
                        priv->children = NULL;
                }
                /*
                 * Create branches.
                 */
                for (i = 0; i < host->proc_number-1; i++)
                        if (newtree[i]->ppid == newtree[i+1]->ppid) {
				proc_priv(newtree[i])->next = newtree[i+1];
                                proc_priv(newtree[i+1])->prev = newtree[i];
				proc_priv(newtree[i+1])->flags = TRUE;
                        }

                /*
                 * Connect branches together.
                 */
                for (i = 0; i < host->proc_number; i++) {
			if (proc_priv(host->proclist[i])->flags)
                                continue;

                        if ((pp = bsearch (&host->proclist[i]->ppid, host->proclist, host->proc_number,
                                           sizeof (struct cis_procinfo *), proc_search_fn))) {
                                p = *pp;

				proc_priv(p)->children = host->proclist[i];
                                proc_priv(host->proclist[i])->prev = p;
                                proc_priv(host->proclist[i])->flags = TRUE;
                        }
                }
                /*
                 * Finally, connect together level 0.
                 */
                root = NULL;
                last = NULL;
                for (i = 0; i < host->proc_number; i++) {
			if (proc_priv(host->proclist[i])->flags)
                                continue;

                        p = host->proclist[i];

                        if (!root)
                                root = last = p;
                        else {
                                proc_priv(last)->next = p;
				proc_priv(p)->prev = last;
                        }
			while (proc_priv(last)->next)
                                last = proc_priv(last)->next;

                        proc_priv(p)->flags = TRUE;
                }

                /*
                 * Store the tree in newtree. Flags are used to store new
                 * levels (why not).
                 */
                p = root;
                i = 0;
                level = 1;
                while (p) {
                        newtree[i++] = p;
                        proc_priv(p)->flags = level;
                        if (i == host->proc_number)
                                break;

                        if (proc_priv(p)->children) {
                                p = proc_priv(p)->children;
                                level++;
                                continue;
                        }
                        if (proc_priv(p)->next) {
                                p = proc_priv(p)->next;
                                continue;
                        }
                        while (proc_priv(p)->prev) {
				if (proc_priv(proc_priv(p)->prev)->children == p) {
                                        level--;
                                        if (proc_priv(proc_priv(p)->prev)->next)
						break;
				}
				p = proc_priv(p)->prev;
			}

                        if (proc_priv(p)->prev)
                                p = proc_priv(proc_priv(p)->prev)->next;
                }
        }
        /*
         * No change.
         */
        if (!memcmp (&oldtree, &newtree, host->proc_number * sizeof (struct cis_procinfo *)))
                return;

        /*
         * Create a table for reordering rows in process tree.
         */
        for (i = 0; i < host->proc_number; i++)
                for (j = 0; j < host->proc_number; j++)
                        if (newtree[i] == oldtree[j])
                                array[i] = host->proc_offset+j+1;

        XmLGridReorderRows (Proclist, array, host->proc_offset+1, host->proc_number);

        /*
         * Set the levels.
         */
        for (i = 0; i < host->proc_number; i++)
                if (proc_priv(newtree[i])->flags != levels[array[i]-host->proc_offset-1])
                        XtVaSetValues(Proclist,
                                      XmNrow, host->proc_offset+i+1,
                                      XmNrowLevel, proc_priv(newtree[i])->flags,
                                      NULL);
}

void proctab_update (struct host_info *host, struct cis_procinfo *ptab, int len)
{
        free (host->proctab);

        host->proctab = ptab;
        host->nproc = len;

        host->proc_updated = TRUE;
}

#define user_match(a,b) (a == 0xffff || (a ? (a == b) : (b > MIN_USER_UID)))

void proclist_update (struct host_info *host)
{
	struct cis_procinfo *p, *newp, **new_proclist;
	struct proc_info *priv;
        struct host_info *h;
        int row, n = 0;
        int redraw_tree = FALSE;
        XmLGridRow rowp;
        XmLGridColumn colp1, colp2;
	char buf[128], *c;
        static int sort = PROC_SORT_PID;
        XmString s;

	for (p = host->proctab; p < host->proctab + host->nproc; p++) {
		p->priv = NULL;
		if (user_match(host->uid, p->uid)) n++;
	}

        if (n != host->proc_number) {
                new_proclist = realloc (host->proclist, n * sizeof (struct cis_procinfo *));
                if (n && !new_proclist)
			return;
		host->proclist = new_proclist;

		for (h = host->next; h; h = h->next)
                        h->proc_offset += n - host->proc_number;
	}

        colp1 = XmLGridGetColumn (Proclist, XmCONTENT, PROC_COL_UPTIME);
        colp2 = XmLGridGetColumn (Proclist, XmCONTENT, PROC_COL_USEDTIME);

	for (row = host->proc_offset+1; row < host->proc_offset+host->proc_number+1;) {
                rowp = XmLGridGetRow (Proclist, XmCONTENT, row);
                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNrowUserData, &p,
                              NULL);

                if (!(newp = proc_find (host->proctab, host->nproc, p->pid))) {

			free (p->priv);
			free (p);
			XmLGridDeleteRows (Proclist, XmCONTENT, row, 1);
                        if (proclist_sort_type == PROC_SORT_TREE)
                                redraw_tree = TRUE;

                        host->proc_number--;

                        continue;
                }

                host->proclist[row - host->proc_offset - 1] = p;

                if (strcmp (newp->cmd, p->cmd))
                        proc_tree_change (row, PROC_COL_CMD , newp);

                if (newp->ppid != p->ppid && proclist_sort_type == PROC_SORT_TREE)
                        redraw_tree = TRUE;

                if (newp->uid != p->uid)
                        proc_tree_change (row, PROC_COL_USER , newp);

                if (newp->priority != p->priority)
                        proc_tree_change (row, PROC_COL_PRIO, newp);

                if (newp->rss != p->rss) {
                        proc_tree_change (row, PROC_COL_RSS , newp);
                        if (proclist_sort_type == PROC_SORT_RSS)
                                redraw_tree = TRUE;
                }
                if (newp->vm != p->vm) {
                        proc_tree_change (row, PROC_COL_VM , newp);
                        if (proclist_sort_type == PROC_SORT_VM)
                                redraw_tree = TRUE;
                }
                if (newp->pCPU != p->pCPU) {
                        proc_tree_change (row, PROC_COL_PCPU, newp);
                        if (proclist_sort_type == PROC_SORT_PCPU)
                                redraw_tree = TRUE;
                }
		if (newp->majflt != p->majflt) {
			proc_tree_change (row, PROC_COL_MAJFLT, newp);
                        if (proclist_sort_type == PROC_SORT_MAJFLT)
				redraw_tree = TRUE;
		}
                if (newp->rd_bytes != p->rd_bytes)
			proc_tree_change (row, PROC_COL_READ, newp);
                if (newp->wr_bytes != p->wr_bytes)
			proc_tree_change (row, PROC_COL_WRITE, newp);

                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNcolumnPtr, colp1,
                              XmNcellString, &s,
                              NULL);
		c = string_to_text(s);
                tprintf (buf, "%ld ", current_time.tv_sec - p->start_time);
                if (strcmp(c, buf))
                        XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, PROC_COL_UPTIME, buf);
		XmStringFree (s);
                free (c);

                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNcolumnPtr, colp2,
                              XmNcellString, &s,
                              NULL);
                c = string_to_text(s);
                tprintf (buf, "%ld ", p->utime+p->stime);
                if (strcmp(c, buf))
                        XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, PROC_COL_USEDTIME, buf);
		XmStringFree (s);
		free (c);
                
                newp->priv = p->priv;
                *p = *newp;

                row++;
	}

	/*
	 * Insert new processes into the process tree.
	 */
	for (p = host->proctab; p < host->proctab + host->nproc; p++) {
		if (p->priv) continue;
		if (!user_match(host->uid, p->uid)) continue;

		priv = calloc (1, sizeof (struct proc_info));
		priv->user = find_user (host, p->uid);

		newp = malloc (sizeof (struct cis_procinfo));
		*newp = *p;
		newp->priv = priv;

                XmLTreeAddRow (Proclist, 1, False, True, row, pixmaps[DOT], pixmasks[DOT], NULL);
                XtVaSetValues (Proclist,
                               XmNrow, row,
                               XmNrowExpands, False,
                               XmNrowUserData, newp,
                               NULL);

                c = buf;
                c += sprintf (c, "%s|%d ", p->cmd, p->pid);
                if (priv->user)
			c += sprintf (c, "| %s", priv->user->name);
		else
			c += sprintf (c, "| %d", p->uid);

		c += sprintf (c, "|%ld ", p->priority);
		c += lprintf (c, "|%ld |%ld ", p->rss, p->vm);
		c += sprintf (c, "|%5.2f ", p->pCPU/100.);
                c += lprintf (c, "|%ld ", p->majflt);
                c += lprintf (c, "|%ld ", p->rd_bytes);
                c += lprintf (c, "|%ld ", p->wr_bytes);
                c += tprintf (c, "|%ld ", current_time.tv_sec - p->start_time);
                c += tprintf (c, "|%ld ", p->utime + p->stime);

		XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, 0, buf);

                redraw_tree = TRUE;

		host->proclist[row - host->proc_offset - 1] = newp;

		row++;
		host->proc_number++;
	}

	XmLSort (host->proclist, host->proc_number, sizeof (struct cis_procinfo *), proc_compare_fn, &sort);

	if (redraw_tree)
                proc_tree_sort (host);

        host->proc_updated = FALSE;
}


void socktab_update (struct host_info *host, struct cis_sockinfo *stab, int len)
{
        free (host->socktab);

        host->socktab = stab;
        host->nsock = len;

        host->sock_updated = TRUE;
}

void socklist_update (struct host_info *host)
{
        struct cis_sockinfo *s, *socket, *oldtab;
        struct link_info **l, *link;
        struct host_info *h;
        unsigned long diff;

        host->send_rate = 0;
        host->recv_rate = 0;
        
        for (l = host->linklist; l < host->linklist + host->link_number; l++) {
                (*l)->flag = FALSE;
                (*l)->is_updated = TRUE;
        }

	for (s = host->socktab_old; s < host->socktab_old + host->nsock_old; s++)
		if (s->priv)
			sock_priv(s)->flags = FALSE;
	
	oldtab = malloc (host->nsock * sizeof (struct cis_sockinfo));
	if (host->nsock && !oldtab)
                return;

        memcpy (oldtab, host->socktab, host->nsock * sizeof (struct cis_sockinfo));

        for (s = oldtab; s < oldtab + host->nsock; s++) {
		if (!user_match(host->uid, s->uid)) continue;

		s->saddr = host->hostinfo->addr;
		if (!s->daddr.s_addr || s->daddr.s_addr == 0x0100007f)
			s->daddr = host->hostinfo->addr;
		else if ((h = find_host(host->hostinfo->caddr, s->daddr)) && s->daddr.s_addr != h->hostinfo->addr.s_addr)
			s->daddr = h->hostinfo->addr;

		if (s->saddr.s_addr != s->daddr.s_addr) {
			host->send_rate += s->sent;
			host->recv_rate += s->rcvd;
		}

		socket = sock_find (host->socktab_old, host->nsock_old, s);

		if (!socket || !socket->priv)
			continue;

		s->priv = socket->priv;
		link = sock_priv(s)->link;
		link->flag = TRUE;
                link->is_updated = FALSE;
                if ((diff = s->sent - socket->sent)) {
                        link->sent      += diff;
                        link->is_updated = TRUE;
                }
                if ((diff = s->rcvd - socket->rcvd)) {
                        link->rcvd      += diff;
                        link->is_updated = TRUE;
                }
                if (s->pid != socket->pid) {
                        link->src_pid    = s->pid;
                        link->is_updated = TRUE;

			for (h = hostlist; h; h = h->next) {
                                socket = sock_find_dest(h->socktab_old, h->nsock_old, s);
				if (!socket)
					continue;
                                link = socket->priv;
                                if (link)
                                        link->dst_pid = s->pid;
                        }
                }
		sock_priv(s)->flags = TRUE;
        }

	for (s = host->socktab_old; s < host->socktab_old + host->nsock_old; s++)
		if (s->priv && !sock_priv(s)->flags)
			free(s->priv);

	free (host->socktab_old);

	host->socktab_old = oldtab;
        host->nsock_old = host->nsock;

        host->sock_updated = FALSE;
}

void linklist_update (void)
{
        struct host_info *h, *host;
        struct cis_sockinfo *s, *socket;
        struct link_info **l, *link;
        int offset = 0;
        int redraw_tree = FALSE;
        XmLGridRow rowp;
        static char buf[128];
        int row, i, len;

	for (h = hostlist; h; h = h->next) {
		h->link_offset += offset;
                if (!h->nsock_old) continue;

		for (s = h->socktab_old; s < h->socktab_old + h->nsock_old; s++) {
			if (!user_match(h->uid, s->uid)) continue;

			if (s->priv && (!sock_priv(s)->link->dst_port || sock_priv(s)->link->dst_pid))
				continue;

			socket = NULL;
                        
			host = find_host (h->hostinfo->caddr, s->daddr);
			if (host)
				socket = sock_find_dest (host->socktab_old, host->nsock_old, s);

			if (s->priv && sock_priv(s)->link) {
				if (socket) {
					sock_priv(s)->link->dst_pid = socket->pid;
					sock_priv(s)->link->is_updated = TRUE;
                                }
                                continue;
                        }

			if (!s->priv)
				s->priv = calloc(1, sizeof(struct sock_info));
			sock_priv(s)->flags = FALSE;

			if ((link = link_find (h, s, host, socket))) {

                                sock_priv(s)->link = link;

                                link->sent += s->sent;
                                link->rcvd += s->rcvd;
                                link->flag = TRUE;
                                link->is_updated = TRUE;

                                continue;
                        }

                        link = calloc (1, sizeof (struct link_info));
                        link->flag = TRUE;
                        link->type = s->type;
                        link->src_pid = s->pid;
                        link->dst_host = host;
                        if (!host)
                                link->dst_addr = s->daddr;
                        if (!socket)
                                link->dst_port = s->dport;
                        else
                                link->dst_pid  = socket->pid;

                        link->sent = s->sent;
                        link->rcvd = s->rcvd;

                        link->is_updated = TRUE;

			sock_priv(s)->link = link;

                        XmLTreeAddRow (Linklist, 1, False, True, h->link_offset+h->link_number+1, pixmaps[DOT], pixmasks[DOT], NULL);
                        XtVaSetValues (Linklist,
                                       XmNrow, h->link_offset+h->link_number+1,
                                       XmNrowExpands, False,
                                       XmNrowUserData, link,
                                       NULL);
                        
                        len = sprintf (buf, "%d |", link->src_pid);
			len += sprintf (buf+len, "%s", host ? host->hostinfo->name : inet_ntoa (link->dst_addr));
                        if (!socket)
                                len += sprintf (buf+len, ":%-4X ||", link->dst_port);
                        else
                                len += sprintf (buf+len, " |%d |", link->dst_pid);

                        len += sprintf (buf+len, "%s |", (link->type == SOCK_STREAM) ? "TCP" : "UDP");
                        len += lprintf (buf+len, "%ld |%ld ",
                                        link->sent, link->rcvd);


                        XmLGridSetStringsPos (Linklist,
                                              XmCONTENT, h->link_offset+h->link_number+1,
                                              XmCONTENT, 0,
                                              buf);

                        redraw_tree = TRUE;
                        offset++;

                        h->linklist = realloc (h->linklist, (h->link_number+1) *
					      sizeof (struct link_info *));
			h->linklist[h->link_number] = link;
			h->link_number++;
                }
        }

        /* Now it's time to update link tree and remove deleted links.*/

        row = 0;
	for (h = hostlist; h; h = h->next) {
		h->link_offset = row;
                if (h->flag)
                        row++;
                
                l = h->linklist;
                
                for (i = 0; i < h->link_number; i++) {

                        rowp = XmLGridGetRow (Linklist, XmCONTENT, row);
                        XtVaGetValues(Linklist,
                                      XmNrowPtr, rowp,
                                      XmNrowUserData, &link,
				      NULL);

                        if (!link->flag) {
                                XmLGridDeleteRows (Linklist, XmCONTENT, row, 1);

                                free (link);
                                continue;
                        }
                        if (link->is_updated) {
				len = sprintf (buf, "%s", link->dst_host ? link->dst_host->hostinfo->name : inet_ntoa (link->dst_addr));
                                if (!link->dst_pid)
                                        len += sprintf (buf+len, ":%-4X ||", link->dst_port);
                                else
                                        len += sprintf (buf+len, " |%d |", link->dst_pid);

                                len += sprintf (buf+len, "%s |", (link->type == SOCK_STREAM) ? "TCP" : "UDP");
                                len += lprintf (buf+len, "%ld |%ld ",
                                                link->sent, link->rcvd);
                                XmLGridSetStringsPos (Linklist, XmCONTENT, row, XmCONTENT, LINK_COL_DSTHOST, buf);
                        }

                        row++;
                        *l = link;
                        l++;
                }

                lprintf (buf, "%ld |%ld ", h->send_rate, h->recv_rate);
                XmLGridSetStringsPos (Linklist, XmCONTENT, h->link_offset, XmCONTENT, LINK_COL_SENT, buf);
                
                h->link_number = l - h->linklist;
        }
}

void dev_tree_change(int row, int column, struct cis_netdevinfo *d)
{
        static char buf[128];

        switch (column) {
        case DEV_COL_STATUS:
                sprintf (buf, "%s", d->status ? "up" : "down");
                break;
                
        case DEV_COL_RX:
                lprintf (buf, "%ld ", d->rx_bytes);
                break;
                
        case DEV_COL_TX:
                lprintf (buf, "%ld ", d->tx_bytes);
                break;
                
        case DEV_COL_COLL:
                lprintf (buf, "%ld ", d->collisions);
                break;
        }

        XmLGridSetStringsPos (Devlist, XmCONTENT, row, XmCONTENT, column, buf);
}

void devtab_update (struct host_info *host, struct cis_netdevinfo *dtab, int len)
{
        struct cis_netdevinfo *d, *dev;
        XYCurve *c;
        XmString s;
        char buf[128];

        for (d = host->devtab; d < host->devtab+host->ndev; d++)
		dev_priv(d)->flags = FALSE;

        for (dev = dtab; dev < dtab+len; dev++) {
                if (!(d = dev_find (host->devtab, host->ndev, dev->name))) {

			sprintf (buf, "%s:%s", host->hostinfo->name, dev->name);

			dev->priv = calloc (1, sizeof (struct dev_info));

			s = XmStringCreateSimple(buf);
			dev_priv(dev)->name = s;

                        net_curves = realloc (net_curves, (nnet_curves+1)* sizeof (XYCurve));
                        c = net_curves + nnet_curves;
                        memset (c, 0, sizeof (XYCurve));
			c->lineStyle = ((dev-dtab) % (XY_N_LINE_STYLES-1))+1;
                        c->name   = s;
			c->points = calloc(time_scale, sizeof(XYPoint));;
			c->nPoints = 0;
			c->markerPixel = host->pixel;
			c->linePixel   = host->pixel;

			nnet_curves++;
                } else {
                        dev->priv = d->priv;
			dev_priv(d)->flags = TRUE;
                }
                        
                net_curve_update (dev_priv(dev)->name, dev->tx_bytes);
        }

        for (d = host->devtab; d < host->devtab+host->ndev; d++)
                if (!dev_priv(d)->flags) {
                        c = find_curve (net_curves, nnet_curves, dev_priv(d)->name);
			free(c->points);
                        for (c++; c < net_curves+nnet_curves; c++)
				*(c-1) = *c;

			nnet_curves--;
			XmStringFree ((XmString) dev_priv(d)->name);
			free (d->priv);
                }

        free (host->devtab);

        host->devtab = dtab;
        host->ndev = len;

        host->dev_updated = TRUE;
}

void devlist_update (struct host_info *host)
{
	struct cis_netdevinfo *d, *newd;
        int diff;
        struct host_info *h;
        int row;
        XmLGridRow rowp;
        char buf[128];
        int len;

        if ((diff = host->ndev - host->dev_number))
                for (h = host->next; h; h = h->next)
                        h->dev_offset += diff;

        for (d = host->devtab; d < host->devtab + host->ndev; d++)
                dev_priv(d)->flags = TRUE;

        for (row = host->dev_offset; row < host->dev_offset+host->dev_number;) {
                rowp = XmLGridGetRow (Devlist, XmCONTENT, row);
                XtVaGetValues(Devlist,
                              XmNrowPtr, rowp,
                              XmNrowUserData, &d,
                              NULL);

                if (!(newd = dev_find (host->devtab, host->ndev, d->name))) {

                        XmLGridDeleteRows (Devlist, XmCONTENT, row, 1);

                        free (d);
                        host->dev_number--;

                        XYSetCurves (net_plot, net_curves, nnet_curves, XY_NO_RESCALE, True);
			continue;
                }

                if (newd->status != d->status)
                        dev_tree_change (row, DEV_COL_STATUS , newd);
                if (newd->rx_bytes != d->rx_bytes)
                        dev_tree_change (row, DEV_COL_RX , newd);
                if (newd->collisions != d->collisions)
                        dev_tree_change (row, DEV_COL_COLL , newd);
                if (newd->tx_bytes != d->tx_bytes)
                        dev_tree_change (row, DEV_COL_TX , newd);

		newd->priv = d->priv;
                *d = *newd;
                
                dev_priv(newd)->flags = FALSE;
		row++;
        }

        /*
         * Insert new devices into the device list and netplot.
         */
        for (d = host->devtab; d < host->devtab + host->ndev; d++) {
                if (!dev_priv(d)->flags) continue;

                newd = malloc (sizeof (struct cis_netdevinfo));
                *newd = *d;

                XmLTreeAddRow (Devlist, 0, False, True, row, pixmaps[DOT], pixmasks[DOT], NULL);
                XtVaSetValues (Devlist,
                               XmNrow, row,
                               XmNrowExpands, False,
                               XmNrowUserData, newd,
                               NULL);

		len = sprintf (buf, "%s|%s|%s|", host->hostinfo->name, d->name, d->status ? "up" : "down");
                lprintf (buf+len, "%ld |%ld |%ld ", d->tx_bytes, d->rx_bytes, d->collisions);

                XmLGridSetStringsPos (Devlist,
                                      XmCONTENT, row,
                                      XmCONTENT, 0,
                                      buf);

                row++;
                host->dev_number++;
                XYSetCurves (net_plot, net_curves, nnet_curves, XY_NO_RESCALE, True);
        }


        host->dev_updated = FALSE;
}

void clear_host_data (struct host_info *host)
{
	struct cis_netdevinfo *dev;

	host->hostinfo_updated = TRUE;

	if (host->flag) {
		((XYCurve *) host->CPU_curve)->nPoints = 0;

		for (dev = host->devtab; dev < host->devtab + host->ndev; dev++)
			(find_curve (net_curves, nnet_curves, dev_priv(dev)->name))->nPoints = 0;
	}

	host->hostinfo->status = HOST_NOT_AVAILABLE;
	host->hostinfo->CPU_available = 0;
	
        proctab_update  (host, NULL, 0);
        socktab_update  (host, NULL, 0);
        devtab_update   (host, NULL, 0);

        host->send_rate = 0;
        host->recv_rate = 0;
}

void clear_data (void)
{
        struct host_info *h;

        /* Clear tables */
	for (h = hostlist; h; h = h->next) {
		clear_host_data (h);

		hostlist_update(h);
		proclist_update(h);
		socklist_update(h);
		devlist_update(h);
	}

        linklist_update();
	
	update_curves_time (0);
        update_time ();
}

void monitor_error (struct host_info *host, int err)
{
        if (err > HOST_NOT_AVAILABLE) {
                cisDisconnect(host->cis->handle);
                host->cis->handle = NULL;
        }
}

void allarm_handler (int sig)
{
        struct host_info *h;
        struct cisinfo *cis;
        struct cis_procinfo *ptab = NULL;
        struct cis_sockinfo *stab = NULL;
        struct cis_netdevinfo *dtab = NULL;
        struct cis_hostinfo *hi;
        int len;
	char *hchanges = NULL;
        char *pchanges = NULL;
        char *schanges = NULL;
        char *dchanges = NULL;
	int hchlen, pchlen, schlen, dchlen;
	int time_saved = FALSE;

	gettimeofday (&current_time, NULL);

        /*
         * Try to connect to all disconnected cis daemons.
         */
	for (cis = cislist; cis; cis = cis->next) {
		if (!cis->handle)
                        cis->handle = cisConnect (cis->name);
        }

        /*
         * Archive previous host information.
         */
	for (h = hostlist; h; h = h->next)
		*h->old_hostinfo = *h->hostinfo;

        /*
	 * Update dynamic host information.
	 */
	for (cis = cislist; cis; cis = cis->next)
		if (cis->handle)
			cisUpdateHostsinfo (cis->handle, CIS_ALL, cis->htab, cis->nhosts, interval);


	/*
         * Update static host information if host became available.
	 */
	for (h = hostlist; h; h = h->next) {
		if (!h->cis ||
		    h->hostinfo->status == h->old_hostinfo->status ||
		    h->hostinfo->status != HOST_AVAILABLE)
			continue;

		hi = cisHostinfo (h->cis->handle, h->hostinfo->addr, perf_type, interval);
		if (hi) {
			hi->performance = h->hostinfo->performance;
			*h->hostinfo = *hi;
			free (hi);
		}
	}

	for (h = hostlist; h; h = h->next) {
                ptab = NULL;
                stab = NULL;
		dtab = NULL;

		h->hostinfo_updated = TRUE;

		if (!h->flag)
			continue;

		if (mode == RECORD)
			hchlen = cisHostinfoPrepareChanges (h->old_hostinfo, h->hostinfo, &hchanges);

		CPU_curve_update(h);

		/*
                 * Processes
		 */
		if (mode != ONLINE || (active_folder == PROCLIST && h->flag&PROC_EXPANDED)) {
			len = 0;
			if (h->cis && h->cis->handle) {
				len = cisProclist (h->cis->handle, &ptab, h->hostinfo->addr, h->uid, interval);
				if (len == -1) {
					monitor_error (h, len);
					len = 0;
				}
			}

			if (mode == RECORD)
				pchlen = cisProclistPrepareChanges (h->proctab, h->nproc, ptab, len, &pchanges);

			h->proc_updated = TRUE;
			proctab_update (h, ptab, len);
		}
		
                /*
                 * Sockets
		 */
		if (mode != ONLINE || (active_folder == LINKLIST/* && h->flag&LINK_EXPANDED*/)) {
			len = 0;
			if (h->cis && h->cis->handle) {
				len = cisSocklist (h->cis->handle, &stab, h->hostinfo->addr, h->uid, h->interval);
				if (len < 0) {
					monitor_error (h, len);
					len = 0;
				}
			}

			if (mode == RECORD)
				schlen = cisSocklistPrepareChanges (h->socktab, h->nsock, stab, len, &schanges);

			h->sock_updated = TRUE;
			socktab_update (h, stab, len);
		}
		
                /*
                 * Network devices
                 */
		len = 0;
		if (h->cis && h->cis->handle) {
			len = cisNdevlist (h->cis->handle, &dtab, h->hostinfo->addr, h->interval);
                        if (len < 0) {
                                monitor_error (h, len);
                                len = 0;
                        }
                }

		if (mode == RECORD)
			dchlen = cisNdevlistPrepareChanges (h->devtab, h->ndev, dtab, len, &dchanges);

		h->dev_updated = TRUE;
		devtab_update (h, dtab, len);

		if (mode != RECORD || !(hchlen || pchlen || schlen || dchlen))
			continue;

		if (mode == RECORD && !time_saved) {
			cisSaveTime (record.file, current_time);
			time_saved = TRUE;
		}
		cisSaveHostID(record.file, h->hostinfo);

		if (hchlen) {
			cisSaveHostinfoChanges (record.file, hchanges, hchlen);
			free (hchanges);
		}
		if (pchlen) {
			cisSaveProclistChanges (record.file, pchanges, pchlen);
			free (pchanges);
		}
		if (schlen) {
			cisSaveSocklistChanges (record.file, schanges, schlen);
			free (schanges);
		}
		if (dchlen) {
			cisSaveNdevlistChanges (record.file, dchanges, dchlen);
			free (dchanges);
		}
	}

        XtNoticeSignal (sigid);
}

int main (int argc,char *argv[])
{
	XYCurve *curve;
	XmLGridCallbackStruct cbs;
	
	int row;
        struct host_info *h;

        struct sigaction allarm_action;
        struct itimerval it;

        setlocale (LC_ALL, "");
        setlocale (LC_NUMERIC, "C");

	if (gethostname ((char *)& myhostname, MAXHOSTNAMELEN) < 0)
                fatal_err ("cannot get my hostname");

	/*
         * Parse command line for config file name.
         */
        getparams (argc, argv);

        /*
         * Parse hostfile for host list and other parameters.
         */
        parse_configfile();

	if (!nhosts)
		fatal_err("no host to monitor");
	
	CPU_curves = calloc (nhosts, sizeof (XYCurve));
	for (curve = CPU_curves; curve < CPU_curves + nhosts; curve++)
		curve->lineStyle = XY_PLAIN_LINE;

	CPU_scale = 0.;
	for (h = hostlist; h; h = h->next)
		if (h->hostinfo->performance > CPU_scale)
			CPU_scale = h->hostinfo->performance;
	CPU_scale *= 1.5;

	if (!(hostlist_sorted = malloc (nhosts * sizeof (struct host_info *))))
		fatal_err ("not enough memory");

        /*
         * Initialization of signal action structures.
         */
        allarm_action.sa_handler = allarm_handler;
        sigemptyset (&allarm_action.sa_mask);
        allarm_action.sa_flags   = 0;

	/*
         * We need to block allarm, while getting data in cb routines.
         */
        if (sigemptyset(&sigblockset) == -1)
		fatal_err("cannot clear set of blocked signals");
	if (sigaddset(&sigblockset, SIGALRM) == -1)
                fatal_err("cannot initialize set of blocked signals");

	create_window(argc, argv);

        /*
         * Initialize the list of hosts
	 */
	for (row = 0, h = hostlist; row < nhosts; row++, h = h->next) {
		hostlist_sorted[row] = h;

                XmLGridAddRows(Hostlist, XmCONTENT, row, 1);
		XtVaSetValues (Hostlist,
                               XmNcolumn, 0,
                               XmNrow, row,
                               XmNcellPixmap, pixmaps[CHECK],
			       XmNcellPixmapMask, pixmasks[UNCHECK],
			       XmNrowUserData, h,
                               NULL);
		h->hostlist_row = row;
		hostlist_update (h);
                h->proc_offset = 0;
                h->link_offset = 0;
		h->dev_offset = 0;
	}

	
	cbs.reason = XmCR_SELECT_CELL;
	cbs.rowType = XmCONTENT;

	for (h = hostlist, row = 0; h; h = h->next, row++) {
		if (!h->flag)
			continue;
		cbs.row = row;
		host_grid_select_cb(Hostlist, NULL, (XtPointer) &cbs);
	}
	hostlist_sort_type = HOST_SORT_NONE;
                        
	/*
         * Let's install handler for SIGALRM.
         */
        sigaction (SIGALRM, &allarm_action, NULL);

        /*
         * Set interval and reset timer.
         */
	sectotval (it.it_interval, interval);
        sectotval (it.it_value   , interval);

        setitimer (ITIMER_REAL, &it, NULL);

	sigid = XtAppAddSignal (app_context, allarm_cb, NULL);
        
        allarm_handler(0);

        XtAppMainLoop(app_context);

        /* Never reached */
        fatal_err ("Unexpected return from XtAppMainLoop");

        exit (0);
}

