/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * As a special exception you are allowed to link xcis executable with
 * Microline Widget Library and Plot Widgets library.
 */

#include "xcis.h"

#define addrtab record.header->hosts

struct rec_info record = {
	NULL,
	NULL,
	"./",
};

static struct host_info *find_host_by_name (char *name)
{
	struct host_info *h;

	for (h = hostlist; h; h = h->next)
		if (!strcmp (name, h->hostinfo->name))
			return (h);

	return (NULL);
}

void record_time(struct record_info r, struct timeval *t)
{
	int i = *(int *) r.data;
	
	t->tv_sec  = record.header->time.tv_sec + i/100;
	t->tv_usec = record.header->time.tv_usec + (i%100)*10000;
	if (t->tv_usec >= 1000000L) {
		t->tv_sec   += t->tv_usec / 1000000L;
		t->tv_usec  %= 1000000L;
	}
}

void parse_full_info (FILE *f)
{
	struct host_info *h;
	struct record_info r;
	struct cis_addr addr;
	struct cis_hostinfo *hi;
	
	while (cisRecordInfo (f, &r) > -1 && r.type != CIS_REC_BREAK) {
		if (r.type == CIS_REC_TIME || r.type == CIS_REC_HOST || r.type == CIS_REC_SYSINFO || h)
			cisReadRecord (f, &r);
		else {
			fseek (f, r.length + CIS_RECHDRLEN, SEEK_CUR);
			continue;
		}

		switch (r.type) {
		case CIS_REC_TIME:
			record_time(r, &current_time);
			free (r.data);
			break;

		case CIS_REC_HOST:
			addr = addrtab[(int) *(char*) r.data];
			free (r.data);
			break;

		case CIS_REC_SYSINFO:
			hi = (struct cis_hostinfo *) r.data;
			h = find_host_by_name(hi->name);
			if (h && !h->flag)
				h = NULL;
			if (h) {
				hi->performance = h->hostinfo->performance;
				*h->hostinfo = *hi;
				CPU_curve_update(h);
				h->hostinfo_updated = TRUE;
			}
			free(r.data);
                        break;

		case CIS_REC_PROCESSES:
                        proctab_update (h, (struct cis_procinfo *)r.data, r.length / sizeof (struct cis_procinfo));
                        break;

		case CIS_REC_SOCKETS:
                        socktab_update (h, (struct cis_sockinfo *)r.data, r.length / sizeof (struct cis_sockinfo));
                        break;

		case CIS_REC_IFACES:
			devtab_update (h, (struct cis_netdevinfo *)r.data, r.length / sizeof (struct cis_netdevinfo));
			break;
		}
	}
	if (r.type == CIS_REC_BREAK) {
		cisReadRecord (f, &r);
		record.offset = *(unsigned int *) r.data;
		free(r.data);
	}
}

void tvaladd (struct timeval *t, float val)
{
	t->tv_sec += (int) val;
	t->tv_usec += (val - (int) val) * 1000000L;
	if (t->tv_usec > 1000000L) {
		t->tv_sec++;
		t->tv_usec -= 1000000L;
	}
}

#define tvgt(a,b) (a.tv_sec > b.tv_sec || (a.tv_sec == b.tv_sec && a.tv_usec > b.tv_usec))

int parse_next_interval (void)
{
        struct record_info r;
	struct host_info *h;
	struct timeval t, end = current_time;
	int err, len;
	struct cis_addr addr;

	tvaladd(&end, interval);
	
	/*
	 * Get time record.
	 */
	if (cisReadRecord (record.file, &r) == -1)
		return 0;

next:
	record_time(r, &t);
	free (r.data);

	if (tvgt (t, end))
		goto out;

	current_time = t;

	while ((err = cisRecordInfo (record.file, &r)) > -1 && r.type != CIS_REC_TIME) {
		struct cis_hostinfo *hi;
		struct cis_procinfo *ptab;
		struct cis_sockinfo *stab;
		struct cis_netdevinfo *dtab;

		if (r.type == CIS_REC_HOST || h)
			cisReadRecord (record.file, &r);
		else {
			fseek (record.file, r.length + CIS_RECHDRLEN, SEEK_CUR);
			continue;
		}

		switch (r.type) {

		case CIS_REC_HOST:
			addr = addrtab[(int) *(char*) r.data];
			h = find_host(addr.server_addr, addr.node_addr);
			if (h && !h->flag)
				h = NULL;
			free (r.data);
			break;

 		case CIS_REC_SYSINFO:
			cisHostinfoApplyChanges (h->hostinfo, r.data, r.length, &hi);
			*h->hostinfo = *hi;
			free(hi);
			free(r.data);
			h->hostinfo_updated = TRUE;
			break;
                        
		case CIS_REC_PROCESSES:
                        len = cisProclistApplyChanges (h->proctab, h->nproc, r.data, r.length, &ptab);
                        free (r.data);
                        proctab_update (h, ptab, len);
                        break;

		case CIS_REC_SOCKETS:
                        len = cisSocklistApplyChanges (h->socktab, h->nsock, r.data, r.length, &stab);
                        free (r.data);
                        socktab_update (h, stab, len);
                        break;

                case CIS_REC_IFACES:
			len = cisNdevlistApplyChanges (h->devtab, h->ndev, r.data, r.length, &dtab);
                        free (r.data);
                        devtab_update (h, dtab, len);
                        break;
		}
	}

	if (err == -1)
		goto out;

	cisReadRecord (record.file, &r);

        goto next;
	
out:
        current_time = end;
	
	for_each_selected_host(h)
		CPU_curve_update(h);

	update_curves_time (0);

	if (r.type == CIS_REC_TIME)
		fseek(record.file, - (CIS_RECHDRLEN+sizeof(int)), SEEK_CUR);
        record.offset = ftell(record.file);

	return 1;
}

unsigned long get_offset(FILE *f)
{
	struct record_info r;
	unsigned long offset;

	fseek(f, -(CIS_RECHDRLEN+sizeof(int)), SEEK_CUR);
	cisReadRecord (f, &r);

	offset = *(unsigned int *) r.data;
	free(r.data);

	return offset;
}

void save_full_info(FILE *f)
{
	struct host_info *h;

//	cisSaveTime (f, current_time);

	for_each_selected_host(h) {
		if (cisSaveHostID (f, h->hostinfo) == -1)
			continue;
		cisSaveHostinfo (f, h->hostinfo);

		cisSaveProclist (f, h->proctab, h->nproc);
		cisSaveSocklist (f, h->socktab, h->nsock);
		cisSaveNdevlist (f, h->devtab,  h->ndev);
	}
}

static void save_curve(FILE *f, XYCurve *c)
{
	cisSaveRecord(f, REC_CURVE_LEN, (char *) &c->nPoints,
		      sizeof (int), 1, sizeof (int));

	cisSaveRecord(f, REC_CURVE, (char *) c->points,
		      sizeof (XYPoint), c->nPoints, sizeof (XYPoint));
}

void save_curves(FILE *f)
{
	struct host_info *h;
	struct cis_netdevinfo *dev;

	for_each_selected_host(h) {
		save_curve (f, h->CPU_curve);
		for (dev = h->devtab; dev < h->devtab + h->ndev; dev++)
			save_curve (f, find_curve (net_curves, nnet_curves, dev_priv(dev)->name));
	}
}

static void read_curve (FILE *f, XYCurve *c)
{
	struct record_info r;
	int err;

	err = cisReadRecord (f, &r);
	c->nPoints = *(int *) r.data;
	free(r.data);

	err = cisReadRecord (f, &r);
	memcpy (c->points, r.data, r.length);
	free(r.data);
}
	
unsigned int parse_curves (FILE *f)
{
	struct host_info *h;
	struct record_info r;
	unsigned int offset;
	struct cis_netdevinfo *dev;

	for_each_selected_host(h) {
		read_curve (f, h->CPU_curve);
		for (dev = h->devtab; dev < h->devtab + h->ndev; dev++)
			read_curve (f, find_curve (net_curves, nnet_curves, dev_priv(dev)->name));
	}
		
	cisReadRecord (f, &r);
	offset = *(unsigned int *) r.data;
	free(r.data);

	return offset;
}
