/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * As a special exception you are allowed to link xcis executable with
 * Microline Widget Library and Plot Widgets library.
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <netdb.h>

#include "xcis.h"

extern float interval;
extern unsigned char reliability;
extern struct user_info *userlist;
extern struct cisinfo *cislist;
extern struct host_info *hostlist;
extern int nhosts;
extern char *CPU_units;
extern char *printpath;
extern int time_scale;
extern float net_scale;
extern struct rec_info record;
extern char *perf_type;
extern char myhostname[];

static FILE *configfile = NULL;

static struct cis_hostinfo *find_host_by_name (struct cisinfo *cis, char *name)
{
	struct cis_hostinfo *h;

	if (!cis)
		return NULL;
	
	for (h = cis->hlist; h < cis->hlist+cis->hlist_length; h++)
		if (!strcmp (name, h->name))
			return (h);

	return (NULL);
}


static struct host_info *add_host (struct cis_hostinfo *hi)
{
	struct host_info *host, *h;

	srand(200);

	if (!(host = calloc (1, sizeof (struct host_info))))
		goto nomem;

	host->hostinfo = hi;
	
	host->old_hostinfo = calloc (1, sizeof (struct cis_hostinfo));
	if (!host->old_hostinfo)
		goto nomem;

	if (!hostlist)
		hostlist = host;
	else {
		for (h = hostlist; h->next; h = h->next)
			;
		h->next = host;
	}
	nhosts++;
	return host;

nomem:
	fatal_err ("cannot allocate memory for host structures");
	return NULL;
}
	

void getparams (int argc, char **argv)
{
	int c;
	int rel;
	struct host_info *host;
	struct cis_hostinfo *hi, *hlist;
	char *hostname = NULL;
	int i;
	int uid = 0;

	while ((c = getopt(argc, argv, "i:c:r:s:p:u:")) != EOF) {

		switch (c) {
		case 'i':
			interval = atof(optarg);
			if (interval == 0.) {
				printf ("interval cannot be set to zero. reset to default %5.2f [s]\n", DEF_INTERVAL);
				interval = DEF_INTERVAL;
			}
                        break;
                case 'r':
			rel = parse_cmd(rel_types, REL_TABLEN, &optarg);
			if (rel == -1)
				fatal_err("wrong reliability type");
			reliability = rel;
			break;
		case 's':
			hostname = strdup(optarg);
			break;
                case 'c':
			if(!(configfile = fopen (optarg, "r")))
				fatal_err ("cannot open config file %s.", optarg);
			break;
                case 'p':
			perf_type = optarg;
			break;
                case 'u':
			uid = atoi(optarg);
			break;
		}
	}

	if (!configfile) {
		cislist = calloc(1, sizeof (struct cisinfo));

		cislist->name = hostname ? hostname : getenv("CIS_SERVER");

		if (!cislist->name)
			cislist->name = strdup(myhostname);

		cislist->handle = cisConnect(cislist->name);
		if (!cislist->handle)
			fatal_err("cannot connect cis server %s", cislist->name);

		cislist->hlist_length = cisHostlist(cislist->handle, &cislist->hlist, reliability, "", interval);
		if (cislist->hlist_length < 1)
			return;

		for (hi = cislist->hlist; hi < cislist->hlist + cislist->hlist_length; hi++) {
			host = add_host (hi);

			host->uid = uid;
			host->cis = cislist;
			host->interval = SOCK_INTERVAL;
		}

		cislist->htab = calloc (nhosts,sizeof(struct cis_hostinfo *));
		if (!cislist->htab)
			fatal_err ("not enough memory");
		cislist->nhosts = nhosts;

		for (host = hostlist, i = 0; i < nhosts; i++, host = host->next)
			cislist->htab[i] = host->hostinfo;

		free (hlist);
	}
}

char *cmds[] = {"#", "CisServer", "Host", "User", "SpeedUnits", "TimeScale", "NetScale", "Interval", "RecPath", "PrintPath"};

enum {
        COMMENT,
	CIS,
	HOST,
        USER,
        SPEEDUNITS,
        TIMESCALE,
        NETSCALE,
        INTERVAL,
        RECPATH,
	PRINTPATH,
	NCMDS,
};

void parse_configfile ()
{
        static int buflen = 0;
        static char *buf  = NULL;
        int cmd;
	char login[32];
	char name[256];
	char color[64];
	char flag[10];
        struct cisinfo *cis = NULL;
        float perf;
        struct host_info *host;
        struct user_info *user;
        struct cis_hostinfo *hi;
        char *c;
	unsigned int uid = 0;
	int line = 0;
	int err;
        CLIENT *handle;
	
	if (!configfile)
		return;
	
	errno = 0;

	while (getline (&buf, &buflen, configfile) != -1) {

		line++;
		c = buf;

		cmd = parse_cmd (cmds, NCMDS, &c);
		if (cmd == 0) continue;

		c += strspn (c, " \t");
		if (c[strlen(c)-1] == '\n')
			c[strlen(c)-1] = 0;

                switch (cmd) {
                        
		case CIS:
			
                        if (sscanf (c, "%s", name) != 1)
				fatal_err ("missing cis server name at line %d", line);

			if (find_cis (name))
				continue;

			handle = cisConnect(name);
			if (!handle) {
				warning ("cannot connect to cis server %s", name);
				continue;
			}
			
			cis = calloc (1, sizeof (struct cisinfo));
			cis->name   = strdup (name);
			cis->handle = handle;
			
			cis->next = cislist;
			cislist = cis;

			host = NULL;

			cis->hlist_length = cisHostlist(cislist->handle, &cis->hlist, reliability, "", interval);

			break;

		case HOST:

			uid = 0; *color = 0;
			err = sscanf (c, "%s %f %s %s %d", name, &perf, flag, color, &uid);

			if (err < 2)
				fatal_err ("wrong host command syntax. line: %s", buf);

			if (!(hi = find_host_by_name (cis, name)))
				hi = calloc (1, sizeof(struct cis_hostinfo));

			snprintf(hi->name, CIS_HOSTNAMELEN, "%s", name);

			host = add_host(hi);

			host->uid = uid;
			host->interval = SOCK_INTERVAL;
			if (*color)
				host->color = strdup (color);
			hi->performance = perf;
			if (!strcasecmp(flag, "on"))
				host->flag = TRUE;

			host->cis = cis;

			if (cis) {
				cis->htab = realloc (cis->htab, (cis->nhosts+1) * sizeof(struct cis_hostinfo *));
				cis->htab[cis->nhosts] = hi;
				cis->nhosts++;
			}
			break;

		case USER:

			*login = *name = 0;

			err = sscanf (c, "%s %d \"%[^\"]\"", login, &uid, name);
			if (err != 3)
				fatal_err ("wrong user command syntax at line: %d", line);

			user = calloc (1, sizeof(struct user_info));
			if (!user)
				fatal_err ("cannot allocate memory for usr structures");

			user->name = strdup (login);
			user->id = uid;
			user->fullname = strdup (name);

			if (cis) {
				user->next = cis->users;
				cis->users = user;
			} else {
				user->next = userlist;
				userlist = user;
			}
			break;

                case SPEEDUNITS:

                        c += strspn (c, " \t");
                        CPU_units = strdup (c);
                        break;

                case TIMESCALE:

                        sscanf (c, "%i", &time_scale);
                        break;

		case NETSCALE:

                        sscanf (c, "%f", &net_scale);
                        break;
                        
                case INTERVAL:

                        sscanf (c, "%f", &interval);
                        break;

                case RECPATH:

                        record.path = strdup(c);
                        break;

                case PRINTPATH:

			printpath = strdup(c);
                        break;

                default:
			break;
                }
        }
	fclose (configfile);

	if (!hostlist)
                fatal_err("no host to monitor");
}
