/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * As a special exception you are allowed to link xcis executable with
 * Microline Widget Library and Plot Widgets library.
 */

#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>

#include "xcis.h"
#include "replay.h"

char *printpath = "./";
static char tmpfilename[16];

static FILE *tmp_file;

extern sigset_t sigblockset;

void folder_activate_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	XmLFolderCallbackStruct *cbs = (XmLFolderCallbackStruct *) cbsp;

	active_folder = cbs->pos;
}

void proclist_expand_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
	struct host_info *h;
	struct cis_procinfo *ptab = NULL;
	XmLGridRow rowp;
	int len;

	if (mode != ONLINE)
		return;

	rowp = XmLGridGetRow (Proclist, XmCONTENT, cbs->row);
	XtVaGetValues(Proclist,
		      XmNrowPtr, rowp,
		      XmNrowUserData, &h,
		      NULL);

	sigprocmask(SIG_BLOCK, &sigblockset, NULL);

	ptab = NULL;
	len = 0;
	if (h->cis && h->cis->handle) {
		len = cisProclist (h->cis->handle, &ptab, h->hostinfo->addr, h->uid, 1);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}
	}
	proctab_update (h, ptab, len);
	proclist_update (h);

	h->flag |= PROC_EXPANDED;
	
	sigprocmask(SIG_UNBLOCK, &sigblockset, NULL);
}

void proclist_collapse_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
	struct host_info *h;
	XmLGridRow rowp;

	rowp = XmLGridGetRow (Proclist, XmCONTENT, cbs->row);
	XtVaGetValues(Proclist,
		      XmNrowPtr, rowp,
		      XmNrowUserData, &h,
		      NULL);

	h->flag &= ~PROC_EXPANDED;
}

void proclist_cell_select_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
        int old_type = proclist_sort_type;
        struct host_info *h;
        
        if (cbs->rowType != XmHEADING)
		return;

        if (cbs->columnType != XmCONTENT)
                return;

        switch (cbs->column) {
        case PROC_COL_CMD:
                proclist_sort_type = PROC_SORT_TREE;
                break;
        case PROC_COL_PID:
                proclist_sort_type = PROC_SORT_PID;
                break;
        case PROC_COL_PCPU:
                proclist_sort_type = PROC_SORT_PCPU;
                break;
        case PROC_COL_RSS:
                proclist_sort_type = PROC_SORT_RSS;
                break;
        case PROC_COL_VM:
                proclist_sort_type = PROC_SORT_VM;
                break;
        case PROC_COL_MAJFLT:
                proclist_sort_type = PROC_SORT_MAJFLT;
                break;
        }

        XtVaSetValues(Proclist, XmNlayoutFrozen, True, NULL);
        if (proclist_sort_type != old_type)
                for (h = hostlist; h; h = h->next)
                        proc_tree_sort (h);
        XtVaSetValues(Proclist, XmNlayoutFrozen, False, NULL);
}

void linklist_expand_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
	struct host_info *h;
        struct cis_sockinfo *stab = NULL;
	XmLGridRow rowp;
	int len;

	if (mode != ONLINE)
		return;

	rowp = XmLGridGetRow (Linklist, XmCONTENT, cbs->row);
	XtVaGetValues(Linklist,
		      XmNrowPtr, rowp,
		      XmNrowUserData, &h,
		      NULL);

	sigprocmask(SIG_BLOCK, &sigblockset, NULL);

	stab = NULL;
	len = 0;
	if (h->cis && h->cis->handle) {
		len = cisSocklist (h->cis->handle, &stab, h->hostinfo->addr, h->uid, h->interval);
		if (len < 0) {
			monitor_error (h, len);
			len = 0;
		}
	}
	socktab_update (h, stab, len);
	socklist_update (h);

	linklist_update();

	h->flag |= LINK_EXPANDED;
	
	sigprocmask(SIG_UNBLOCK, &sigblockset, NULL);
}

void linklist_collapse_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
	struct host_info *h;
	XmLGridRow rowp;

	rowp = XmLGridGetRow (Linklist, XmCONTENT, cbs->row);
	XtVaGetValues(Linklist,
		      XmNrowPtr, rowp,
		      XmNrowUserData, &h,
		      NULL);

	h->flag &= ~LINK_EXPANDED;
}


void linklist_cell_activate_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
        struct host_info *h;
        struct cis_procinfo *process;
	struct cis_procinfo *ptab = NULL;
        int len;
        struct link_info *link;
        XmLGridRow rowp;
        unsigned int pid;
        int level, row;
        
        if (cbs->rowType != XmCONTENT)
                return;

        if (cbs->columnType != XmCONTENT)
                return;

        if (cbs->column != LINK_COL_PID && cbs->column != LINK_COL_DSTPID)
                return;

        rowp = XmLGridGetRow (Linklist, XmCONTENT, cbs->row);
        XtVaGetValues(Linklist,
                      XmNrowPtr, rowp,
                      XmNrowLevel, &level,
                      XmNrowUserData, &link,
                      NULL);

        if (!level)
                return;
        
        switch (cbs->column) {
	case LINK_COL_PID:
		for_each_selected_host(h)
			if (h->link_offset + h->link_number + 1 >= cbs->row)
				break;

                pid = link->src_pid;
                break;
                
        case LINK_COL_DSTPID:

                h = link->dst_host;
                if (!h)
                        return;
                pid = link->dst_pid;
                break;
	}

	if (mode == ONLINE) {
		ptab = NULL;
		len = 0;
		if (h->cis && h->cis->handle) {
			len = cisProclist (h->cis->handle, &ptab, h->hostinfo->addr, h->uid, 1);
			if (len == -1) {
				monitor_error (h, len);
				len = 0;
			}
		}
		proctab_update (h, ptab, len);
		proclist_update (h);
	}
        XtVaSetValues(Proclist,
                      XmNrow, h->proc_offset,
                      XmNrowIsExpanded, True,
                      NULL);

        for (row = h->proc_offset+1; row < h->proc_offset + h->proc_number + 1; row++) {
                rowp = XmLGridGetRow (Proclist, XmCONTENT, row);
		XtVaGetValues(Proclist,
			      XmNrowPtr, rowp,
                              XmNrowUserData, &process,
			      NULL);

                if (process->pid == pid) {
			XmLFolderSetActiveTab (Folder, PROCLIST, FALSE);
			active_folder = PROCLIST;
			h->flag |= PROC_EXPANDED;
                        XmLGridSelectRow (Proclist, row, 0);
                        XmLGridSetFocus (Proclist, row, 0);
                }
        }
        XmUpdateDisplay (Xcis);
}

void scroll_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        XtVaSetValues(Proclist, XmNlayoutFrozen, False, NULL);
}

void record_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        time_t ct;
        struct tm *tmp;
        char buff[MAXHOSTNAMELEN+32], *newname;
	struct host_info *h;
	static struct cis_addr addrtab[256];
	int i, len;
	struct timeval t;

        if (mode == RECORD) {

		XtVaSetValues (w, XmNshadowType, XmSHADOW_OUT, NULL);
                XtVaSetValues (Buttons[REP], XmNsensitive, True, NULL);
                mode = ONLINE;
                fclose (record.file);

	} else {

                ct = time (NULL);
                tmp = localtime (&ct);

                len = strftime (buff, MAXHOSTNAMELEN+34, "%Y.%m.%d_%H:%M:%S", tmp);
                sprintf (buff+len, "_%s.rec", myhostname);
                newname = realloc (record.filename, strlen (buff)+strlen(record.path)+1);
                if (!newname)
                        return;
                record.filename = newname;

                strcpy (newname, record.path);
                strcat (newname, buff);
                
                record.file = fopen (record.filename, "w+");
                if (!record.file)
                        return;

		for (i = 0, h = hostlist; h && i < 256;  h = h->next)
			if (h->flag) {
				addrtab[i].server_addr = h->hostinfo->caddr;
				addrtab[i].node_addr   = h->hostinfo->addr;
				i++;
			}
                if (i == 256)
			printf("reached maximal number of hosts in record file (256)");

		gettimeofday(&t, NULL);
		if (cisSaveHeader(record.file, &t, i, addrtab, interval) == -1) {
			printf("error while saving record file header");
			return;
		}

		for_each_selected_host(h) {
			cisSaveHostID (record.file, h->hostinfo);
			cisSaveHostinfo (record.file, h->hostinfo);

			cisSaveProclist (record.file, h->proctab, h->nproc);
			cisSaveSocklist (record.file, h->socktab, h->nsock);
			cisSaveNdevlist (record.file, h->devtab,  h->ndev);
		}
		cisSaveBreak (record.file, 0);

                XtVaSetValues (w, XmNshadowType, XmSHADOW_IN, NULL);
                XtVaSetValues (Buttons[REP], XmNsensitive, False, NULL);
                mode = RECORD;
        }
}

void replay_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	struct itimerval it;
	Widget *button;

        if (mode == REPLAY) {
                XtVaSetValues (w, XmNshadowType, XmSHADOW_OUT, NULL);
                XtVaSetValues (Buttons[RECORD], XmNsensitive, True, NULL);

		for (button = Buttons+BEGIN; button <= Buttons+END; button++)
			XtUnmanageChild(*button);

                fclose (record.file);
		fclose (tmp_file);
                unlink (tmpfilename);
		
                clear_data();

                mode = ONLINE;

                sectotval (it.it_interval, interval);
                sectotval (it.it_value   , interval);
                setitimer (ITIMER_REAL, &it, NULL);

                allarm_handler(0);
                XtNoticeSignal (sigid);
        } else {

                XtManageChild (File_selection);
                
                XtVaSetValues (w, XmNshadowType, XmSHADOW_IN, NULL);
        }
}

char *print_extensions[] = {
        "hosts",
        "processes",
        "links",
        "netdevs",
        "CPUplot.eps",
        "Netplot.eps",
};

void print_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        time_t ct;
        struct tm *tmp;
        int active_folder;
        char buff[MAXHOSTNAMELEN+34], *filename, *newname;
        int len;
        FILE *file;

        XtVaGetValues (Folder, XmNactiveTab, &active_folder, NULL);

        ct = current_time.tv_sec;
        tmp = localtime (&ct);

        len = strftime (buff, MAXHOSTNAMELEN+34, "%Y.%m.%d_%H:%M:%S", tmp);
        sprintf (buff+len, "_%s.%s", myhostname, print_extensions[active_folder]);

        filename = strdup (printpath);
        newname = realloc (filename, strlen (buff)+strlen(filename)+1);
        if (!newname)
                goto end;
        filename = newname;

        strcat (filename, buff);

        file = fopen (filename, "w+");
        if (!file)
                goto end;

        switch (active_folder) {
        case HOSTLIST:
                XmLGridWrite (Hostlist, file, XmFORMAT_PAD, 0, False);
                break;
        case PROCLIST:
                XmLGridWrite (Proclist, file, XmFORMAT_PAD, 0, False);
                break;
        case LINKLIST:
                XmLGridWrite (Linklist, file, XmFORMAT_PAD, 0, False);
                break;
        case DEVLIST:
                XmLGridWrite (Devlist, file, XmFORMAT_PAD, 0, False);
                break;
        case CPUPLOT:
                XYPrintContents (CPU_plot, filename);
                break;
        case NETPLOT:
                XYPrintContents (net_plot, filename);
                break;
        }

        fclose (file);
        
    end:
        free (filename);
}

void progress_cb (int value)
{
        XtVaSetValues (Progress_bar, XmNvalue, value, NULL);
        
        while (XtAppPending (app_context))
                XtAppProcessEvent(app_context, XtIMAll);
}

void file_selection_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        static struct itimerval it = {{0,0}, {0,0}};
        XmFileSelectionBoxCallbackStruct *cbs = cbsp;
        char *filename;
	Widget *button;
	struct stat fst;
	unsigned long length, offset;
	int i, value, end;
        
        if (cbs->reason == XmCR_OK) {

                filename = string_to_text(cbs->value);

                XtUnmanageChild (w);

		setitimer (ITIMER_REAL, &it, NULL);

                record.file = fopen (filename, "r");
                if (!record.file) {
                        printf ("cannot open file %s\n", filename);
                        goto cancel;
		}
		if (fstat (fileno(record.file), &fst) == -1)
			goto cancel;

		record.length = length = fst.st_size;

		if (cisReadHeader (record.file, &record.header) < 0 ||
		    record.header->version != CIS_RECORD_VERSION) {
                        fclose (record.file);
                        goto cancel;
		}

		sprintf(tmpfilename, "/tmp/xcis.%d", getpid());
		tmp_file = fopen (tmpfilename, "w+");
		if (!tmp_file) {
			printf ("cannot open temporary file %s\n", tmpfilename);
			goto cancel;
		}

		XtManageChild (Progress_bar);

		clear_data();
		parse_full_info(record.file);

		record.start_time = old_time = current_time = record.header->time;
		i = time_scale;
		record.start = ftell(record.file);

		end = FALSE;

		for(;;) {

			offset = ftell(record.file);
			value = 100 - ((length - offset)*100)/length;
			XtVaSetValues (Progress_bar, XmNvalue, value, NULL);

			if (i == time_scale || end) {

				record.end = ftell(tmp_file);
				cisSaveTime(tmp_file, current_time);
				save_full_info(tmp_file);
				cisSaveBreak(tmp_file, offset);
				save_curves(tmp_file);
				cisSaveBreak(tmp_file, ftell(tmp_file) - record.end);

				i = 0;
			}
			if (end)
				break;

			end = !parse_next_interval();

			i++;
		}

		XtUnmanageChild (Progress_bar);

		old_time = current_time = record.start_time;

		clear_data();
		rewind(tmp_file);
		parse_full_info(tmp_file);
		fseek(record.file, record.offset, SEEK_SET);
                parse_curves (tmp_file);

		record.counter = 0;
		
                update_time ();

		for (button = Buttons+BEGIN; button <= Buttons+END; button++)
			XtManageChild(*button);

                XtVaSetValues (Buttons[RECORD], XmNsensitive, False, NULL);

                mode = REPLAY;

                XmUpdateDisplay (Xcis);
		XtNoticeSignal (sigid);

                return;
        }

    cancel:
        
        XtVaSetValues (Buttons[REP], XmNshadowType, XmSHADOW_OUT, NULL);
        XtUnmanageChild (w);
}

XtIntervalId timer;

void timer_cb (XtPointer button_cbp, XtIntervalId *id)
{
	if (!timer)
		return;
	((XtCallbackProc) button_cbp) (NULL,NULL,NULL);
}

void disarm_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	if (!timer)
		return;
	XtRemoveTimeOut(timer);
	timer = 0;
}

void begin_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	rewind(tmp_file);
	parse_full_info(tmp_file);
	fseek(record.file, record.offset, SEEK_SET);
	parse_curves (tmp_file);
	record.counter = 0;

	update_time ();
	XtNoticeSignal (sigid);
}

void rew_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	if (record.offset == record.start)
		return;

	if (!record.counter)
		fseek(tmp_file, -(get_offset(tmp_file) + CIS_RECHDRLEN + sizeof(int)), SEEK_CUR);
	fseek(tmp_file, -(get_offset(tmp_file) + CIS_RECHDRLEN + sizeof(int)), SEEK_CUR);

	parse_full_info(tmp_file);
	fseek(record.file, record.offset, SEEK_SET);
	old_time = current_time;
	parse_curves (tmp_file);
	record.counter = 0;

	timer = XtAppAddTimeOut (app_context, DELAY, timer_cb, rew_button_cb);

	update_time ();
	XtNoticeSignal (sigid);
}

void step_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	struct record_info r;
	
	if (record.offset == record.length)
		return;

	if (record.counter == time_scale) {
		/*
		 * Go to next full info
		 */
		while (cisRecordInfo(tmp_file, &r) == 0 && r.type != CIS_REC_BREAK)
			fseek(tmp_file, r.length + CIS_RECHDRLEN, SEEK_CUR);

		fseek(tmp_file, r.length + CIS_RECHDRLEN, SEEK_CUR);

		while (cisRecordInfo(tmp_file, &r) == 0 && r.type != CIS_REC_BREAK)
			fseek(tmp_file, r.length + CIS_RECHDRLEN, SEEK_CUR);

		fseek(tmp_file, r.length + CIS_RECHDRLEN, SEEK_CUR);
	}

	parse_next_interval();
        record.counter++;

	timer = XtAppAddTimeOut (app_context, DELAY, timer_cb, step_button_cb);

	update_time ();
	XtNoticeSignal (sigid);
}

void fwd_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	if (record.offset == record.length)
		return;

	parse_full_info(tmp_file);

	fseek(record.file, record.offset, SEEK_SET);
	old_time = current_time;
	parse_curves (tmp_file);
	record.counter = 0;

	timer = XtAppAddTimeOut (app_context, DELAY, timer_cb, fwd_button_cb);

	update_time ();
	XtNoticeSignal (sigid);
}

void end_button_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
	if (record.offset == record.length)
		return;

	fseek(tmp_file, record.end, SEEK_SET);
	record.counter = 0;

	parse_full_info(tmp_file);
	fseek(record.file, record.offset, SEEK_SET);
	old_time = current_time;
	parse_curves (tmp_file);

	update_time ();
	XtNoticeSignal (sigid);
}

void host_grid_select_cb(Widget w, XtPointer client_data, XtPointer cbsp)
{
        XmLGridCallbackStruct *cbs = (XmLGridCallbackStruct *) cbsp;
        XmLGridRow row;
        XmLGridColumn column;
        Pixmap mask;
        struct host_info *host, *h;
        XmString s;
	XYCurve *c;
	XYPoint *points;
        int old_type;

        if (cbs->reason != XmCR_SELECT_CELL)
                return;
        if (cbs->rowType != XmCONTENT) {

                old_type = hostlist_sort_type;

                switch (cbs->column) {
                case HOST_COL_NAME:
                        hostlist_sort_type = HOST_SORT_NONE;
                        break;
                case HOST_COL_PERF_MAX:
                        hostlist_sort_type = HOST_SORT_PERF_MAX;
                        break;
                case HOST_COL_PERF_AVAIL:
                        hostlist_sort_type = HOST_SORT_PERF_AVAIL;
                        break;
                default:
                        return;
                }
                if (hostlist_sort_type == old_type)
                        return;
                hostlist_sort ();
                return;
        }

	if (mode != ONLINE)
		return;
        
        row = XmLGridGetRow(w, cbs->rowType, cbs->row);
        column = XmLGridGetColumn(w, XmCONTENT, 0);

        XtVaGetValues(w,
                      XmNrowPtr, row,
                      XmNcolumnPtr, column,
                      XmNcellPixmapMask, &mask,
                      XmNrowUserData, &host,
                      NULL);
        
	if (mask == pixmasks[UNCHECK]) {
		
		points = calloc (time_scale, sizeof (XYPoint));
		if (!points) {
			printf("not enough memory\n");
			return;
		}

                mask = pixmasks[CHECK];

		s = XmStringCreateSimple(host->hostinfo->name);
                XmLTreeAddRow(Proclist, 0, True, False, host->proc_offset,
                              XmUNSPECIFIED_PIXMAP, XmUNSPECIFIED_PIXMAP, s);
                XmLTreeAddRow(Linklist, 0, True, False, host->link_offset,
                              XmUNSPECIFIED_PIXMAP, XmUNSPECIFIED_PIXMAP, s);
//                XmLTreeAddRow(Devlist, 0, True, True, host->dev_offset,
//                              XmUNSPECIFIED_PIXMAP, XmUNSPECIFIED_PIXMAP, s);

		XmStringFree (s);
		XtVaSetValues (Proclist, XmNrow, host->proc_offset,
			       XmNrowUserData, host, NULL);
		XtVaSetValues (Linklist, XmNrow, host->link_offset,
			       XmNrowUserData, host, NULL);

		for (h = host->next; h; h = h->next) {
			h->proc_offset++;
                        h->link_offset++;
//			h->dev_offset++;
		}

		c = NULL;
		for (h = host->next; h; h = h->next)
			if (h->flag) {
				c = h->CPU_curve;
				break;
			}
		if (c) {
			memmove(c+1, c, (nCPU_curves - (c - CPU_curves))*sizeof (XYCurve));
			for (h = host->next; h; h = h->next)
				if (h->flag)
					((XYCurve *) h->CPU_curve)++;
		} else
			c = CPU_curves+nCPU_curves;
		
		c->name   = XmStringCreateSimple(host->hostinfo->name);

                c->points = points;
                c->nPoints = 0;
                
		host->CPU_curve = c;
		c->markerPixel = host->pixel;
		c->linePixel   = host->pixel;

		nCPU_curves++;

                host->flag = TRUE;

        } else {
                mask = pixmasks[UNCHECK];

                clear_host_data (host);
                proclist_update (host);
                socklist_update (host);
                devlist_update  (host);
                linklist_update ();

                XmLGridDeleteRows (Proclist, XmCONTENT, host->proc_offset, 1);
                XmLGridDeleteRows (Linklist, XmCONTENT, host->link_offset, 1);
//                XmLGridDeleteRows (Devlist,  XmCONTENT, host->dev_offset,  1);
		for (h = host->next; h; h = h->next) {
                        h->proc_offset--;
                        h->link_offset--;
//                        h->dev_offset--;
                }
		c = host->CPU_curve;
		XmStringFree (c->name);

		memmove(c, c+1, (nCPU_curves - (c - CPU_curves) - 1)*sizeof (XYCurve));
		
                for (h = host->next; h; h = h->next)
			if (h->flag)
				((XYCurve *) h->CPU_curve)--;

                host->CPU_curve = NULL;

                nCPU_curves--;

                host->flag = FALSE;
        }
        XtVaSetValues(w,
                      XmNrow, cbs->row,
                      XmNcolumn, 0,
                      XmNcellPixmapMask, mask,
                      NULL);

	XYSetCurves (CPU_plot, CPU_curves, nCPU_curves, XY_NO_RESCALE, True);
}

void allarm_cb (XtPointer cl_data, XtSignalId *id)
{
        struct host_info *h;
        int update_links = FALSE;

	for (h = hostlist; h; h = h->next) {
		if (h->hostinfo_updated)
                        hostlist_update (h);
                if (h->proc_updated)
                        proclist_update (h);
                if (h->sock_updated) {
                        socklist_update (h);
                        update_links = TRUE;
                }
                if (h->dev_updated)
                        devlist_update (h);
        }
        if (update_links)
                linklist_update();

        if (hostlist_sort_type != HOST_SORT_NONE)
                hostlist_sort ();

        update_curves_time (1);
        update_time ();
}
