/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * As a special exception you are allowed to link xcis executable with
 * Microline Widget Library and Plot Widgets library.
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <locale.h>

#include <Xm/Xm.h>
#if XmVersion >= 1002
#include <Xm/PrimitiveP.h>
#endif
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/DrawnB.h>
#include <Xm/Label.h>
#include <Xm/Scale.h>
#include <Xm/TextF.h>
#include <Xm/FileSB.h>

#include <XmL/Folder.h>
#include <XmL/Tree.h>
#include <XmL/Progress.h>

#include <Xm/DrawnB.h>

#include <X11/Intrinsic.h>
#include "XY.h"

#define _WITH_X

#include "xcis.h"

struct {
	int hostlist:1;
	int proclist:1;
	int linklist:1;
	int devlist:1;
	int CPUplot :1;
} changed;

struct rec_info record = {
	NULL,
	NULL,
	"./",
	DEF_RECORD_DENSITY,
	{(int) DEF_TIME_SCALE, (DEF_TIME_SCALE - (int) DEF_TIME_SCALE) * 1000000}
};

struct timeval current_time, old_time;

struct user_info *userlist = NULL;
struct cis_info  *cislist = NULL;
struct host_info *hostlist = NULL, **hostlist_sorted = NULL;

int nhosts = 0;

char *perf_type = "";

struct user_info *find_user(struct host_info *host, unsigned short uid);

char myhostname[MAXHOSTNAMELEN];

Display *display;
Widget Xcis;

Pixmap pixmaps[2];
Pixmap pixmasks[3];

float interval = DEF_INTERVAL;
struct timeval interval_tval = {(int) DEF_INTERVAL, (DEF_INTERVAL - (int) DEF_INTERVAL) * 1000000};

float time_scale  = DEF_TIME_SCALE;
int   mypid;

int proclist_sort_type = PROC_SORT_TREE;
int hostlist_sort_type = HOST_SORT_NONE;

Screen *screen;
XtAppContext app_context;

Widget Folder;
Widget Time_field;
Widget Hostlist;
Widget Proclist;
Widget Linklist;
Widget Devlist;

Widget Buttons[NBUTTONS];

Widget File_selection;
Widget Progress_bar;
int proclist_size=0;
//Pixmap pixmap, pixmask;
XtSignalId sigid;

int mode = ONLINE;

int nCPU_curves = 0;

Widget CPU_plot;
XYCurve *CPU_curves = NULL;
float CPU_scale = DEF_CPU_SCALE;
char *CPU_units = NULL;

Widget net_plot;
int nnet_curves = 0;
XYCurve *net_curves = NULL;
float net_scale = DEF_NET_SCALE;
char *net_units = NULL;

unsigned char reliability = 0;

struct cis_info *find_cis (char *name)
{
	struct cis_info *cis;

	for (cis = cislist; cis; cis = cis->next)
		if (!strcmp (name, cis->name))
			return (cis);

	return (NULL);
}

struct host_info *find_host_by_name (char *name)
{
	struct host_info *h;

	for (h = hostlist; h; h = h->next)
		if (!strcmp (name, h->name))
			return (h);

	return (NULL);
}


char *string_to_text(XmString s)
{
        XmStringContext ctxt;
        XmStringCharSet charset;
        XmStringDirection direction;
        Boolean sep, ret;
        char *text;

        XmStringInitContext(&ctxt, s);
        ret = XmStringGetNextSegment(ctxt, &text, &charset, &direction, &sep);
        XmStringFreeContext(ctxt);
        XtFree((char *)charset);

        if (!ret)
                return NULL;
        
        return text;
}
        

/*
 * Function for comparing two timeval structures.
 * It returns an integer less than, equal to, or greater than zero if t1 is
 * found, respectively, to be less than, to match, or be greater than t2.
 */


int tvalcmp (struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

void tvaladd (struct timeval *t, struct timeval inc)
{
        t->tv_sec  += inc.tv_sec;
        t->tv_usec += inc.tv_usec;
        if (t->tv_usec > 1000000L) {
                t->tv_usec -= 1000000L;
                t->tv_sec++;
        }
}

void tvalsub (struct timeval *t, struct timeval dec)
{
        t->tv_sec -= dec.tv_sec;

        if (t->tv_usec < dec.tv_usec) {
                t->tv_sec--;
                t->tv_usec -= 1000000L - dec.tv_usec;
        } else
                t->tv_usec -= dec.tv_usec;
}


void update_time (void)
{
        time_t t = current_time.tv_sec;
        struct tm *tmp;
        static char time_buff[128];
        XmString s;
        
        XmUpdateDisplay (Xcis);
        tmp = localtime (&t);
        strftime (time_buff, 128, "%x %X", tmp);
        s = XmStringCreateSimple (time_buff);
        XtVaSetValues (Time_field, XmNlabelString, s, NULL);
        XmStringFree(s);

        XmUpdateDisplay (Xcis);
}
/*
void xcisRefreshScreen (void)
{
        while (XtAppPending (app_context))
                XtAppProcessEvent(app_context, XtIMAll);
} */

struct user_info *find_user(struct host_info *host, unsigned short uid)
{
	struct user_info *u;

	for (u = host->users; u; u = u->next)
		if (u->id == uid)
			return u;

	if (host->cis)
		for (u = host->cis->users; u; u = u->next)
			if (u->id == uid)
				return u;

	for (u = userlist; u; u = u->next)
		if (u->id == uid)
			return u;
	
	return NULL;
}

struct host_info *find_host(struct in_addr addr)
{
        struct host_info *h;
        struct in_addr *a;

        for (h = hostlist; h; h = h->next)
                for (a = h->addrlist; a->s_addr ; a++)
                        if (a->s_addr == addr.s_addr)
                                return h;
        return NULL;
}

void hostlist_update (struct host_info *host)
{
        int row = host->hostlist_row;
        struct cis_hostinfo *hi = host->hostinfo;
        static char buf[128];

        if (!hi || hi->status != HOST_AVAILABLE)
                sprintf (buf, "%s|%5.2f |not|available||||||| ",
			 host->name,
			 host->hostinfo->performance);
	else
		/*                Performance      Ram     Swap  NProc   Avg.load        */
		sprintf (buf, "%s|%5.2f |%5.2f |%ld |%ld |%ld |%ld |%d |%5.3f |%5.3f |%5.3f |%s",
			 host->name,
			 host->hostinfo->performance,
			 (host->hostinfo->CPU_available * host->hostinfo->performance)/ 1e4,
			 hi->totalram,
			 hi->freeram + hi->bufferram + hi->cachedram,
			 hi->totalswap,
			 hi->freeswap,
                         hi->procnum,
			 hi->loads[0], hi->loads[1], hi->loads[2],
			 rel_types[hi->reliability]);

        XmLGridSetStringsPos (Hostlist, XmCONTENT, row, XmCONTENT, 1, buf);
        host->hostinfo_updated = FALSE;
}

int host_compare_fn (void *user_data, void *l,void *r)
{
        int type = *(int *)user_data;
        struct host_info *left  = *(struct host_info **) l;
        struct host_info *right = *(struct host_info **) r;

        switch (type) {
        case HOST_SORT_NONE:
                return (left - right);
        case HOST_SORT_PERF_MAX:
		return (right->hostinfo->performance*100 - left->hostinfo->performance*100);
        case HOST_SORT_PERF_AVAIL:
                if (!left->hostinfo) return -1;
                if (!right->hostinfo) return 1;
                return ((right->hostinfo->CPU_available/1e4) * right->hostinfo->performance -
                        (left->hostinfo->CPU_available/1e4) * left->hostinfo->performance);
        }
        return 0;
}

void hostlist_sort (void)
{
        static int *newlist = NULL;
        static struct host_info **oldlist = NULL;
        int i, sort_type = hostlist_sort_type;

	if (sort_type == HOST_SORT_NONE)
                return;
	
        if (!newlist)
                newlist = malloc (nhosts*sizeof(int));
        if (!oldlist)
                oldlist = malloc (nhosts*sizeof(struct host_info *));

        memcpy (oldlist, hostlist_sorted, nhosts*sizeof(struct host_info *));

        XmLSort (hostlist_sorted, nhosts, sizeof (struct host_info *), host_compare_fn, &sort_type);

        for (i = 0; i < nhosts; i++)
                newlist[i] = hostlist_sorted[i]->hostlist_row;

        if (memcmp (oldlist, hostlist_sorted, nhosts*sizeof(struct host_info *)))
                XmLGridReorderRows (Hostlist, newlist, 0, nhosts);

        for (i = 0; i < nhosts; i++)
                hostlist_sorted[i]->hostlist_row = i;
}

void update_curve_time (XYCurve *curve, long tdiff)
{
        XYPoint *p, *q;

        for (p = curve->points; p < curve->points + curve->nPoints; p++)
                p->x -= tdiff;

        for (p = curve->points; p < curve->points + curve->nPoints; p++)
                if (p->x >= -time_scale)
                        break;

        if (p == curve->points)
                return;

        for (q = curve->points; p < curve->points + curve->nPoints; p++, q++)
                *q = *p;

        curve->nPoints = q - curve->points;
}

XYCurve *find_curve (XYCurve *curves, int nCPU_curves, XmString name)
{
        for (; nCPU_curves; nCPU_curves--, curves++)
                if (curves->name == name)
                        return curves;
        return NULL;
}

void add_value_to_curve (XYCurve *curve, float value, float time)
{
        XYPoint *p, *newp;

        if (!curve)
                return;

        newp = realloc (curve->points, (curve->nPoints + 1) * sizeof (XYPoint));
	if (!newp) {
		printf("not enough memory\n");
                return;
	}
        p = newp+curve->nPoints;
        
        p->y = value;
        p->x = time;

        curve->nPoints++;
        curve->points = newp;
}

void CPU_curve_update (struct host_info *host)
{
        float tdiff = tvaltosec (current_time) - old_time.tv_sec;
        float value = (host->hostinfo->CPU_available*host->hostinfo->performance) / 1e4;

        if (!host->CPU_curve) return;

        add_value_to_curve (host->CPU_curve, value, tdiff);
}

void net_curve_update (XmString name, unsigned long value)
{
        float tdiff = tvaltosec (current_time) - old_time.tv_sec;

        add_value_to_curve (find_curve (net_curves, nnet_curves, name),
                            value/1024., tdiff);
}

void update_curves (void)
{
        struct host_info *h;
        struct cis_netdevinfo *dev;
        unsigned long tdiff = current_time.tv_sec - old_time.tv_sec;

        if (tdiff)
                for (h = hostlist; h; h = h->next) {
                        if (!h->flag) continue;

                        update_curve_time (h->CPU_curve, tdiff);

                        for (dev = h->devtab; dev < h->devtab + h->ndev; dev++)
                                update_curve_time (find_curve (net_curves, nnet_curves, dev->priv),
                                                   tdiff);
                }

        old_time = current_time;
        
        XYUpdateCurveData (CPU_plot, CPU_curves, XY_NO_RESCALE, True);
        XYUpdateCurveData (net_plot, net_curves, XY_NO_RESCALE, True);
}


int proc_compare_fn (void *user_data, void *l,void *r)
{
        int type = *(int *)user_data;
        struct cis_procinfo *left  = *(struct cis_procinfo **) l;
        struct cis_procinfo *right = *(struct cis_procinfo **) r;
        int result;

        switch (type) {
        case PROC_SORT_TREE:
                if ((result = left->ppid - right->ppid))
                        return (result);
        case PROC_SORT_PID:
                return (left->pid - right->pid);
                break;
        case PROC_SORT_PCPU:
                return (right->pCPU - left->pCPU);
                break;
        case PROC_SORT_RSS:
                return (right->rss - left->rss);
                break;
        case PROC_SORT_VM:
                return (right->vm - left->vm);
                break;
        }
        return 0;
}

int proc_search_fn (const void *key, const void *memb)
{
        int ppid = *(int *) key;
        struct cis_procinfo *proc = *(struct cis_procinfo **) memb;

        return (ppid - proc->pid);
}

struct cis_procinfo *proc_find_parent(struct cis_procinfo *proclist, int ppid)
{
        struct cis_procinfo *p;

        for (p = proclist; p->flags; p++)
                if (p->pid == ppid)
                        return p;

        return NULL;
}

void proc_tree_change(int row, int column, struct cis_procinfo *p)
{
        static char buf[128];

        
        switch (column) {
        case PROC_COL_CMD:
                sprintf (buf, "%s", p->cmd);
                break;
        case PROC_COL_USER:
                if (p->priv)
                        sprintf (buf, " %s", ((struct user_info *) p->priv)->name);
                else
                        sprintf (buf, " %d", p->uid);
                break;
        case PROC_COL_PRIO:
                sprintf (buf, "%ld ", p->priority);
                break;
        case PROC_COL_RSS:
                lprintf (buf, "%ld ", p->rss);
                break;
        case PROC_COL_VM:
                lprintf (buf, "%ld ", p->vm);
                break;
        case PROC_COL_PCPU:
                sprintf (buf, "%5.2f ", p->pCPU/100.);
                break;
        }
        XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, column, buf);
}

void proc_tree_sort (struct host_info *host)
{
        static struct cis_procinfo **oldtree = NULL;
        static struct cis_procinfo **newtree = NULL;
        static int *array = NULL;
        static int *levels = NULL;

        struct cis_procinfo *p, **pp, *root, *last;
        int i,j;
        XmLGridRow rowp;

        if (!host->proc_number)
                return;
        
        oldtree  = realloc (oldtree, host->proc_number * sizeof (struct cis_procinfo *));
        newtree  = realloc (newtree, host->proc_number * sizeof (struct cis_procinfo *));
        array    = realloc (array, host->proc_number * sizeof (int));
        levels   = realloc (levels, host->proc_number * sizeof (int));

        /*
         * Get the last state.
         */
        for (i = 0; i < host->proc_number; i++) {
                rowp = XmLGridGetRow (Proclist, XmCONTENT, host->proc_offset+i+1);
                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNrowUserData, &oldtree[i],
                              XmNrowLevel, &levels[i],
                              NULL);

                oldtree[i]->flags = 1; /* level for all sorts except tree */
        }
        /*
         * Create new tree.
         */
        memcpy (newtree, oldtree, host->proc_number*sizeof (struct cis_procinfo *));
        XmLSort (newtree, host->proc_number, sizeof (struct cis_procinfo *), proc_compare_fn, &proclist_sort_type);

        if (proclist_sort_type == PROC_SORT_TREE) {
                int level;
                
                /*
                 * Clean the tree.
                 */
                for (i = 0; i < host->proc_number; i++) {
                        newtree[i]->flags = FALSE;
                        newtree[i]->next = NULL;
                        newtree[i]->prev = NULL;
                        newtree[i]->children = NULL;
                }
                /*
                 * Create branches.
                 */
                for (i = 0; i < host->proc_number-1; i++)
                        if (newtree[i]->ppid == newtree[i+1]->ppid) {
                                newtree[i]->next = newtree[i+1];
                                newtree[i+1]->prev = newtree[i];
                                newtree[i+1]->flags = TRUE;
                        }

                /*
                 * Connect branches together.
                 */
                for (i = 0; i < host->proc_number; i++) {
                        if (host->proclist[i]->flags)
                                continue;

                        if ((pp = bsearch (&host->proclist[i]->ppid, host->proclist, host->proc_number,
                                           sizeof (struct cis_procinfo *), proc_search_fn))) {
                                p = *pp;

                                p->children = host->proclist[i];
                                host->proclist[i]->prev = p;
                                host->proclist[i]->flags = TRUE;
                        }
                }
                /*
                 * Finally, connect together level 0.
                 */
                root = NULL;
                last = NULL;
                for (i = 0; i < host->proc_number; i++) {
                        if (host->proclist[i]->flags)
                                continue;

                        p = host->proclist[i];
                        
                        if (!root)
                                root = last = p;
                        else {
                                last->next = p;
                                p->prev = last;
                        }
                        while (last->next)
                                last = last->next;

                        p->flags = TRUE;
                }

                /*
                 * Store the tree in newtree. Flags are used to store new
                 * levels (why not).
                 */
                p = root;
                i = 0;
                level = 1;
                while (p) {
                        newtree[i++] = p;
                        p->flags = level;
                        if (i == host->proc_number)
                                break;

                        if (p->children) {
                                p = p->children;
                                level++;
                                continue;
                        }
                        if (p->next) {
                                p = p->next;
                                continue;
                        }
                        while (p->prev) {
                                if (p->prev->children == p) {
                                        level--;
                                        if (p->prev->next)
                                                break;
                                }
                                p = p->prev;
                        }

                        if (p->prev)
                                p = p->prev->next;
                }
        }
        /*
         * No change.
         */
        if (!memcmp (&oldtree, &newtree, host->proc_number * sizeof (struct cis_procinfo *)))
                return;

        /*
         * Create a table for reordering rows in process tree.
         */
        for (i = 0; i < host->proc_number; i++)
                for (j = 0; j < host->proc_number; j++)
                        if (newtree[i] == oldtree[j])
                                array[i] = host->proc_offset+j+1;

        XmLGridReorderRows (Proclist, array, host->proc_offset+1, host->proc_number);

        /*
         * Set the levels.
         */
        for (i = 0; i < host->proc_number; i++)
                if (newtree[i]->flags != levels[array[i]-host->proc_offset-1])
                        XtVaSetValues(Proclist,
                                      XmNrow, host->proc_offset+i+1,
                                      XmNrowLevel, newtree[i]->flags,
                                      NULL);
}

struct cis_procinfo *proc_find (struct cis_procinfo *tab, int len, int pid)
{
        int i;
        for (i = 0; i < len; i++, tab++)
                if (tab->pid == pid)
                        return tab;
        return NULL;
}

void proctab_update (struct host_info *host, struct cis_procinfo *ptab, int len)
{
        free (host->proctab);

        host->proctab = ptab;
        host->nproc = len;

        host->proc_updated = TRUE;
}

void proclist_update (struct host_info *host)
{
        struct cis_procinfo *p, *newp, **new_proclist;
        struct host_info *h;
        int row;
        int redraw_tree = FALSE;
        XmLGridRow rowp;
        XmLGridColumn colp1, colp2;
        char buf[128], *c;
        static int sort = PROC_SORT_PID;
        XmString s;

        if (host->nproc != host->proc_number) {
                new_proclist = realloc (host->proclist, host->nproc * sizeof (struct cis_procinfo *));
                if (host->nproc && !new_proclist)
                        return;
                host->proclist = new_proclist;

                for (h = host->next; h; h = h->next)
                        h->proc_offset += host->nproc - host->proc_number;
        }

        for (p = host->proctab; p < host->proctab + host->nproc; p++)
                p->flags = TRUE;

        colp1 = XmLGridGetColumn (Proclist, XmCONTENT, PROC_COL_UPTIME);
        colp2 = XmLGridGetColumn (Proclist, XmCONTENT, PROC_COL_USEDTIME);

        for (row = host->proc_offset+1; row < host->proc_offset+host->proc_number+1;) {
                rowp = XmLGridGetRow (Proclist, XmCONTENT, row);
                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNrowUserData, &p,
                              NULL);

                if (!(newp = proc_find (host->proctab, host->nproc, p->pid))) {

                        XmLGridDeleteRows (Proclist, XmCONTENT, row, 1);
                        if (proclist_sort_type == PROC_SORT_TREE)
                                redraw_tree = TRUE;

                        free (p);
                        host->proc_number--;

                        continue;
                }

                host->proclist[row - host->proc_offset - 1] = p;
                        
                if (strcmp (newp->cmd, p->cmd))
                        proc_tree_change (row, PROC_COL_CMD , newp);

                if (newp->ppid != p->ppid && proclist_sort_type == PROC_SORT_TREE)
                        redraw_tree = TRUE;

                if (newp->uid != p->uid)
                        proc_tree_change (row, PROC_COL_USER , newp);

                if (newp->priority != p->priority)
                        proc_tree_change (row, PROC_COL_PRIO, newp);

                if (newp->rss != p->rss) {
                        proc_tree_change (row, PROC_COL_RSS , newp);
                        if (proclist_sort_type == PROC_SORT_RSS)
                                redraw_tree = TRUE;
                }
                if (newp->vm != p->vm) {
                        proc_tree_change (row, PROC_COL_VM , newp);
                        if (proclist_sort_type == PROC_SORT_VM)
                                redraw_tree = TRUE;
                }
                if (newp->pCPU != p->pCPU) {
                        proc_tree_change (row, PROC_COL_PCPU, newp);
                        if (proclist_sort_type == PROC_SORT_PCPU)
                                redraw_tree = TRUE;
                }

                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNcolumnPtr, colp1,
                              XmNcellString, &s,
                              NULL);
                c = string_to_text(s);
                tprintf (buf, "%ld ", current_time.tv_sec - p->start_time);
                if (strcmp(c, buf))
                        XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, PROC_COL_UPTIME, buf);
                free (c);

                XtVaGetValues(Proclist,
                              XmNrowPtr, rowp,
                              XmNcolumnPtr, colp2,
                              XmNcellString, &s,
                              NULL);
                c = string_to_text(s);
                tprintf (buf, "%ld ", p->utime+p->stime);
                if (strcmp(c, buf))
                        XmLGridSetStringsPos (Proclist, XmCONTENT, row, XmCONTENT, PROC_COL_USEDTIME, buf);
                free (c);

                
                newp->priv = p->priv;
                *p = *newp;

                newp->flags = FALSE;
                row++;
        }

        /*
         * Insert new processes into the process tree.
         */
        for (p = host->proctab; p < host->proctab + host->nproc; p++) {
                if (!p->flags) continue;

                p->priv = find_user (host, p->uid);
                
                newp = malloc (sizeof (struct cis_procinfo));
                *newp = *p;


                XmLTreeAddRow (Proclist, 1, False, True, row, pixmaps[DOT], pixmasks[DOT], NULL);
                XtVaSetValues (Proclist,
                               XmNrow, row,
                               XmNrowExpands, False,
                               XmNrowUserData, newp,
                               NULL);

                c = buf;
                c += sprintf (c, "%s|%d ", p->cmd, p->pid);
                if (p->priv)
                        c += sprintf (c, "| %s", ((struct user_info *) p->priv)->name);
                else
                        c += sprintf (c, "| %d", p->uid);

                c += sprintf (c, "|%ld ", p->priority);
                c += lprintf (c, "|%ld |%ld ", p->rss, p->vm);
                c += sprintf (c, "|%5.2f ", p->pCPU/100.);
                c += tprintf (c, "|%ld ", current_time.tv_sec - p->start_time);
                c += tprintf (c, "|%ld ", p->utime + p->stime);

                XmLGridSetStringsPos (Proclist,
                                      XmCONTENT, row,
                                      XmCONTENT, 0,
                                      buf);

                redraw_tree = TRUE;

                host->proclist[row - host->proc_offset - 1] = newp;
                
                row++;
                host->proc_number++;
        }
        XmLSort (host->proclist, host->proc_number, sizeof (struct cis_procinfo *), proc_compare_fn, &sort);

        if (redraw_tree)
                proc_tree_sort (host);

        host->proc_updated = FALSE;
}

struct cis_sockinfo *sock_find (struct cis_sockinfo *tab, int len, struct cis_sockinfo *s)
{
        int i;
        
        for (i = 0; i < len; i++, tab++)
                if (tab->type         == s->type &&
                    tab->saddr.s_addr == s->saddr.s_addr &&
                    tab->sport        == s->sport &&
                    tab->daddr.s_addr == s->daddr.s_addr &&
                    tab->dport        == s->dport)
                        return tab;
        return NULL;
}

struct cis_sockinfo *sock_find_dest (struct cis_sockinfo *tab, int len, struct cis_sockinfo *s)
{
        int i;
        
        for (i = 0; i < len; i++, tab++)
                if (tab->type         == s->type &&
                    tab->saddr.s_addr == s->daddr.s_addr &&
                    tab->sport        == s->dport /*&&
                    tab->daddr.s_addr == s->saddr.s_addr &&
                    tab->dport        == s->sport*/)
                        return tab;
        return NULL;
}

void socktab_update (struct host_info *host, struct cis_sockinfo *stab, int len)
{
        free (host->socktab);

        host->socktab = stab;
        host->nsock = len;

        host->sock_updated = TRUE;
}

void socklist_update (struct host_info *host)
{
        struct cis_sockinfo *s, *socket, *oldtab;
        struct link_info **l, *link;
        struct host_info *h;
        unsigned long diff;

        host->send_rate = 0;
        host->recv_rate = 0;
        
        for (l = host->linklist; l < host->linklist + host->link_number; l++) {
                (*l)->flag = FALSE;
                (*l)->is_updated = TRUE;
        }

        oldtab = malloc (host->nsock * sizeof (struct cis_sockinfo));
        if (host->nsock && !oldtab)
                return;

        memcpy (oldtab, host->socktab, host->nsock * sizeof (struct cis_sockinfo));

        for (s = oldtab; s < oldtab + host->nsock; s++) {

		s->saddr = host->addrlist[0];
		if (!s->daddr.s_addr || s->daddr.s_addr == 0x0100007f)
			s->daddr = host->addrlist[0];
		else if ((h = find_host(s->daddr)) && s->daddr.s_addr != h->addrlist[0].s_addr)
			s->daddr = h->addrlist[0];

		if (s->saddr.s_addr != s->daddr.s_addr) {
                        host->send_rate += s->sent;
                        host->recv_rate += s->rcvd;
                }

                s->flags = TRUE;

                socket = sock_find (host->socktab_old, host->nsock_old, s);

                if (!socket || !socket->priv)
                        continue;

                s->priv = socket->priv;
                link = (struct link_info *) s->priv;
                link->flag = TRUE;
                link->is_updated = FALSE;
                if ((diff = s->sent - socket->sent)) {
                        link->sent      += diff;
                        link->is_updated = TRUE;
                }
                if ((diff = s->rcvd - socket->rcvd)) {
                        link->rcvd      += diff;
                        link->is_updated = TRUE;
                }
                if (s->pid != socket->pid) {
                        link->src_pid    = s->pid;
                        link->is_updated = TRUE;

			for (h = hostlist; h; h = h->next) {
                                socket = sock_find_dest(h->socktab_old, h->nsock_old, s);
				if (!socket)
					continue;
                                link = socket->priv;
                                if (link)
                                        link->dst_pid = s->pid;
                        }
                }
                s->flags = FALSE;
        }

        free (host->socktab_old);
        host->socktab_old = oldtab;
        host->nsock_old = host->nsock;

        host->sock_updated = FALSE;
}

struct link_info *link_find (struct host_info *src_host, struct cis_sockinfo *s1, struct host_info *dst_host, struct cis_sockinfo *s2)
{
        struct link_info *link, **l;

        for (l = src_host->linklist; l < src_host->linklist + src_host->link_number; l++) {
                link = *l;
                if (link->src_pid != s1->pid)
                        continue;
                if (link->type != s1->type)
                        continue;
                if (dst_host) {
                        if (link->dst_host != dst_host)
                                continue;
                } else if (link->dst_addr.s_addr != s1->daddr.s_addr)
                        continue;

                if (s2) {
                        if (s2->pid != link->dst_pid)
                                continue;
                } else if (s1->dport != link->dst_port)
                        continue;

                return link;
        }
        return NULL;
}
        
void linklist_update (void)
{
        struct host_info *h, *host;
        struct cis_sockinfo *s, *socket;
        struct link_info **l, *link;
        int offset = 0;
        int redraw_tree = FALSE;
        XmLGridRow rowp;
        static char buf[128];
        int row, i, len;

        /* Find destination process and try to find the appropriate link */
	for (h = hostlist; h; h = h->next) {
		h->link_offset += offset;
                if (!h->nsock_old) continue;

                for (s = h->socktab_old; s < h->socktab_old + h->nsock_old; s++) {
                        link = s->priv;

                        if (!s->flags && (!link->dst_port || link->dst_pid))
                                continue;

                        socket = NULL;
                        
                        host = find_host (s->daddr);
                        if (host)
                                socket = sock_find_dest (host->socktab_old, host->nsock_old, s);

                        if (link) {
                                if (socket) {
                                        link->dst_pid = socket->pid;
                                        link->is_updated = TRUE;
                                }
                                continue;
                        }
                        
                        if ((link = link_find (h, s, host, socket))) {
                                s->priv = link;

                                link->sent += s->sent;
                                link->rcvd += s->rcvd;
                                link->flag = TRUE;
                                link->is_updated = TRUE;

                                s->flags = FALSE;
                                continue;
                        }

                        link = calloc (1, sizeof (struct link_info));
                        link->flag = TRUE;
                        link->type = s->type;
                        link->src_pid = s->pid;
                        link->dst_host = host;
                        if (!host)
                                link->dst_addr = s->daddr;
                        if (!socket)
                                link->dst_port = s->dport;
                        else
                                link->dst_pid  = socket->pid;

                        link->sent = s->sent;
                        link->rcvd = s->rcvd;

                        link->is_updated = TRUE;
                        s->flags = FALSE;

                        s->priv = link;

                        XmLTreeAddRow (Linklist, 1, False, True, h->link_offset+h->link_number+1, pixmaps[DOT], pixmasks[DOT], NULL);
                        XtVaSetValues (Linklist,
                                       XmNrow, h->link_offset+h->link_number+1,
                                       XmNrowExpands, False,
                                       XmNrowUserData, link,
                                       NULL);
                        
                        len = sprintf (buf, "%d |", link->src_pid);
                        len += sprintf (buf+len, "%s", host ? host->name : inet_ntoa (link->dst_addr));
                        if (!socket)
                                len += sprintf (buf+len, ":%-4X ||", link->dst_port);
                        else
                                len += sprintf (buf+len, " |%d |", link->dst_pid);

                        len += sprintf (buf+len, "%s |", (link->type == SOCK_STREAM) ? "TCP" : "UDP");
                        len += lprintf (buf+len, "%ld |%ld ",
                                        link->sent, link->rcvd);


                        XmLGridSetStringsPos (Linklist,
                                              XmCONTENT, h->link_offset+h->link_number+1,
                                              XmCONTENT, 0,
                                              buf);

                        redraw_tree = TRUE;
                        offset++;

                        h->linklist = realloc (h->linklist, (h->link_number+1) *
					      sizeof (struct link_info *));
			h->linklist[h->link_number] = link;
			h->link_number++;
                }
        }

        /* Now it's time to update link tree and remove deleted links.*/

        row = 0;
	for (h = hostlist; h; h = h->next) {
		h->link_offset = row;
                if (h->flag)
                        row++;
                
                l = h->linklist;
                
                for (i = 0; i < h->link_number; i++) {

                        rowp = XmLGridGetRow (Linklist, XmCONTENT, row);
                        XtVaGetValues(Linklist,
                                      XmNrowPtr, rowp,
                                      XmNrowUserData, &link,
				      NULL);

                        if (!link->flag) {
                                XmLGridDeleteRows (Linklist, XmCONTENT, row, 1);

                                free (link);
                                continue;
                        }
                        if (link->is_updated) {
                                len = sprintf (buf, "%s", link->dst_host ? link->dst_host->name : inet_ntoa (link->dst_addr));
                                if (!link->dst_pid)
                                        len += sprintf (buf+len, ":%-4X ||", link->dst_port);
                                else
                                        len += sprintf (buf+len, " |%d |", link->dst_pid);

                                len += sprintf (buf+len, "%s |", (link->type == SOCK_STREAM) ? "TCP" : "UDP");
                                len += lprintf (buf+len, "%ld |%ld ",
                                                link->sent, link->rcvd);
                                XmLGridSetStringsPos (Linklist, XmCONTENT, row, XmCONTENT, LINK_COL_DSTHOST, buf);
                        }

                        row++;
                        *l = link;
                        l++;
                }

                lprintf (buf, "%ld |%ld ", h->send_rate, h->recv_rate);
                XmLGridSetStringsPos (Linklist, XmCONTENT, h->link_offset, XmCONTENT, LINK_COL_SENT, buf);
                
                h->link_number = l - h->linklist;
        }
}

void dev_tree_change(int row, int column, struct cis_netdevinfo *d)
{
        static char buf[128];

        switch (column) {
        case DEV_COL_STATUS:
                sprintf (buf, "%s", d->status ? "up" : "down");
                break;
                
        case DEV_COL_RX:
                lprintf (buf, "%ld ", d->rx_bytes);
                break;
                
        case DEV_COL_TX:
                lprintf (buf, "%ld ", d->tx_bytes);
                break;
                
        case DEV_COL_COLL:
                lprintf (buf, "%ld ", d->collisions);
                break;
        }

        XmLGridSetStringsPos (Devlist, XmCONTENT, row, XmCONTENT, column, buf);
}

struct cis_netdevinfo *dev_find (struct cis_netdevinfo *tab, int len, char *name)
{
        int i;
        for (i = 0; i < len; i++, tab++)
                if (!strncmp (tab->name, name, IFNAMSIZ))
                        return tab;
        return NULL;
}


void devtab_update (struct host_info *host, struct cis_netdevinfo *dtab, int len)
{
        struct cis_netdevinfo *d, *dev;
        XYCurve *c;
        XmString s;
        char buf[128];

        for (d = host->devtab; d < host->devtab+host->ndev; d++)
                d->flags = FALSE;

        for (dev = dtab; dev < dtab+len; dev++) {
                if (!(d = dev_find (host->devtab, host->ndev, dev->name))) {

                        sprintf (buf, "%s:%s", host->name, dev->name);

                        s = XmStringCreateSimple(buf);
                        dev->priv = s;

                        net_curves = realloc (net_curves, (nnet_curves+1)* sizeof (XYCurve));
                        c = net_curves + nnet_curves;
                        memset (c, 0, sizeof (XYCurve));
			c->lineStyle = ((dev-dtab) % (XY_N_LINE_STYLES-1))+1;
                        c->name   = s;
                        c->points = NULL;
			c->nPoints = 0;
			c->markerPixel = host->pixel;
			c->linePixel   = host->pixel;

			nnet_curves++;
                } else {
                        dev->priv = d->priv;
                        d->flags = TRUE;
                }
                        
                net_curve_update (dev->priv, dev->tx_bytes);
        }

        for (d = host->devtab; d < host->devtab+host->ndev; d++)
                if (!d->flags) {
                        c = find_curve (net_curves, nnet_curves, d->priv);
                        for (c++; c < net_curves+nnet_curves; c++)
				*(c-1) = *c;

                        nnet_curves--;
                        XmStringFree ((XmString) d->priv);
                }

        free (host->devtab);

        host->devtab = dtab;
        host->ndev = len;

        host->dev_updated = TRUE;
}

void devlist_update (struct host_info *host)
{
        struct cis_netdevinfo *d, *newd;
        int diff;
        struct host_info *h;
        int row;
        XmLGridRow rowp;
        char buf[128];
        int len;

        if ((diff = host->ndev - host->dev_number))
                for (h = host->next; h; h = h->next)
                        h->dev_offset += diff;

        for (d = host->devtab; d < host->devtab + host->ndev; d++)
                d->flags = TRUE;

        for (row = host->dev_offset; row < host->dev_offset+host->dev_number;) {
                rowp = XmLGridGetRow (Devlist, XmCONTENT, row);
                XtVaGetValues(Devlist,
                              XmNrowPtr, rowp,
                              XmNrowUserData, &d,
                              NULL);

                if (!(newd = dev_find (host->devtab, host->ndev, d->name))) {

                        XmLGridDeleteRows (Devlist, XmCONTENT, row, 1);

                        free (d);
                        host->dev_number--;

                        XYSetCurves (net_plot, net_curves, nnet_curves, XY_NO_RESCALE, True);
			continue;
                }

                if (newd->status != d->status)
                        dev_tree_change (row, DEV_COL_STATUS , newd);
                if (newd->rx_bytes != d->rx_bytes)
                        dev_tree_change (row, DEV_COL_RX , newd);
                if (newd->collisions != d->collisions)
                        dev_tree_change (row, DEV_COL_COLL , newd);
                if (newd->tx_bytes != d->tx_bytes)
                        dev_tree_change (row, DEV_COL_TX , newd);

                *d = *newd;
                
                newd->flags = FALSE;
		row++;
        }

        /*
         * Insert new devices into the device list and netplot.
         */
        for (d = host->devtab; d < host->devtab + host->ndev; d++) {
                if (!d->flags) continue;

                newd = malloc (sizeof (struct cis_netdevinfo));
                *newd = *d;

                XmLTreeAddRow (Devlist, 0, False, True, row, pixmaps[DOT], pixmasks[DOT], NULL);
                XtVaSetValues (Devlist,
                               XmNrow, row,
                               XmNrowExpands, False,
                               XmNrowUserData, newd,
                               NULL);

		len = sprintf (buf, "%s|%s|%s|", host->name, d->name, d->status ? "up" : "down");
                lprintf (buf+len, "%ld |%ld |%ld ", d->tx_bytes, d->rx_bytes, d->collisions);

                XmLGridSetStringsPos (Devlist,
                                      XmCONTENT, row,
                                      XmCONTENT, 0,
                                      buf);

                row++;
                host->dev_number++;
                XYSetCurves (net_plot, net_curves, nnet_curves, XY_NO_RESCALE, True);
        }


        host->dev_updated = FALSE;
}

void clear_host_data (struct host_info *host)
{
	XYCurve *c;

	host->hostinfo_updated = TRUE;

        if (host->flag) {
		c = host->CPU_curve;
		c->nPoints = 0;
                free (c->points);
		c->points = NULL;
	}

        proctab_update  (host, NULL, 0);
        socktab_update  (host, NULL, 0);
        devtab_update   (host, NULL, 0);

        host->send_rate = 0;
        host->recv_rate = 0;
}

void clear_data (void)
{
        struct host_info *h;

        /* Clear tables */
	for (h = hostlist; h; h = h->next)
                clear_host_data (h);

        update_curves ();
        update_time ();
}

inline void host_error (struct host_info *host, int err)
{
        if (err > HOST_NOT_AVAILABLE) {
                cisDisconnect(host->cis->handle);
                host->cis->handle = NULL;
        }
}

void allarm_handler (int sig)
{
        struct host_info *h;
        struct cis_info *cis;
        static struct cis_procinfo *ptab = NULL;
        static struct cis_sockinfo *stab = NULL;
        static struct cis_netdevinfo *dtab = NULL;
        struct cis_hostinfo *hi;
        int len;
        char *changes = NULL;
        int chlen = 0;

        gettimeofday (&current_time, NULL);

        /*
         * Try to connect to all disconnected cis daemons.
         */
	for (cis = cislist; cis; cis = cis->next) {
		if (!cis->handle)
                        cis->handle = cisConnect (cis->name);
        }

        /*
         * Archive previous host information.
         */
	for (h = hostlist; h; h = h->next)
		*h->old_hostinfo = *h->hostinfo;

        /*
	 * Update dynamic host information.
	 */
	for (cis = cislist; cis; cis = cis->next) {
                if (cis->handle)
                        cisUpdateHostsInfo (cis->handle, CIS_ALL, cis->htab, cis->nhosts);
	}

	/*
         * Update static host information if host became available.
	 */
	for (h = hostlist; h; h = h->next) {
		if (!h->cis ||
		    h->hostinfo->status == h->old_hostinfo->status ||
		    h->hostinfo->status != HOST_AVAILABLE)
			continue;

		hi = cisHostInfo (h->cis->handle, h->name, perf_type);
		if (hi) {
			hi->performance = h->hostinfo->performance;
			*h->hostinfo = *hi;
			free (hi);
		}
	}

	for (h = hostlist; h; h = h->next) {
                ptab = NULL;
                stab = NULL;
                dtab = NULL;
                
                if (mode == RECORD && h->flag) {
                        chlen = cisHostinfoPrepareChanges (h->old_hostinfo, h->hostinfo, &changes);

                        if (h->hostinfo_saved_bytes >= record.density) {
                                cisSaveHostinfo (record.file, h->hostaddr, h->hostinfo);
                                h->hostinfo_saved_bytes = 0;
                        } else {
				cisSaveHostinfoChanges (record.file, h->hostaddr, changes, chlen);
				h->hostinfo_saved_bytes += chlen;
                        }

                        free (changes);
                }

                h->hostinfo_updated = TRUE;

                if (h->flag)
                        CPU_curve_update(h);

                /*
                 * Processes
                 */
                if (h->flag && h->cis && h->cis->handle) {
                        len = cisProcessList (h->cis->handle, &ptab, h->hostaddr, h->uid);
                        if (len < 0) {
                                host_error (h, len);
                                len = 0;
                        }
                } else
                        len = 0;

                chlen = cisProcPrepareChanges (h->proctab, h->nproc, ptab, len, &changes);

                if (mode == RECORD) {
                        if (h->proc_saved_bytes >= record.density) {
                                cisSaveProcinfo (record.file, h->hostaddr, ptab, len);
                                h->proc_saved_bytes = 0;
                        } else if (chlen) {
                                cisSaveProcChanges (record.file, h->hostaddr, changes, chlen);
                                h->proc_saved_bytes += chlen;
                        }
                }

                if (chlen)
                        free (changes);

                h->proc_updated = TRUE;
                proctab_update (h, ptab, len);

                /*
                 * Sockets
                 */
                if (h->flag && h->cis && h->cis->handle) {
                        len = cisSocketList (h->cis->handle, &stab, h->hostaddr, h->uid, h->interval);
                        if (len < 0) {
                                host_error (h, len);

                                h->links_available = FALSE;
                                len = 0;
                        } else
                                h->links_available = TRUE;
                } else
                        len = 0;

                chlen = cisSockPrepareChanges (h->socktab, h->nsock, stab, len, &changes);

                if (mode == RECORD) {
                        if (h->sock_saved_bytes >= record.density) {
                                cisSaveSockinfo (record.file, h->hostaddr, stab, len);
                                h->sock_saved_bytes = 0;
                        } else {
                                cisSaveSockChanges (record.file, h->hostaddr, changes, chlen);
                                h->sock_saved_bytes += chlen;
                        }
                }

                if (changes) {
                        h->sock_updated = TRUE;
                        free (changes);
                }
                socktab_update (h, stab, len);
                
                /*
                 * Network devices
                 */
                if (h->flag && h->cis && h->cis->handle) {
                        len = cisNetdevList (h->cis->handle, &dtab, h->hostaddr, h->interval);
                        if (len < 0) {
                                host_error (h, len);
                                len = 0;
                        }
                } else
                        len = 0;

                chlen = cisNetdevPrepareChanges (h->devtab, h->ndev, dtab, len, &changes);

                if (mode == RECORD) {
                        if (h->dev_saved_bytes >= record.density) {
                                cisSaveNetdevinfo (record.file, h->hostaddr, dtab, len);
                                h->dev_saved_bytes = 0;
                        } else {
                                cisSaveNetdevChanges (record.file, h->hostaddr, changes, chlen);
                                h->dev_saved_bytes += chlen;
                        }
                }

                if (changes) {
                        h->dev_updated = TRUE;
                        free (changes);
                }
                devtab_update (h, dtab, len);
        }

        XtNoticeSignal (sigid);
}

void allarm_cb (XtPointer cl_data, XtSignalId *id)
{
        struct host_info *h;
        int update_links = FALSE;

	for (h = hostlist; h; h = h->next) {
                if (h->hostinfo_updated)
                        hostlist_update (h);
                if (h->proc_updated)
                        proclist_update (h);
                if (h->sock_updated) {
                        socklist_update (h);
                        update_links = TRUE;
                }
                if (h->dev_updated)
                        devlist_update (h);
        }
        if (update_links)
                linklist_update();

        if (hostlist_sort_type != HOST_SORT_NONE)
                hostlist_sort ();

        update_curves ();
        update_time ();
}

int main (int argc,char *argv[])
{
	XYCurve *curve;
	XmLGridCallbackStruct cbs;
	
	int row;
        struct host_info *h;

        struct sigaction allarm_action;
        struct itimerval it;

        setlocale (LC_ALL, "");
        setlocale (LC_NUMERIC, "C");

	if (gethostname ((char *)& myhostname, MAXHOSTNAMELEN) < 0)
                fatal_err ("cannot get my hostname");

	/*
         * Parse command line for config file name.
         */
        getparams (argc, argv);
        
        /*
         * Parse hostfile for host list and other parameters.
         */
        parse_configfile();

	if (!nhosts)
		fatal_err("no host to monitor");
	
	CPU_curves = calloc (nhosts, sizeof (XYCurve));
	for (curve = CPU_curves; curve < CPU_curves + nhosts; curve++)
		curve->lineStyle = XY_PLAIN_LINE;

	CPU_scale = 0.;
	for (h = hostlist; h; h = h->next)
		if (h->hostinfo->performance > CPU_scale)
			CPU_scale = h->hostinfo->performance;
	CPU_scale *= 1.5;

	if (!(hostlist_sorted = malloc (nhosts * sizeof (struct host_info *))))
		fatal_err ("not enough memory");

        /*
         * Initialization of signal action structures.
         */
        allarm_action.sa_handler = allarm_handler;
        sigemptyset (&allarm_action.sa_mask);
        allarm_action.sa_flags   = 0;

	create_window(argc, argv);

        /*
         * Initialize the list of hosts
	 */
	for (row = 0, h = hostlist; row < nhosts; row++, h = h->next) {
		hostlist_sorted[row] = h;

                XmLGridAddRows(Hostlist, XmCONTENT, row, 1);
		XtVaSetValues (Hostlist,
                               XmNcolumn, 0,
                               XmNrow, row,
                               XmNcellPixmap, pixmaps[CHECK],
			       XmNcellPixmapMask, pixmasks[UNCHECK],
			       XmNrowUserData, h,
                               NULL);
		h->hostlist_row = row;
		hostlist_update (h);
                h->proc_offset = 0;
                h->link_offset = 0;
		h->dev_offset = 0;
	}

	
	cbs.reason = XmCR_SELECT_CELL;
	cbs.rowType = XmCONTENT;

	for (h = hostlist, row = 0; h; h = h->next, row++) {
		if (!h->flag)
			continue;
		cbs.row = row;
		host_grid_select_cb(Hostlist, NULL, (XtPointer) &cbs);
	}
	hostlist_sort_type = HOST_SORT_NONE;
                        
	/*
         * Let's install handler for SIGALRM.
         */
        sigaction (SIGALRM, &allarm_action, NULL);

        /*
         * Set interval and reset timer.
         */
	sectotval (it.it_interval, interval);
        sectotval (it.it_value   , interval);

        setitimer (ITIMER_REAL, &it, NULL);

	sigid = XtAppAddSignal (app_context, allarm_cb, NULL);
        
        allarm_handler(0);

        XtAppMainLoop(app_context);

        /* Never reached */
        fatal_err ("Unexpected return from XtAppMainLoop");

        exit (0);
}

