/*
 * Xcis - X11/Lesstif client for Cluster Information Service
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * As a special exception you are allowed to link xcis executable with
 * Microline Widget Library and Plot Widgets library.
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <netdb.h>

#include "xcis.h"

extern float interval;
extern struct timeval interval_tval;
extern unsigned char reliability;
extern struct user_info *userlist;
extern struct cis_info *cislist;
extern struct host_info *hostlist;
extern int nhosts;
extern char *CPU_units;
extern char *printpath;
extern float time_scale;
extern float net_scale;
extern struct rec_info record;
extern char *perf_type;
extern char myhostname[];

static FILE *configfile = NULL;

static struct host_info *add_host (char *name)
{
	struct host_info *host;
        struct hostent *hent;
        int i;

	srand(200);
	
	if ((hent = gethostbyname (name)) == NULL)
		fatal_err("cannot get the address of host %s", name);

	if (!(host = calloc (1, sizeof (struct host_info))))
		goto nomem;

	for (i = 0; hent->h_addr_list[i]; i++)
		memcpy (&host->addrlist[i], hent->h_addr_list[i], hent->h_length);
	host->addrlist[i].s_addr = 0;

	host->hostinfo = calloc(1, sizeof(struct cis_hostinfo));
	if (!host->hostinfo)
		goto nomem;
	host->old_hostinfo = calloc (1, sizeof (struct cis_hostinfo));
	if (!host->old_hostinfo)
		goto nomem;

	host->next = hostlist;
	hostlist = host;
	nhosts++;
	return host;

nomem:
	fatal_err ("cannot allocate memory for host structures");
	return NULL;
}

void getparams (int argc, char **argv)
{
	int c;
	char **hlist, **h;
	int rel;
	struct host_info *host;
	struct cis_hostinfo *hi;
	int i;

	while ((c = getopt(argc, argv, "i:c:r:s:p:")) != EOF) {

                switch (c) {
                case 'i':
			interval = atof(optarg);
			if (interval == 0.) {
				printf ("interval cannot be set to zero. reset to default %5.2f [s]\n", DEF_INTERVAL);
				interval = DEF_INTERVAL;
			}
			interval_tval.tv_sec  = (int) interval;
			interval_tval.tv_usec = (interval - (int) interval) * 1000000;

                        break;
                case 'r':
			rel = parse_cmd(rel_types, REL_TABLEN, &optarg);
			if (rel == -1)
				fatal_err("wrong reliability type");
			reliability = rel;
			break;
                case 's':
			cislist = calloc(1, sizeof (struct cis_info));

			cislist->handle = cisConnect(optarg);
			if (!cislist->handle)
				fatal_err("cannot connect cis server %s", optarg);

			cislist->name = strdup(optarg);

			break;
                case 'c':
			if(!(configfile = fopen (optarg, "r")))
				fatal_err ("cannot open config file %s.", optarg);
			break;
                case 'p':
			perf_type = optarg;
			break;
		}
	}

	if (!configfile && !cislist) {
		cislist = calloc(1, sizeof (struct cis_info));

		cislist->handle = cisConnect(myhostname);
		if (!cislist->handle)
			fatal_err("cannot connect cis server %s", myhostname);

		cislist->name = strdup(myhostname);
	}
	
	if (cislist) {
		hlist = cisHostList(cislist->handle, reliability);
		if (!hlist)
			return;

		for (h = hlist; strlen(*h); h++) {
			host = add_host (*h);

			host->name = *h;
			host->cis = cislist;

			if (!cislist->handle || !(hi = cisHostInfo(cislist->handle, *h, perf_type))) {
				host->hostinfo->status = HOST_UNKNOWN;
				host->hostinfo->addr = host->hostaddr;
				continue;
			}
			*host->hostinfo = *hi;
			free (hi);

			host->interval = SOCK_INTERVAL;
		}
		nhosts = h - hlist;

		cislist->htab = calloc (nhosts,sizeof(struct cis_hostinfo *));
		if (!cislist->htab)
			fatal_err ("not enough memory");
		cislist->nhosts = nhosts;

		for (host = hostlist, i = 0; i < nhosts; i++, host = host->next)
			cislist->htab[i] = host->hostinfo;
	}
	free (hlist);
}

char *cmds[] = {"#", "CisServer", "Host", "User", "SpeedUnits", "TimeScale", "NetScale", "Interval", "RecPath", "RecDensity", "PrintPath"};

enum {
        COMMENT,
	CIS,
	HOST,
        USER,
        SPEEDUNITS,
        TIMESCALE,
        NETSCALE,
        INTERVAL,
        RECPATH,
        RECDENSITY,
	PRINTPATH,
	NCMDS,
};

void parse_configfile ()
{
        static int buflen = 0;
        static char *buf  = NULL;
        int cmd;
	char login[32];
	char name[256];
	char color[64];
	char flag[10];
        struct cis_info *cis = NULL;
        float perf;
        struct host_info *host;
        struct user_info *user;
        struct cis_hostinfo *hi;
        char *c;
	unsigned int uid = 0;
	int line = 0;
	int err;
        
	if (!configfile)
		return;
	
	errno = 0;

	while (getline (&buf, &buflen, configfile) != -1) {

		line++;
		c = buf;

		cmd = parse_cmd (cmds, NCMDS, &c);
		if (cmd == 0) continue;

		c += strspn (c, " \t");
		if (c[strlen(c)-1] == '\n')
			c[strlen(c)-1] = 0;

                switch (cmd) {
                        
		case CIS:
			
                        if (sscanf (c, "%s", name) != 1)
				fatal_err ("missing cis server name at line %d", line);

			if (find_cis (name))
				continue;

			cis = calloc (1, sizeof (struct cis_info));
			cis->name   = strdup (name);
			cis->handle = cisConnect(name);
			cis->next = cislist;
			cislist = cis;
			break;

		case HOST:

			uid = 0; *color = 0;
			err = sscanf (c, "%s %f %s %s %d", name, &perf, flag, color, &uid);

			if (err < 2)
				fatal_err ("wrong host command syntax. line: %s", buf);

			if (find_host_by_name (name))
				continue;

			host = add_host(name);

			host->name = strdup (name);
			if (!host->name)
				fatal_err("not enough memory");

			host->uid = uid;
			host->interval = SOCK_INTERVAL;
			if (*color)
				host->color = strdup (color);
			host->hostinfo->performance = perf;
			if (!strcasecmp(flag, "on"))
				host->flag = TRUE;
			break;

                case SPEEDUNITS:

                        c += strspn (c, " \t");
                        CPU_units = strdup (c);
                        break;

                case TIMESCALE:

                        sscanf (c, "%f", &time_scale);
                        record.jump.tv_sec  = (int) time_scale;
                        record.jump.tv_usec = (time_scale - (int) time_scale) * 1000000L;
                        break;

		case NETSCALE:

                        sscanf (c, "%f", &net_scale);
                        break;
                        
                case INTERVAL:

                        sscanf (c, "%f", &interval);
                        interval_tval.tv_sec  = (int) interval;
                        interval_tval.tv_usec = (interval - (int) interval) * 1000000;
                        break;

                case RECPATH:

                        record.path = strdup(c);
                        break;

                case RECDENSITY:

                        sscanf (c, "%d", &record.density);
                        break;

                case PRINTPATH:

			printpath = strdup(c);
                        break;

                default:
			break;
                }
        }
	rewind (configfile);
        line = 0;
	cis = NULL;
	host = NULL;
	while (getline (&buf, &buflen, configfile) != -1) {

		line++;
		c = buf;
		
		cmd = parse_cmd (cmds, NCMDS, &c);
		if (cmd == 0) continue;

                c += strspn (c, " \t");

                switch (cmd) {

		case USER:

			*login = *name = 0;

			err = sscanf (c, "%s %d \"%[^\"]\"", login, &uid, name);
			if (err != 3)
				fatal_err ("wrong user command syntax at line: %d", line);

			user = calloc (1, sizeof(struct user_info));
			if (!user)
				fatal_err ("cannot allocate memory for usr structures");

			user->name = strdup (login);
			user->id = uid;
			user->fullname = strdup (name);

			if (host) {
				user->next = host->users;
				host->users = user;
			} else if (cis) {
				user->next = cis->users;
				cis->users = user;
			} else {
				user->next = userlist;
				userlist = user;
			}
			break;

		case CIS:

			sscanf (c, "%s", name);
			cis = find_cis (name);
			host = NULL;
			
			break;

		case HOST:
			sscanf (c, "%s", name);

			host = find_host_by_name (name);

			host->cis = cis;

			if (!cis || !cis->handle || !(hi = cisHostInfo(cis->handle, name, perf_type))) {
				host->hostinfo->status = HOST_UNKNOWN;
				host->hostinfo->addr = host->hostaddr;
				break;
			}

			hi->performance = host->hostinfo->performance;
			*host->hostinfo = *hi;

			free (hi);

			if (cis) {
				cis->htab = realloc (cis->htab, (cis->nhosts+1)*sizeof(struct cis_hostinfo *));
				if (!cis->htab)
					fatal_err ("cannot allocate memory for host structures");

				cis->htab[cis->nhosts] = host->hostinfo;
				cis->nhosts++;
			}
			break;

		}
	}
	fclose (configfile);

	if (!hostlist)
                fatal_err("no host to monitor");
}
