/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_info: gets basic information about cluster
 */

#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <locale.h>

#include "cis.h"
#include "cis_clnt.h"

#include <regex.h>

char myhostname[MAXHOSTNAMELEN];
char buff[256];
struct cis_hostinfo *total;

struct cisinfo {
	char *host;
        CLIENT *handle;

	struct cis_info *info;

	int nhosts;
	struct cis_hostinfo *hlist;
	
	int *nsensors;
	struct cis_lmsinfo  **sensors;

} *cis = NULL;

struct column_info {
	char *name;
	int width;
} *columns = NULL;
int ncolumns;

void repeat(char c, int length)
{
	for (; length; length--)
		printf ("%c", c);
}

#define blank(len) repeat(' ', len)

int sort_fn(const void *left, const void *right)
{
	return (strcmp(((struct column_info *) left)->name, ((struct column_info *) right)->name));
}

int main (int argc,char *argv[])
{
	struct timeval current_time;
	struct column_info *col;
	int c,i,j,k,len,err;
	struct cis_lmsinfo *sensor;
	int maxnamelen = 5;
	char *expr = NULL;
	regex_t regex;
		
	
	setlocale(LC_ALL, "");
	setlocale(LC_NUMERIC, "C");

	if (gethostname((char *)& myhostname, MAXHOSTNAMELEN) < 0) {
		fprintf(stderr, "cannot get my hostname\n");
		exit(1);
	}

	cis = calloc(1, sizeof (struct cisinfo));
	if (!cis) {
		fprintf(stderr, "not enough memory\n");
		exit(1);
	}
        cis->host = getenv("CIS_SERVER");

	total = calloc(1, sizeof (struct cis_hostinfo));
	if (!total) {
		fprintf(stderr, "not enough memory\n");
		exit(1);
	}

	while ((c = getopt(argc, argv, "e:s:")) != EOF) {

		switch (c) {
		case 's':
			cis->host = strdup(optarg);  break;
		case 'e':
			expr = strdup(optarg);
			if (regcomp(&regex, expr, REG_ICASE|REG_NOSUB)) {
				fprintf(stderr, "wrong regular expression\n");
				exit(1);
			}
                        break;
		case ':':
                case '?':
			fprintf(stderr, "usage: cis_weather [-s cis_server] [-e regexp]\n");
			exit(1);
		}
        }

	if (!cis->host)
		cis->host = strdup(myhostname);

	cis->handle = cisConnect(cis->host);
	if (!cis->handle) {
		fprintf(stderr, "cannot connect to cis server %s\n", cis->host);
		exit(1);
	}

	cis->info = cisInfo(cis->handle);
	if (!cis->info) {
		fprintf(stderr, "cannot get information from cis server %s\n", cis->host);
		exit(1);
	}

	cis->nhosts = cisHostlist(cis->handle, &cis->hlist, 0, "", 1);
	if (cis->nhosts < 1) {
		fprintf(stderr, "no hosts\n");
		exit(1);
	}

	cis->nsensors = calloc(cis->nhosts, sizeof (int));
	cis->sensors  = calloc(cis->nhosts, sizeof (struct cis_lmsinfo *));
	if ( !cis->nsensors || !cis->sensors) {
		fprintf(stderr, "not enough memory\n");
		exit(1);
	}

	for (i = 0; i < cis->nhosts; i++) {
		len = strlen(cis->hlist[i].name);
		if (len > maxnamelen)
			maxnamelen = len;

		cis->nsensors[i] = cisLMsensors(cis->handle, &cis->sensors[i], cis->hlist[i].addr);
		if (cis->nsensors[i] < 0) {
			cis->nsensors[i] = 0;
			continue;
		}

		for (j = 0; j < cis->nsensors[i]; j++) {
			sensor = &cis->sensors[i][j];
			
			for (k = 0; k < ncolumns; k++)
				if (!strcmp(sensor->name, columns[k].name))
					break;
			if (k < ncolumns)
				continue;
			if (expr && (err = regexec(&regex, sensor->name, 0, NULL, REG_NOTBOL)))
				continue;

			columns = realloc(columns, (ncolumns+1) * sizeof(struct column_info));
			if (!columns) {
				fprintf(stderr, "not enough memory\n");
				exit(1);
			}
			col = &columns[ncolumns];
			col->name = strdup(sensor->name);
			if (!col->name) {
				fprintf(stderr, "not enough memory\n");
				exit(1);
			}
			col->width = strlen(col->name);
			len = sprintf(buff, "%.2f", (float) sensor->value / sensor->divider);
			if (len > col->width)
				col->width = len;

			ncolumns++;
		}
	}

	gettimeofday(&current_time, NULL);

	qsort(columns, ncolumns, sizeof(struct column_info), sort_fn);

	printf("\n");
	printf("LM sensors in cluster: %s\n", cis->info->name ? cis->info->name : "<noname>");
	printf("\n");

	printf("Name"); blank(maxnamelen-2);

	for (k = 0; k < ncolumns; k++) {
		len = columns[k].width - strlen(columns[k].name);
		blank(len);
		printf("%s  ", columns[k].name);
	}
	printf("\n");
	repeat('-', maxnamelen);
	for (k = 0; k < ncolumns; k++)
		repeat('-', columns[k].width+2);
	printf("\n");


	for (i = 0; i < cis->nhosts; i++) {
		printf ("%s", cis->hlist[i].name);
		blank (maxnamelen-strlen(cis->hlist[i].name)+2);
		for (k = 0; k < ncolumns; k++) {
			struct cis_lmsinfo *sensor;
			
			for (j = 0; j < cis->nsensors[i]; j++)
				if (!strcmp(cis->sensors[i][j].name, columns[k].name))
					break;
			sensor = &cis->sensors[i][j];
			len = sprintf(buff, "%.2f", (float) sensor->value / sensor->divider);
			blank(columns[k].width-len);
			printf("%s  ", buff);
		}
		printf ("\n");
	}
	exit(0);
}

