/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_rec: background (daemon) process for saving information from CIS server
 *          into record file. The file can be analyzed using xcis or cis_print.
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <syslog.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <locale.h>

#include "cis.h"
#include "cis_clnt.h"
#include "cis_rec.h"

#define DEF_INTERVAL                1.
#define SOCK_INTERVAL               5
#define DATELEN                     32

#define log(tag, fmt, args...) \
	if (daemon_mode) \
               syslog(tag, fmt , ## args); \
	else \
	       fprintf(stderr, fmt"\n" , ## args);

#define sectotval(A,B) { A.tv_sec  = (int) (B) ; \
        A.tv_usec = (int) (((B) - (int) (B)) * 1000000L); }

char myhostname[MAXHOSTNAMELEN];
char *logfilename = NULL;
int  logfile = 0;

float interval = DEF_INTERVAL;
struct timeval interval_tval = {(int) DEF_INTERVAL, (DEF_INTERVAL - (int) DEF_INTERVAL) * 1000000};

unsigned short myuid;
unsigned short uid;

void allarm_handler (int sig);
void hup_handler (int sig);
void daemonize (void);

int daemon_mode = FALSE;

struct {
        char               *filename;
        FILE               *file;
        char               *path;
} record = {NULL, NULL, NULL};

struct timeval current_time;

struct cisinfo {
	char *name;
        CLIENT *handle;
        struct cis_hostinfo **htab;
        int nhosts;
	struct cis_hostinfo *hlist;

} *cis = NULL;

struct host_info {
        float interval;
        
        struct cis_hostinfo *hostinfo, *old_hostinfo;
        struct timeval time;
        
        int nproc;
        struct cis_procinfo *proctab;
        
        int nsock;
        struct cis_sockinfo *socktab;

        int ndev;
        struct cis_netdevinfo *devtab;
        
        int saved_bytes;
	struct host_info *next;
} *hostlist = NULL;

int nhosts = 0;
struct cis_addr *addrtab = NULL;

static struct host_info *add_host (struct cis_hostinfo *hi)
{
	struct host_info *host, *h;

	srand(200);

	if (!(host = calloc (1, sizeof (struct host_info))))
		goto nomem;

	host->hostinfo = hi;

	host->old_hostinfo = calloc (1, sizeof (struct cis_hostinfo));
	if (!host->old_hostinfo)
		goto nomem;

	if (!hostlist)
		hostlist = host;
	else {
		for (h = hostlist; h->next; h = h->next)
			;
		h->next = host;
	}
	
	nhosts++;
	return host;

nomem:
	log(LOG_ERR, "cannot allocate memory for host structures");
	exit(1);
}

void getparams(int argc, char **argv)
{
	int c;
	struct host_info *host;
	struct cis_hostinfo *hi;
	int i;

	if (!myuid)
		uid = myuid;

	cis = calloc(1, sizeof (struct cisinfo));
	if (!cis) {
		log(LOG_ERR, "not enough memory");
		exit(1);
	}
        cis->name = getenv("CIS_SERVER");

	while ((c = getopt(argc, argv, "i:s:p:u:cd")) != EOF) {

		switch (c) {
		case 'i':
			interval = atof(optarg);
			if (interval == 0.) {
				log(LOG_ERR, "interval cannot be set to zero. reset to default %5.2f [s]", DEF_INTERVAL);
				interval = DEF_INTERVAL;
			}
			break;
		case 's':
			cis->name = strdup(optarg);
			break;
                case 'p':
			record.path = strdup(optarg);
			break;
                case 'c':
			record.file = stdout;
			break;
                case 'u':
			sscanf(optarg, "%hd", &uid);
			break;
                case 'd':
			daemon_mode = TRUE;
			break;
                case ':':
                case '?':
                        goto arg_error;
                }
        }

	if (!cis->name)
		cis->name = strdup(myhostname);
	
	cis->handle = cisConnect(cis->name);
	if (!cis->handle) {
		log(LOG_ERR, "cannot connect cis server %s", cis->name);
		exit(1);
	}

	cis->nhosts = cisHostlist(cis->handle, &cis->hlist, 0, "", interval);
	if (cis->nhosts < 1) {
		log(LOG_ERR, "no host to monitor");
		exit(1);
	}

	for (hi = cis->hlist; hi < cis->hlist + cis->nhosts; hi++) {
		host = add_host(hi);
		host->interval = SOCK_INTERVAL;
	}

	cis->htab = calloc(nhosts,sizeof(struct cis_hostinfo *));
	if (!cis->htab) {
		log(LOG_ERR, "not enough memory");
		exit(1);
	}

	for (host = hostlist, i = 0; i < nhosts; i++, host = host->next)
		cis->htab[i] = host->hostinfo;

	if (!record.path)
                record.path = ".";

	if (record.file == stdout)
		daemon_mode = FALSE;
	else if (!(record.filename = malloc(strlen(record.path)+strlen(cis->name)+DATELEN+6))) {
		log(LOG_ERR, "not enough memory");
		exit(1);
	}

	return;

arg_error:
	log(LOG_ERR, "usage: cis_rec [-c] [-d] [-s cis_server] [-i interval] [-p path] [-u uid]");
	exit(1);
}

/*
 * Function for comparing two timeval structures.
 * It returns an integer less than, equal to, or greater than zero if t1 is
 * found, respectively, to be less than, to match, or be greater than t2.
 */
int tvalcmp(struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

void tvaladd(struct timeval *t, struct timeval inc)
{
        t->tv_sec  += inc.tv_sec;
        t->tv_usec += inc.tv_usec;
        if (t->tv_usec > 1000000L) {
                t->tv_usec -= 1000000L;
                t->tv_sec++;
        }
}

void tvalsub(struct timeval *t, struct timeval dec)
{
        t->tv_sec -= dec.tv_sec;

        if (t->tv_usec < dec.tv_usec) {
                t->tv_sec--;
                t->tv_usec -= 1000000L - dec.tv_usec;
        } else
                t->tv_usec -= dec.tv_usec;
}

inline void monitor_error (struct host_info *host, int err)
{
        if (err > HOST_NOT_AVAILABLE) {
		cisDisconnect(cis->handle);
		cis->handle = NULL;
	}
}

void allarm_handler (int sig)
{
        struct host_info *h;
        struct cis_procinfo *ptab = NULL;
        struct cis_sockinfo *stab = NULL;
        struct cis_netdevinfo *dtab = NULL;
        struct cis_hostinfo *hi;
	int len;
        char *hchanges = NULL;
        char *pchanges = NULL;
        char *schanges = NULL;
        char *dchanges = NULL;
	int hchlen, pchlen, schlen, dchlen;
	int time_saved = FALSE;

	if (!cis->handle)
		cis->handle = cisConnect (cis->name);

	gettimeofday (&current_time, NULL);

	/*
         * Archive previous host information.
         */
	for (h = hostlist; h; h = h->next)
		*h->old_hostinfo = *h->hostinfo;

        /*
	 * Update dynamic host information.
	 */
	if (cis->handle)
		cisUpdateHostsinfo (cis->handle, CIS_ALL, cis->htab, cis->nhosts, interval);

	/*
         * Update static host information if host became available.
	 */
	for (h = hostlist; h; h = h->next) {
		if (h->hostinfo->status == h->old_hostinfo->status ||
		    h->hostinfo->status != HOST_AVAILABLE)
			continue;

		hi = cisHostinfo (cis->handle, h->hostinfo->addr, "", interval);
		if (hi) {
			hi->performance = h->hostinfo->performance;
			*h->hostinfo = *hi;
			free (hi);
		}
	}

        for (h = hostlist; h; h = h->next) {
		ptab = NULL;
                stab = NULL;
		dtab = NULL;
		
		hchlen = cisHostinfoPrepareChanges (h->old_hostinfo, h->hostinfo, &hchanges);

		h->time = current_time;

		/* Processes */
		len = cisProclist (cis->handle, &ptab, h->hostinfo->addr, uid, interval);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}

		pchlen = cisProclistPrepareChanges (h->proctab, h->nproc, ptab, len, &pchanges);
		if (h->nproc)
			free (h->proctab);
		h->proctab = ptab;
		h->nproc = len;
		
		/* Sockets */
		len = cisSocklist (cis->handle, &stab, h->hostinfo->addr, uid, h->interval);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}
		
		schlen = cisSocklistPrepareChanges (h->socktab, h->nsock, stab, len, &schanges);
		if (h->nsock)
			free (h->socktab);
		h->socktab = stab;
		h->nsock = len;

		/* Network devices */
		len = cisNdevlist (cis->handle, &dtab, h->hostinfo->addr, h->interval);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}
		dchlen = cisNdevlistPrepareChanges (h->devtab, h->ndev, dtab, len, &dchanges);
		if (h->ndev)
			free (h->devtab);
		h->devtab = dtab;
		h->ndev = len;

		if (!(hchlen || pchlen || schlen || dchlen))
			continue;

		if (!time_saved) {
			cisSaveTime (record.file, current_time);
			time_saved = TRUE;
		}

		cisSaveHostID(record.file, h->hostinfo);

		if (hchlen) {
			cisSaveHostinfoChanges (record.file, hchanges, hchlen);
			free (hchanges);
		}
		if (pchlen) {
			cisSaveProclistChanges (record.file, pchanges, pchlen);
			free (pchanges);
		}
		if (schlen) {
			cisSaveSocklistChanges (record.file, schanges, schlen);
			free (schanges);
		}
		if (dchlen) {
			cisSaveNdevlistChanges (record.file, dchanges, dchlen);
			free (dchanges);
		}
	}
	fflush(record.file);
}

void init_record_file (void)
{
	char buff[DATELEN];
	struct host_info *h;
	time_t ct = current_time.tv_sec;
	struct tm *tmp = localtime(&ct);

	if (record.file != stdout) {
		strftime(buff, DATELEN, "%Y.%m.%d_%H:%M:%S", tmp);
		sprintf(record.filename, "%s/%s_%s.rec", record.path, buff, cis->name);

		record.file = fopen(record.filename, "w+");
		if (!record.file) {
			log(LOG_ERR, "cannot create the file '%s'", record.filename);
			exit(1);
		}
	}

	if (cisSaveHeader(record.file, &current_time, nhosts, addrtab, interval) == -1) {
		log(LOG_ERR, "error while saving record file header");
		exit(1);
	}

	for (h = hostlist; h; h = h->next) {
		cisSaveHostID(record.file, h->hostinfo);
		cisSaveHostinfo (record.file, h->hostinfo);

		if (h->hostinfo->status != HOST_AVAILABLE || !cis->handle)
			continue;

		cisSaveProclist (record.file, h->proctab, h->nproc);
		cisSaveSocklist (record.file, h->socktab, h->nsock);
		cisSaveNdevlist (record.file, h->devtab,  h->ndev);
	}

	cisSaveBreak(record.file, 0);
	fflush(record.file);
}

void cleanup (int sig)
{
        fclose (record.file);
	if (daemon_mode)
		closelog();
        exit (0);
}


void hup_handler (int sig)
{
	if (!daemon_mode)
		cleanup(0);
	
	fclose (record.file);
	record.filename = NULL;
	init_record_file();
}

int main (int argc,char *argv[])
{
        struct sigaction allarm_action;
        struct sigaction hup_action;
	struct itimerval it;
        struct host_info *h;
        int i;
	static struct cis_procinfo *ptab = NULL;
        static struct cis_sockinfo *stab = NULL;
        static struct cis_netdevinfo *dtab = NULL;
        int len;
	
        myuid = getuid();

        setlocale(LC_ALL, "");
        setlocale(LC_NUMERIC, "C");

	if (!myuid)
		openlog("cis_rec", LOG_CONS, LOG_DAEMON);

        /*
         * Initialization of signal action structures.
         */
        allarm_action.sa_handler = allarm_handler;
	sigemptyset (&allarm_action.sa_mask);
	sigaddset (&allarm_action.sa_mask, SIGHUP);
        allarm_action.sa_flags   = 0;

	hup_action.sa_handler = hup_handler;
        sigemptyset (&hup_action.sa_mask);
	sigaddset (&hup_action.sa_mask, SIGALRM);
        hup_action.sa_flags   = 0;

        if (gethostname((char *)& myhostname, MAXHOSTNAMELEN) < 0) {
                log(LOG_ERR, "cannot get my hostname");
                exit(1);
	}

        getparams(argc, argv);

	if (!record.path)
		record.path = getcwd(NULL, 0);
	
	addrtab = malloc(nhosts*sizeof(struct cis_addr));
	for (i = 0, h = hostlist; i < nhosts; i++, h = h->next) {
		addrtab[i].server_addr = h->hostinfo->caddr;
		addrtab[i].node_addr   = h->hostinfo->addr;
	}
        /*
         * Let's install handler for SIGALRM.
         */
        sigaction(SIGALRM, &allarm_action, NULL);
        sigaction(SIGHUP, &hup_action, NULL);

        /*
         * We need to unregister rpc service before terminating ...
         */
        signal(SIGINT, cleanup);
        signal(SIGTERM, cleanup);
        
        /*
         * Set interval and reset timer.
         */
        sectotval(it.it_interval, interval);
        sectotval(it.it_value   , interval);

	if (!cis->handle)
		cis->handle = cisConnect (cis->name);

	gettimeofday (&current_time, NULL);

	if (cis->handle)
		cisUpdateHostsinfo (cis->handle, CIS_ALL, cis->htab, cis->nhosts, interval);

	for (h = hostlist; h; h = h->next) {

		if (h->hostinfo->status != HOST_AVAILABLE || !cis->handle)
			continue;

		len = cisProclist (cis->handle, &ptab, h->hostinfo->addr, uid, interval);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}
		h->proctab = ptab;
		h->nproc = len;

		len = cisSocklist (cis->handle, &stab, h->hostinfo->addr, uid, h->interval);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}
		h->socktab = stab;
		h->nsock = len;

		len = cisNdevlist (cis->handle, &dtab, h->hostinfo->addr, h->interval);
		if (len == -1) {
			monitor_error (h, len);
			len = 0;
		}
		h->devtab = dtab;
		h->ndev = len;
	}

	init_record_file ();

	if(daemon_mode)
		daemonize();

        setitimer(ITIMER_REAL, &it, NULL);

        while (1)
                select(0, NULL, NULL, NULL, NULL);

        exit(0);
}

