/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_print: prints binary record file in text form to stdout.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <netdb.h>
#include <locale.h>
#include <sys/time.h>
#include <sys/param.h> /* for MAXHOSTNAMELEN */

#include "cis.h"
#include "cis_rec.h"

char *names[] = {
        "",
        "",
        "",
        "sysinfo    ",
        "processes  ",
        "sockets    ",
        "interfaces ",
};

char *sock_types[] = {
        "",
        "  stream",
        "datagram",
};

char *disk_types[] = {
        "hd1",
        "hd2",
        "hd3",
        "hd4",
};

extern int h_errno;


struct host_info {
        struct cis_addr addr;

        struct cis_hostinfo *hostinfo;

        int nproc;
        struct cis_procinfo *proctab;
        
        int nsock;
        struct cis_sockinfo *socktab;

        int ndev;
        struct cis_netdevinfo *devtab;
        
} *hosttab = NULL;
int hosttab_length = 0;

char timebuff[32];
int dense = 0;
struct record_info r;
struct host_info *host;
FILE *file = NULL;

void getparams(int argc, char **argv)
{
        int c;
	
	while ((c = getopt(argc, argv, "d")) != EOF) {
		switch (c) {
		case 'd':
			dense = 1;
			break;
		case ':':
		case '?':
			goto arg_error;
		}
	}

	if (argv[optind] && !(file = fopen (argv[optind], "r"))) {
		printf ("cannot open file %s.\n", argv[optind]);
		exit (1);
	}
	return;

arg_error:
	printf ("usage: cis_print [-d] [record_file]\n");
	exit (1);
}

struct host_info *findhost (struct cis_addr addr)
{
        struct host_info *h;

	for (h = hosttab; h < hosttab+hosttab_length; h++)
		if (!memcmp(&h->addr, &addr, sizeof(struct cis_addr)))
			return h;
        return NULL;
}

int tprintf (char *fmt, ...)
{
	va_list ap;
	int len;

	if (dense)
		printf ("%s %s %s: ", timebuff, host->hostinfo->name, names[(int) r.type]);
	else
		printf ("          ");

        va_start(ap, fmt);
	len = vprintf (fmt, ap);
	va_end(ap);

	printf ("\n");

	return len;
}

#define SI_PRINT(el, fmt) tprintf (fmt, n->el)
#define SI_CHPRINT(el, fmt) if (!o || o->el != n->el) SI_PRINT (el, fmt)

void print_hostinfo (struct cis_hostinfo *o, struct cis_hostinfo *n)
{
	int i;
	
	if (n->status != HOST_AVAILABLE)
		return;

	if (!n) {
		tprintf ("not available");
                return;
        }
        if (o && !memcmp (o, n, sizeof (struct cis_hostinfo))) {
//                printf ( "        no change\n");
                return;
        }

	SI_CHPRINT (idle,     "Idle time              %ld");
	SI_CHPRINT (procnum,  "Number of processes    %d");
	
	if (!o || o->loads[0] != n->loads[0])
		tprintf("Average load in 1  min %-5.2f", n->loads[0]/100.);
        if (!o || o->loads[1] != n->loads[1])
		tprintf("Average load in 5  min %-5.2f", n->loads[1]/100.);
	if (!o || o->loads[2] != n->loads[2])
		tprintf("Average load in 15 min %-5.2f", n->loads[2]/100.);

	SI_CHPRINT (totalram, "Total memory           %ld");
        SI_CHPRINT (freeram,  "Free memory            %ld");
        SI_CHPRINT (sharedram,"Shared memory          %ld");
        SI_CHPRINT (bufferram,"Memory buffers         %ld");
        SI_CHPRINT (cachedram,"Cached memory          %ld");
        SI_CHPRINT (totalswap,"Total swap             %ld");
        SI_CHPRINT (freeswap, "Free swap              %ld");
        SI_CHPRINT (swpin,    "Swap in rate           %ld");
        SI_CHPRINT (swpout,   "Swap out rate          %ld");
        SI_CHPRINT (ctx_swtch,"Context switch rate    %ld");

	for (i = 0; i < 4; i++)
		if (!o || o->disk_read[i] != n->disk_read[i] || o->disk_write[i] != n->disk_write[i])
			tprintf("Disk %s read/write   %d/%d", disk_types[i], n->disk_read[i], n->disk_write[i]);

        if (!o || o->CPU_available != n->CPU_available)
		tprintf ( "CPU available [%%]      %4.2f ", n->CPU_available / 1e2);
}

void print_procinfo_full (struct cis_procinfo *tab, int len)
{
        struct cis_procinfo *p;

	for (p = tab; p < tab + len; p++)
                tprintf ("%-16s %5d %5d %5d %3ld %9ld %9ld %4.2f",
                        p->cmd, p->pid, p->ppid, p->uid, p->priority, p->rss, p->vm, p->pCPU / 1e2);
}

#define PI_PRINT(el, fmt) tprintf ("Changed    %5d %-16s "fmt, q->pid, q->cmd, q->el)
#define PI_CHPRINT(el, fmt) if (p->el != q->el) PI_PRINT (el, fmt)

void print_procinfo_changes (struct cis_procinfo *oldtab, int nold, struct cis_procinfo *newtab, int nnew)
{
        struct cis_procinfo *p;
        struct cis_procinfo *q;

        p = oldtab;
        q = newtab;
        
        while (p < oldtab+nold || q < newtab+nnew) {

                if (q < newtab+nnew && (p == oldtab+nold || p->pid > q->pid)) {
			tprintf ("New        %5d %-16s %5d %5d %3ld %9ld %9ld",
				 q->pid, q->cmd, q->ppid, q->uid, q->priority, q->rss, q->vm);
			q++;
                        continue;
                }
                if (p < oldtab+nold && (q == newtab+nnew || p->pid < q->pid)) {
			tprintf ("Terminated %5d %-16s %5d %5d",
                                p->pid, p->cmd, p->ppid, p->uid);
			p++;
			continue;
                }

                PI_CHPRINT(ppid,      "parent pid      %-5d");
                if (strncmp (p->cmd, q->cmd, CMD_LENGTH))
                        PI_PRINT(cmd, "command         %s");

                PI_CHPRINT(priority,  "priority        %-5ld");
                PI_CHPRINT(utime,     "user time       %-5ld");
                PI_CHPRINT(stime,     "system time     %-5ld");
                PI_CHPRINT(rss,       "resident memory %-9ld");
                PI_CHPRINT(vm,        "virtual memory  %-9ld");

                if (p->pCPU != q->pCPU)
			tprintf ("Changed    %5d %-16s CPU usage[%%]    %4.2f",
                                q->pid, q->cmd, q->pCPU / 1e2);

		PI_CHPRINT(majflt,    "major faults    %-5ld");
		PI_CHPRINT(disk_read, "disk read rate  %-5ld");
		PI_CHPRINT(disk_write,"disk write rate %-5ld");
                        
                p++;
                q++;
        }
}

void print_sockinfo_full (struct cis_sockinfo *tab, int len)
{
        struct cis_sockinfo *s;

	for (s = tab; s < tab + len; s++)
                tprintf ("%-8s %04X %08X:%04X %5d %5d %12ld %12ld",
                        sock_types[s->type], s->sport, s->daddr.s_addr, s->dport,
                        s->pid, s->uid, s->sent, s->rcvd);
}

int sock_cmp (const void *l, const void *r)
{
        struct cis_sockinfo *left  = (struct cis_sockinfo *) l;
        struct cis_sockinfo *right = (struct cis_sockinfo *) r;
        int res;
        
        if ((res = left->saddr.s_addr - right->saddr.s_addr))
                return (res);
        if ((res = left->type - right->type))
                return (res);
        if ((res = left->sport - right->sport))
                return (res);
        if ((res = left->daddr.s_addr - right->daddr.s_addr))
                return (res);
        if ((res = left->dport - right->dport))
                return (res);

        return (0);
}

void print_sockinfo_changes (struct cis_sockinfo *oldtab, int nold, struct cis_sockinfo *newtab, int nnew)
{
        struct cis_sockinfo *s;
        struct cis_sockinfo *t;

        s = oldtab;
        t = newtab;
        
        while (s < oldtab+nold || t < newtab+nnew) {

                if (t < newtab+nnew &&
                    (s == oldtab+nold || sock_cmp (s, t) > 0)) {
                        tprintf ("New     %-8s %04X %08X:%04X %5d %5d %12ld %12ld",
                                sock_types[t->type], t->sport, t->daddr.s_addr, t->dport,
                                t->pid, t->uid, t->sent, t->rcvd);

                        t++;
                        continue;
                }
                if (s < oldtab+nold &&
                    (t == newtab+nnew || sock_cmp (s, t) < 0)) {
                        tprintf ("Closed  %-8s %04X %08X:%04X %5d %5d",
				 sock_types[s->type], s->sport, s->daddr.s_addr, s->dport,
				 s->pid, s->uid);
                        s++;
                        continue;
                }

                if (s->sent != t->sent)
			tprintf ("Changed %-8s %04X %08X:%04X send rate      %-12ld",
				 sock_types[t->type], t->sport, t->daddr.s_addr, t->dport, t->sent);
                if (s->rcvd != t->rcvd)
			tprintf ("Changed %-8s %04X %08X:%04X receive rate   %-12ld",
				 sock_types[t->type], t->sport, t->daddr.s_addr, t->dport, t->rcvd);

                s++;
                t++;
        }
}

#define DI_PRINT(el, fmt) tprintf ("Changed     %-15s       "fmt, n->name, n->el)
#define DI_CHPRINT(el, fmt) if (o->el != n->el) DI_PRINT (el, fmt)

void print_devinfo_full (struct cis_netdevinfo *tab, int len)
{
        struct cis_netdevinfo *d;

	for (d = tab; d < tab + len; d++)
		tprintf ("%-15s %-4s rx:%10ld tx:%10ld coll:%10ld",
			 d->name,
			 d->status ? "up" : "down",
			 d->receive, d->transmit, d->collisions);
}

void print_devinfo_changes (struct cis_netdevinfo *oldtab, int nold, struct cis_netdevinfo *newtab, int nnew)
{
        struct cis_netdevinfo *o;
        struct cis_netdevinfo *n;

        o = oldtab;
        n = newtab;

        while (o < oldtab+nold || n < newtab+nnew) {

                if (n < newtab+nnew &&
                    (o == oldtab+nold || strcmp (o->name, n->name) > 0)) {
			tprintf ("New     %-15s %-4s rx:%10ld tx:%10ld coll:%10ld",
				 n->name,
				 n->status ? "up" : "down",
				 n->receive, n->transmit, n->collisions);

                        n++;
                        continue;
                }
                if (o < oldtab+nold &&
                    (n == newtab+nnew || strcmp (o->name, n->name) < 0)) {
			tprintf ("Closed  %-15s rx:%10ld tx:%10ld coll:%10ld",
				 o->name,
				 o->receive, o->transmit, o->collisions);
                        o++;
                        continue;
                }

                if (o->status != n->status)
			tprintf ("Changed     %-15s status          %-4s",
				 n->name, n->status ? "up" : "down");

		DI_CHPRINT(receive,   "receive rate    %-5ld");
		DI_CHPRINT(transmit,  "transmit rate   %-5ld");
		DI_CHPRINT(collisions,"collisions rate %-5ld");

                o++;
                n++;
        }
}

int main (int argc, char *argv[])
{
        struct cis_header *header;
	struct timeval t;
	struct tm *tmp;
        struct cis_hostinfo *hi;
        struct cis_procinfo *ptab;
        struct cis_sockinfo *stab;
	struct cis_netdevinfo *dtab;
        int len;
        int i;

        setlocale (LC_ALL, "");
        setlocale (LC_NUMERIC, "C");

	getparams(argc, argv);


	if (cisReadHeader (file?file:stdin, &header) < 0 || header->version != CIS_RECORD_VERSION) {
                printf ("incorrect record file\n");
                exit (1);
	}
	
	hosttab = calloc (header->nhosts, sizeof (struct host_info));
	if (!hosttab) {
		printf ("not enough memory\n");
		exit (1);
	}
	
	hosttab_length = header->nhosts;
	for (i = 0; i < hosttab_length; i++) {
		host = &hosttab[i];
		host->addr = header->hosts[i];
		host->hostinfo = calloc (1, sizeof (struct cis_hostinfo));
	}

	tmp = localtime (&header->time.tv_sec);
	strftime (timebuff, 32, "%Y.%m.%d %H:%M:%S", tmp);
	
	while (cisReadRecord (file?file:stdin, &r) > -1 && r.type != CIS_REC_BREAK) {
		if (!dense && r.type != CIS_REC_HOST && r.type != CIS_REC_SYSINFO)
			printf ("\n%s %s %s \n\n", timebuff, host->hostinfo->name, names[(int) r.type]);
		switch (r.type) {

		case CIS_REC_HOST:
			host = &hosttab[(int) *(char*) r.data];
			free (r.data);
			break;

		case CIS_REC_SYSINFO:
			free (host->hostinfo);
			host->hostinfo = (struct cis_hostinfo *) r.data;
			if (!dense && r.type != CIS_REC_HOST)
				printf ("\n%s %s %s \n\n", timebuff, host->hostinfo->name, names[(int) r.type]);
			print_hostinfo (NULL, host->hostinfo);
			break;

		case CIS_REC_PROCESSES:
			free (host->proctab);
                        host->proctab = (struct cis_procinfo *) r.data;
                        host->nproc = r.length / sizeof (struct cis_procinfo);
                        print_procinfo_full (host->proctab, host->nproc);
                        break;

		case CIS_REC_SOCKETS:
                        free (host->socktab);
                        host->socktab = (struct cis_sockinfo *) r.data;
                        host->nsock = r.length / sizeof (struct cis_sockinfo);
                        print_sockinfo_full (host->socktab, host->nsock);
                        break;

		case CIS_REC_IFACES:
                        free (host->devtab);
                        host->devtab = (struct cis_netdevinfo *) r.data;
                        host->ndev = r.length / sizeof (struct cis_netdevinfo);
                        print_devinfo_full (host->devtab, host->ndev);
			break;

		}
	}
	while (cisReadRecord (file?file:stdin, &r) > -1) {
		if (!dense && r.type != CIS_REC_HOST && r.type != CIS_REC_TIME)
			printf ("\n%s %s %s \n\n", timebuff, host->hostinfo->name, names[(int) r.type]);
		switch (r.type) {

		case CIS_REC_TIME:
			i = *(int *) r.data;
			t.tv_sec  = header->time.tv_sec + i/100;
			t.tv_usec = header->time.tv_usec + (i%100)*10000;
			if (t.tv_usec >= 1000000L) {
				t.tv_sec  += t.tv_usec / 1000000L;
				t.tv_usec %= 1000000L;
			}
			tmp = localtime (&t.tv_sec);
			strftime (timebuff, 32, "%Y.%m.%d %H:%M:%S", tmp);
			free (r.data);
			break;

		case CIS_REC_HOST:
			host = &hosttab[(int) *(char*) r.data];
			free (r.data);
			break;

		case CIS_REC_SYSINFO:
                        if (!r.length) break;
			cisHostinfoApplyChanges (host->hostinfo, r.data, r.length, &hi);
			print_hostinfo (host->hostinfo, hi);
                        free (host->hostinfo);
                        host->hostinfo = hi;
                        free (r.data);
                        break;

                        
		case CIS_REC_PROCESSES:
			if (!r.length) break;
			len = cisProclistApplyChanges (host->proctab, host->nproc, r.data, r.length, &ptab);
                        print_procinfo_changes (host->proctab, host->nproc, ptab, len);
                        free (host->proctab);
                        host->proctab = ptab;
                        host->nproc   = len;
                        if (r.length)
                                free (r.data);
                        break;


		case CIS_REC_SOCKETS:
                        if (!r.length) break;
			len = cisSocklistApplyChanges (host->socktab, host->nsock, r.data, r.length, &stab);
                        print_sockinfo_changes (host->socktab, host->nsock, stab, len);
                        free (host->socktab);
                        host->socktab = stab;
                        host->nsock   = len;
                        if (r.length)
                                free (r.data);
                        break;


                case CIS_REC_IFACES:
                        if (!r.length) break;
			len = cisNdevlistApplyChanges (host->devtab, host->ndev, r.data, r.length, &dtab);
                        print_devinfo_changes (host->devtab, host->ndev, dtab, len);
                        free (host->devtab);
                        host->devtab = dtab;
                        host->ndev   = len;
                        if (r.length)
                                free (r.data);
                        break;
                }
                fflush (stdout);
	}

        exit (0);
}