/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_info: gets basic information about cluster
 */

#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <locale.h>

#include "cis.h"
#include "cis_clnt.h"

#define DEF_INTERVAL                1.

int (*sort_function)(const void *l,const void *r);

char myhostname[MAXHOSTNAMELEN];
float interval = DEF_INTERVAL;
char buff[256];
struct cis_hostinfo *total;

struct cisinfo {
	char *host;
        CLIENT *handle;

	struct cis_info *info;

	int nhosts;
	struct cis_hostinfo *hlist;

} *cis = NULL;

enum {
	NAME,
	PERF_MAX,
	PERF_AVAIL,
	MEM_MAX,
	MEM_AVAIL,
	SWAP_MAX,
	SWAP_AVAIL,
        SWAPRATE,
	DISKRATE,
	PROC,
	CTX,
	LOAD1,
	LOAD5,
	LOAD15,
	RELIAB,
};

struct column_info {
	int num;
	int indent;
	int print;
	int width;
} columns[] = {
	{NAME,      -1, 1},
	{PERF_MAX,   1, 1},
	{PERF_AVAIL, 1, 1},
	{MEM_MAX,    1, 1},
	{MEM_AVAIL,  1, 1},
	{SWAP_MAX,   1, 1},
	{SWAP_AVAIL, 1, 1},
	{SWAPRATE,   0, 0},
	{DISKRATE,   0, 0},
	{PROC,       1, 1},
	{CTX,        1, 0},
	{LOAD1,      1, 1},
	{LOAD5,      1, 1},
	{LOAD15,     1, 1},
	{RELIAB,     0, 0},
};

struct header_info {
	char *text;
	int pos;
	int ncols;
} header1[] = {
	{"Performance",  PERF_MAX, 2},
	{"RAM [KB]",     MEM_MAX,  2},
	{"Swap [KB]",    SWAP_MAX, 2},
	{"Swap",         SWAPRATE, 1},
	{"Disk",         DISKRATE, 1},
	{"Average load", LOAD1,    3},
}, header2[] = {
	{"Name",        NAME,        1},
	{"max",         PERF_MAX,    1},
	{"avail",       PERF_AVAIL,  1},
	{"max",         MEM_MAX,     1},
	{"avail",       MEM_AVAIL,   1},
	{"max",         SWAP_MAX,    1},
	{"avail",       SWAP_AVAIL,  1},
	{"in/out",      SWAPRATE,    1},
	{" r/w ",       DISKRATE,    1},
	{"Proc",        PROC,        1},
	{"Ctx",         CTX,         1},
	{"1min",        LOAD1,       1},
	{"5min",        LOAD5,       1},
	{"15min",       LOAD15,      1},
	{"Reliability", RELIAB,      1},
};
	

#define for_each_column(col) for (col = columns; (char*) col < (char*)&columns+sizeof(columns); col++)

#define for_each_host(hi) for (hi = cis->hlist; hi < cis->hlist+cis->nhosts; hi++)

#define getwidth(type, fmt, args...) \
	if (columns[type].print) \
                for_each_host(hi) { \
	                len = sprintf(buff, fmt, ## args); \
	                if (len > columns[type].width) \
	                         columns[type].width = len; \
	        }

#define for_each_header(h, tab) for (h = tab; (char*) h < (char*)&tab+sizeof(tab); h++)

void repeat(char c, int length)
{
	for (; length; length--)
		printf ("%c", c);
}

#define blank(len) repeat(' ', len)

void hprint(struct header_info *h)
{
	int i, w, len;

	if (!columns[h->pos].print)
		return;
	
	for (i = h->pos, w = -1; i < h->pos+h->ncols; i++)
		w += columns[i].width+1;

	len = w - strlen(h->text);

	if (columns[h->pos].indent != -1)
		blank(len - len/2);

	printf("%s", h->text);
	if (columns[h->pos].indent != -1)
		blank(len/2 + 1);
	else
		blank(len+1);
}

void cprint(int type, char *fmt, ...)
{
	va_list ap;
	int len;

	if (!columns[type].print)
		return;

	va_start(ap, fmt);

	len = vsprintf(buff, fmt, ap);
	len = columns[type].width - len;

	switch (columns[type].indent) {
	case -1:
		printf("%s", buff);
		blank(len+1);
		break;
	case 1:
		blank(len);
		printf("%s", buff);
		blank(1);
		break;
	case 0:
		blank(len - len / 2);
		printf("%s", buff);
		blank(len/2 + 1);
	}
}

int sort_name(const void *left, const void *right)
{
	struct cis_hostinfo *l = (struct cis_hostinfo *) left;
        struct cis_hostinfo *r = (struct cis_hostinfo *) right;

	return (strcmp(l->name, r->name));
}

int sort_perf(const void *left, const void *right)
{
	struct cis_hostinfo *l = (struct cis_hostinfo *) left;
        struct cis_hostinfo *r = (struct cis_hostinfo *) right;

	return ((int) (r->CPU_available/1e2*r->performance -
		       l->CPU_available/1e2*l->performance));
}

int sort_mem(const void *left, const void *right)
{
	struct cis_hostinfo *l = (struct cis_hostinfo *) left;
        struct cis_hostinfo *r = (struct cis_hostinfo *) right;

	return ((int) (r->freeram+r->bufferram+r->cachedram -
		       (l->freeram+l->bufferram+l->cachedram)));
}

#define sum(type) hi->type[0]+hi->type[1]+hi->type[2]+hi->type[3]

int main (int argc,char *argv[])
{
	struct cis_hostinfo *hi;
	struct timeval current_time;
	int c, pos, len;
	struct column_info *col;
	struct header_info *h;
	float perf_av = 0.;
	
	setlocale(LC_ALL, "");
	setlocale(LC_NUMERIC, "C");

	if (gethostname((char *)& myhostname, MAXHOSTNAMELEN) < 0) {
		fprintf(stderr, "cannot get my hostname\n");
		exit(1);
	}

	cis = calloc(1, sizeof (struct cisinfo));
	if (!cis) {
		fprintf(stderr, "not enough memory\n");
		exit(1);
	}
        cis->host = getenv("CIS_SERVER");

	total = calloc(1, sizeof (struct cis_hostinfo));
	if (!total) {
		fprintf(stderr, "not enough memory\n");
		exit(1);
	}

	sort_function = sort_name;
	
	while ((c = getopt(argc, argv, "i:s:cdrwPM")) != EOF) {

		switch (c) {
		case 'i':
			interval = atof(optarg);
			if (interval == 0.) {
				fprintf(stderr, "interval cannot be set to zero. reset to default %5.2f [s]\n", DEF_INTERVAL);
				interval = DEF_INTERVAL;
			}
			break;
		case 's':
			cis->host = strdup(optarg);  break;
		case 'r':
			columns[RELIAB  ].print = 1; break;
		case 'w':
			columns[SWAPRATE].print = 1; break;
		case 'c':
			columns[CTX     ].print = 1; break;
		case 'd':
			columns[DISKRATE].print = 1; break;
		case 'P':
			sort_function = sort_perf; break;
		case 'M':
			sort_function = sort_mem; break;
		case ':':
                case '?':
			fprintf(stderr, "usage: cis_info [-s cis_server] [-cdrwPM]\n");
			exit(1);
		}
        }

	if (!cis->host)
		cis->host = strdup(myhostname);

	cis->handle = cisConnect(cis->host);
	if (!cis->handle) {
		fprintf(stderr, "cannot connect to cis server %s\n", cis->host);
		exit(1);
	}

	cis->info = cisInfo(cis->handle);
	if (!cis->info) {
		fprintf(stderr, "cannot get information from cis server %s\n", cis->host);
		exit(1);
	}

	cis->nhosts = cisHostlist(cis->handle, &cis->hlist, 0, "", interval);
	if (cis->nhosts < 1) {
		fprintf(stderr, "no hosts\n");
		exit(1);
	}

	gettimeofday(&current_time, NULL);

	for_each_column(col)
		col->width = strlen(header2[col-columns].text);

	for_each_host(hi) {
		perf_av += hi->CPU_available/1e4*hi->performance;
		total->performance += hi->performance;
		total->totalram    += hi->totalram;
		total->freeram     += hi->freeram+hi->bufferram+hi->cachedram;
		total->totalswap   += hi->totalswap;
		total->freeswap    += hi->freeswap;
	}
	total->CPU_available = perf_av/total->performance*1e4;
	
	getwidth (NAME      , "%s",   hi->name);
	getwidth (PERF_MAX  , "%.2f", hi->performance);
	getwidth (PERF_AVAIL, "%.2f", hi->CPU_available/1e4*hi->performance);
	getwidth (MEM_MAX   , "%ld",  hi->totalram);
	getwidth (MEM_AVAIL , "%ld",  hi->freeram+hi->bufferram+hi->cachedram);
	getwidth (SWAP_MAX  , "%ld",  hi->totalswap);
	getwidth (SWAP_AVAIL, "%ld",  hi->freeswap);
	getwidth (SWAPRATE  , "%ld/%ld", hi->swpin, hi->swpout);
	getwidth (DISKRATE  , "%ld/%ld", sum(disk_read), sum(disk_write));
	getwidth (PROC      , "%d",   hi->procnum);
	getwidth (CTX       , "%ld",  hi->ctx_swtch);
	getwidth (LOAD1     , "%.2f", hi->loads[0]/100.);
	getwidth (LOAD5     , "%.2f", hi->loads[1]/100.);
	getwidth (LOAD15    , "%.2f", hi->loads[2]/100.);
	getwidth (RELIAB    , "%s",   rel_types[hi->reliability]);

	getwidth (PERF_MAX  , "%.2f", total->performance);
	getwidth (PERF_AVAIL, "%.2f", total->CPU_available/1e4*total->performance);
	getwidth (MEM_MAX   , "%ld",  total->totalram);
	getwidth (MEM_AVAIL , "%ld",  total->freeram);
	getwidth (SWAP_MAX  , "%ld",  total->totalswap);
	getwidth (SWAP_AVAIL, "%ld",  total->freeswap);

	printf("\n");
	printf("Cluster: %s\n", cis->info->name ? cis->info->name : "<noname>");
	printf("\n");

	pos = 0;
	for_each_header(h, header1) {
		for (col = columns+pos; col < columns+h->pos; col++)
			if (col->print)
				blank(col->width+1);
		hprint(h);
		pos = h->pos+h->ncols;
	}
	printf("\n");

	pos = 0;
	for_each_header(h, header2) {
		hprint(h);
		pos = h->pos+h->ncols;
	}
	printf("\n");

	for_each_column(col)
		if (col->print)
			repeat('-', col->width+1);
	printf("\n");

	qsort(cis->hlist, cis->nhosts, sizeof(struct cis_hostinfo), sort_function);
	
	for_each_host(hi) {
		cprint (NAME      , "%s",      hi->name);
		cprint (PERF_MAX  , "%.2f",    hi->performance);
		cprint (PERF_AVAIL, "%.2f",    hi->CPU_available/1e4*hi->performance);
		cprint (MEM_MAX   , "%ld",     hi->totalram);
		cprint (MEM_AVAIL , "%ld",     hi->freeram+hi->bufferram+hi->cachedram);
		cprint (SWAP_MAX  , "%ld",     hi->totalswap);
		cprint (SWAP_AVAIL, "%ld",     hi->freeswap);
		cprint (SWAPRATE  , "%ld/%ld", hi->swpin, hi->swpout);
		cprint (DISKRATE  , "%ld/%ld", sum(disk_read), sum(disk_write));
		cprint (PROC      , "%d",      hi->procnum);
		cprint (CTX       , "%ld",     hi->ctx_swtch);
		cprint (LOAD1     , "%.2f",    hi->loads[0]/100.);
		cprint (LOAD5     , "%.2f",    hi->loads[1]/100.);
		cprint (LOAD15    , "%.2f",    hi->loads[2]/100.);
		cprint (RELIAB    , "%s",      rel_types[hi->reliability]);
		printf("\n");

	}
	for_each_column(col)
		if (col->print)
			repeat('-', col->width+1);
	printf("\n");
	cprint (NAME      , "%s"  ,"Total");
	cprint (PERF_MAX  , "%.2f", total->performance);
	cprint (PERF_AVAIL, "%.2f", total->CPU_available/1e4*total->performance);
	cprint (MEM_MAX   , "%ld",  total->totalram);
	cprint (MEM_AVAIL , "%ld",  total->freeram);
	cprint (SWAP_MAX  , "%ld",  total->totalswap);
	cprint (SWAP_AVAIL, "%ld",  total->freeswap);
	printf("\n");

	exit(0);
}

