/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * System monitoring functions.
 */

#include "cis.h"
#include "mon_common.h"

static FILE *sysinfofile;
static FILE *loadavg;
static FILE *meminfo;
static FILE *uptime;
static FILE *statfile;

static int mode = 0;
static struct utsname my_uname;
static struct sysinfo sinfo;
static int    defprio;

static void sys_init (void)
{
	struct stat st;

	if (uname(&my_uname) == -1) {
		syslog(LOG_ERR, "cannot get system information");
		exit(1);
	}

	if (monitor_type == MONITOR || monitor_type == SYSMON) {
		sysinfo(&sinfo);
		sinfo.totalram  >>= 10;
		sinfo.totalswap >>= 10;
		defprio        = (20*ct_per_sec/100);
	}

	/*
	 * Initialize input.
	 */
	if (stat("/proc/cis_sysinfo", &st) == -1)
		mode |= PROC_MODE;
	
	if (mode&PROC_MODE) {
		loadavg     = proc_open("loadavg");
		uptime      = proc_open("uptime");
		meminfo     = proc_open("meminfo");
		statfile    = proc_open("stat");
	} else {
		sysinfofile = proc_open("cis_sysinfo");
	}
}

static void sys_send_init (void)
{
	pack_init(SYSMON_INIT);
	put_object(my_uname);
	put_object(defprio);
	put_object(sinfo.totalram);
	put_object(sinfo.totalswap);
	send_msg();
	recv_interval();
}

static void sys_send_exit (void)
{
	return (send_exit(SYSMON_EXIT));
}

#define add_change(el) \
	if (hi->el != oldhi->el) {                 \
	*(chptr++) = OFFSET(cis_hostinfo, el);     \
	*(int *) chptr = hi->el - oldhi->el;       \
	chptr += sizeof(int);                      \
	oldhi->el = hi->el;                        \
	}

static inline void put_changes(struct cis_hostinfo *hi, struct cis_hostinfo *oldhi)
{
	unsigned char chbuff[sizeof(struct cis_hostinfo)];
	unsigned char *chptr = chbuff;
	int i;

	add_change(procnum);
	add_change(loads[0]);
	add_change(loads[1]);
	add_change(loads[2]);
	add_change(totalswap);
	add_change(freeswap);
	add_change(freeram);
	add_change(sharedram);
	add_change(bufferram);
	add_change(cachedram);

	add_change(idle);
	add_change(ctx_swtch);
	add_change(swpin);
	add_change(swpout);

	for (i = 0; i < 4; i++)
		add_change(disk_read[i]);
	for (i = 0; i < 4; i++)
		add_change(disk_write[i]);

	if (chptr == chbuff)
		return;
	
	*(chptr++) = 0xff;
	
	put(chbuff, chptr - chbuff);
}

static void sys_check(void)
{
	static char *line;
	static int linelen;

	static struct cis_hostinfo hi, oldhi;
	float loads[3];

	/*
	 * Get the host system information.
	 */
	if (!(mode&PROC_MODE)) {
		sysinfofile = freopen("/proc/cis_sysinfo","r", sysinfofile);
		reset_time();
		rewind(sysinfofile);
		fread(&hi, HOSTINFO_LEN, 1, sysinfofile);
	} else {
		float idle;
		
		loadavg = freopen("/proc/loadavg", "r", loadavg);
		fscanf(loadavg, "%f %f %f %*d/%hd",
		       &loads[0], &loads[1], &loads[2], &hi.procnum);
		hi.loads[0] = loads[0]*100;
		hi.loads[1] = loads[1]*100;
		hi.loads[2] = loads[2]*100;
		
		uptime = freopen("/proc/uptime", "r", uptime);
		fscanf(uptime, "%*f %f", &idle);
		hi.idle = idle * ct_per_sec;

		statfile = freopen("/proc/stat", "r", uptime);
		do {
			getline(&line, &linelen, statfile);
		} while (strncmp (line, "disk_rblk", 9));
		sscanf(line, "%*s %ld %ld %ld %ld",
		       &hi.disk_read[0], &hi.disk_read[1], &hi.disk_read[2], &hi.disk_read[3]);

		getline(&line, &linelen, statfile);
		sscanf(line, "%*s %ld %ld %ld %ld",
		       &hi.disk_write[0], &hi.disk_write[1], &hi.disk_write[2], &hi.disk_write[3]);

		do {
			getline(&line, &linelen, statfile);
		} while (strncmp (line, "swap", 4));
		sscanf(line, "%*s %ld %ld", &hi.swpin, &hi.swpout);

		do {
			getline(&line, &linelen, statfile);
		} while (strncmp (line, "ctxt", 4));
		sscanf(line, "%*s %ld", &hi.ctx_swtch);

		meminfo = freopen("/proc/meminfo", "r", meminfo);
		getline(&line, &linelen, meminfo); /* Field names */
		getline(&line, &linelen, meminfo); /* Memory */
		sscanf(line, "%*s %*d %*d %ld %ld %ld %ld",
		       &hi.freeram, &hi.sharedram, &hi.bufferram, &hi.cachedram);
		
		getline(&line, &linelen, meminfo); /* Swap */
		sscanf(line, "%*s %ld %*d %ld",
		       &hi.totalswap, &hi.freeswap);
		
		hi.freeram   >>= 10;
		hi.sharedram >>= 10;
		hi.bufferram >>= 10;
		hi.cachedram >>= 10;
		hi.totalswap >>= 10;
		hi.freeswap  >>= 10;
	}

	if (full_msg) {
		init_msg(SYSMON_FULL);
		put(&hi, HOSTINFO_LEN);
	} else {
		init_msg(SYSMON_CHANGES);
		put_changes(&hi, &oldhi);
	}

	if (full_msg)
		send_msg();

        oldhi = hi;
}

struct monitor_desc sysmon = {
	SYSMON,
	-1,
	sys_init,
	sys_check,
	0,
	sys_send_init,
	sys_send_exit,
};
