/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Socket monitoring functions.
 */

#include "cis.h"
#include "cis_mon.h"
#include "mon_common.h"

#define HT_LENGTH(a)  (a * 3 / 2)
#define hash(a)    (((a * 3) / 2) % ht_length)
#define isroot()   (!myuid)

#ifndef DEF_SOCKTAB_LENGTH
#define DEF_SOCKTAB_LENGTH        10
#endif

struct socket_info {
	struct socket_info *prev, *next;
	unsigned char flags;
	struct cis_sockinfo data;
};

/* Socket information */
static FILE               *sockfile;
static struct socket_info *socklist;
static int                nsock;
static int                ht_capacity = DEF_SOCKTAB_LENGTH;
static int                ht_length   = HT_LENGTH (DEF_SOCKTAB_LENGTH);
static struct socket_info **hash_table;

static int kernel_patched = 0;

static void sock_init (void)
{
	struct sockaddr_nl nladdr={AF_NETLINK,0,0,0xffffffff};
	kernel_patched = TRUE;

	if (!full_mode) {
		hash_table = malloc(ht_length * sizeof (struct socket_info *));
		if (hash_table == NULL) {
			syslog(LOG_ERR, "cannot allocate memory for the hash table.");
			exit(1);
		}

		memset(hash_table, 0, ht_length * sizeof (struct socket_info *));
	}

	socklist = NULL; nsock = 0;

	sockfile = fopen("/proc/cis_socklist","r");
        if (!sockfile) {
                syslog(LOG_ERR, "cannot open /proc/cis_socklist (kernel not patched)");
		goto err;
	}
	
	sockmon.fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_SOCKMON);
        if (sockmon.fd < 0) {
		syslog(LOG_ERR, "cannot create netlink socket (kernel not patched)");
		goto err;
        }

        nladdr.nl_pid = mypid;
        if (bind (sockmon.fd, (struct sockaddr *) &nladdr, sizeof (nladdr)) < 0) {
                syslog(LOG_ERR, "cannot bind netlink socket");
		goto err;
        }

	return;

err:
	kernel_patched = FALSE;
}

static void sock_send_init (void)
{
	pack_init(SOCKMON_INIT);
	send_msg();
	recv_interval();
}

static void sock_send_exit (void)
{
	return (send_exit(SOCKMON_EXIT));
}


/*
 * Searching in the hash table. If the requested socket was not found, a
 * pointer to the next free cell is returned.
 */
static inline struct socket_info **ht_find(struct cis_sockinfo *socket)
{
	static struct socket_info **hs;
        
	hs = &hash_table[hash(socket->sport)];

        while (*hs && memcmp(&(*hs)->data, socket, SOCKINFO_HDRLEN))
                if (++hs == hash_table + ht_length)
                        hs = hash_table;

        return (hs);
}

static inline void ht_remove(struct socket_info **hole)
{
	struct socket_info **hs = hole;

        *(hole) = NULL;

        for(;;) {
                if (++hs == hash_table + ht_length)
                        hs = hash_table;

                if (!(*hs)) break;

		if (!(*ht_find(&(*hs)->data)))
                        *hole = *hs, *hs = NULL, hole = hs;
        }
}

static struct socket_info *register_socket(struct cis_sockinfo *socket, struct socket_info **hs)
{
	struct socket_info *s, *tmp;

	s = malloc(sizeof (struct socket_info));
        if (!s)
                return (NULL);

	memcpy(&s->data, socket, SOCKINFO_LEN);

	if (++nsock > ht_capacity) {
		struct socket_info **new_ht;

		/*
                 * Expand the hash_table.
                 */
                ht_capacity = ht_capacity * 3 / 2;
                ht_length   = HT_LENGTH(ht_capacity);

		new_ht = calloc(ht_length, sizeof (struct socket_info *));
		if (!new_ht) {
			free(s);
			syslog(LOG_WARNING, "not enough memory for hash table");
			return (NULL);
		}

                free(hash_table);
                hash_table = new_ht;

                for (tmp = socklist; tmp; tmp = tmp->next)
                        *(ht_find(&tmp->data)) = tmp;

		*(ht_find(&s->data)) = s;

	} else
		*hs = s;

	s->flags = CONSISTENT;
        
	list_insert(socklist, s);
        return (s);
}

#define add_change(el) \
        if (s->el != msg->el) {                           \
               *(chptr++) = OFFSET(cis_sockinfo, el);     \
               *(int *) chptr = msg->el - s->el;          \
               chptr += sizeof(int);                      \
               s->el = msg->el;                           \
        }

#define add_element(el) \
        {                                                 \
               *(chptr++) = OFFSET(cis_sockinfo, el);     \
               memcpy(chptr,  &msg->el, sizeof(msg->el)); \
               memcpy(&s->el, &msg->el, sizeof(msg->el)); \
               chptr += sizeof(msg->el);                  \
        }

static void put_changes(struct cis_sockinfo *s, struct cis_sockinfo *msg)
{
        char chbuff[sizeof(struct cis_sockinfo)+5];
        char *chptr = chbuff;

        memcpy (chbuff, msg, SOCKINFO_HDRLEN);
        chptr += SOCKINFO_HDRLEN;

	if (s->pid      != msg->pid)       add_element(pid);
	if (s->uid      != msg->uid)       add_element(uid);
        if (s->sent     != msg->sent)      add_change(sent);
        if (s->rcvd     != msg->rcvd)      add_change(rcvd);
        
        if (chptr == chbuff+SOCKINFO_HDRLEN)
                return;

        *(chptr++) = 0;
            
//        if ((bufflen + (chptr - chbuff)) > MON_BUFFLEN)
//                buffer_send();
        put(chbuff, chptr - chbuff);
}

static void put_socket(char tag, struct cis_sockinfo *s)
{
        if ((bufflen + SOCKINFO_LEN + 1) > MON_BUFFLEN)
                buffer_send();

	put (s, SOCKINFO_HDRLEN);
        *(buffptr++) = tag;

        put (((char *) s)+SOCKINFO_HDRLEN, SOCKINFO_LEN-SOCKINFO_HDRLEN);
}

#define put_new(s)        put_socket(1, s)
#define put_closed(s)     put_socket(2, s)

/*
 * Read information about sockets and prepare message for server.
 */
static void sock_read(void)
{
	static struct cis_sockinfo sock;
	struct socket_info *s, **hs;

	if (!full_mode)
		for (s = socklist; s; s = s->next)
			s->flags &= ~CONSISTENT;

	rewind(sockfile);
	while (fread((char *) &sock, sizeof(struct cis_sockinfo), 1, sockfile)) {

                /*
                 * Test addresses and change 127.0.0.1 and 0.0.0.0
                 * to local address.
                 */
//                if (!sock.saddr.s_addr || sock.saddr.s_addr == 0x0100007f)
//                        sock.saddr = addrlist[0];

//                if (!sock.daddr.s_addr || sock.daddr.s_addr == 0x0100007f)
//                        sock.daddr = addrlist[0];
                
                hs = ht_find(&sock);

		if (!(*hs)) {  /* New socket */
			if (!(s = register_socket(&sock, hs)))
                                continue;

			if (!full_msg)
				put_new(&s->data);
                } else {
                        /*
                         * Socket was found. Log the changes (if any).
                         */
                        s = *hs;

                        if (memcmp(&s->data, &sock, SOCKINFO_LEN)) {
                                if (!full_msg)
                                        put_changes(&s->data, &sock);
				else
					memcpy(&s->data, &sock, SOCKINFO_LEN);
			}
		}
		s->flags |= CONSISTENT;
		if (full_msg)
			put(&s->data, SOCKINFO_LEN);
	}
}
        
static void clear_closed(void)
{
	struct socket_info *s, *clsock, **hs;

        s = socklist;

        while(s) {
                if (!s->flags&CONSISTENT) {
                        clsock = s;
                        s = s->next;

			hs = ht_find(&clsock->data);
                        if (!full_msg)
				put_closed(&clsock->data);

			ht_remove(hs);

			list_remove(socklist, clsock);
                        free(clsock);
			nsock--;

		} else
			s = s->next;
        }
}

static void sock_check(void)
{
	if (!kernel_patched)
		return;
	
	init_msg(full_msg ? SOCKMON_FULL : SOCKMON_CHANGES);
        sock_read();
        
	if (!full_mode)
		clear_closed();

	if (full_msg)
		send_msg();
}

static void sock_getmsg(void)
{
	struct cis_sockinfo sock, *msg = &sock;
	struct socket_info *s, **hs;
	int err;

	err = read(sockmon.fd, msg, sizeof(struct cis_sockinfo));
	if (err != sizeof(struct cis_sockinfo))
		return;

	hs = ht_find(msg);
	if (!(*hs)) {
		if (!(s = register_socket(msg, hs)))
			return;
		hs = ht_find(msg);

		init_msg(SOCKMON_NEW);
		put(&s->data, SOCKINFO_LEN);
		send_msg();
	}
	s = *hs;

	if ((int) msg->priv == OBJ_CREATE)
		return;

	init_msg(SOCKMON_CHANGES);
	put_closed(&s->data);
	send_msg();

	ht_remove(hs);
	list_remove(socklist, s);
	free(s);
	nsock--;
}

struct monitor_desc sockmon = {
	SOCKMON,
	-1,
	sock_init,
	sock_check,
	sock_getmsg,
	sock_send_init,
	sock_send_exit,
};

