/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Common functions for all monitors.
 */

#include "cis.h"
#include "mon_common.h"

char monitor_type;

int            socketfd;
clock_t        start_ct, time_ct;
int            ct_per_sec;
struct in_addr addrlist[MAX_ADDR_NUM];
char           *srv_name;
struct in_addr srv_addr;
int            full_mode = FALSE;
char           my_hostname[MAXHOSTNAMELEN];

int mypid;
int myuid;

float interval;
struct timeval interval_tv;

struct monitor_msg buff = {0, 0, 1};
char  *buffptr = buff.data;
short *endptr = (short *)buff.data;
int   full_msg;
char  tag;
short zero = 0;

struct timeval _start_time, _end_time;

int server_ok;

char *mon_name[] = {"", "monitor", "sysmon", "procmon", "sockmon", "netmon"};

void parse_args(int argc, char **argv)
{
	int c;
	
	while ((c = getopt(argc, argv, "f")) != EOF) {
		
		switch (c) {
		case 'f':
			full_mode = TRUE;
			break;
		case ':':
		case '?':
			goto arg_error;
		}
	}
	if (argv[optind])
		srv_name = argv[optind];
	return;

arg_error:
	printf ("usage: %s [hostname]\n", mon_name[(int) monitor_type]);
	exit(1);
}

FILE *proc_open(char *name)
{
	char buff[128];
	FILE *f;
	
	sprintf(buff, "/proc/%s", name);
	f = fopen(buff,"r");
	if (!f) {
		syslog(LOG_ERR, "cannot open /proc/%s", name);
		exit(1);
	}
	return f;
}

#define SOCKFLAGS     (MSG_DONTWAIT | MSG_NOSIGNAL)

void init_msg(char msgtag)
{
	*endptr = bufflen;

	tag = msgtag;
	put_object(tag);
	endptr = (short *)buffptr;
	put_object(zero);
}

void buffer_send(void)
{
	int err;
	
	buff.time = times(NULL);
	buff.seqnum++;
        *endptr = bufflen;

	err = send(socketfd, &buff, bufflen, SOCKFLAGS);
	buffptr = buff.data;
	if (buff.flags & MSG_FIRST)
		buff.flags &= ~MSG_FIRST;
	init_msg(tag);
}

void send_msg(void)
{
	int err;

	buff.time = times(NULL);
	buff.seqnum++;
	buff.flags |= MSG_LAST;
	*endptr = bufflen;

	err = send(socketfd, &buff, bufflen, SOCKFLAGS);
	buffptr = buff.data;
	endptr = (short *)buffptr;

	buff.flags = MSG_FIRST;
}

void pack_init(char msgtag)
{
	int i, n;

	start_ct = times(NULL);
	init_msg(msgtag);
	buff.flags |= MSG_INIT;

	put_object(start_ct);
	put_object(ct_per_sec);
	for (i = 0; addrlist[i].s_addr; i++)
		;
	n = i;
	put_object(n);
	for (i = 0; addrlist[i].s_addr; i++)
		put_object(addrlist[i]);

}

void send_exit(char msgtag)
{
	start_ct = times(NULL);
	init_msg(msgtag);
	send_msg();
}

static void cleanup(int sig)
{
	init_msg((monitor_type << 4) + 1);
	send_msg();
#ifndef DEBUG
	closelog();
#endif
	exit(0);
}

void prepare_init(int argc, char *argv[])
{
	struct hostent *hent;
	int i;
	
	setlocale(LC_ALL,"");
	setlocale(LC_NUMERIC, "C");

#ifndef DEBUG
	if (getuid()) {
		fprintf(stderr, "Only root can execute %s.\n", mon_name[(int) monitor_type]);
		exit (1);
	}
	
	openlog(mon_name[monitor_type], LOG_CONS, LOG_DAEMON);
#endif
	
	mypid = getpid();
	myuid = getuid();

	if (isroot())
		setpriority(PRIO_PROCESS, mypid, MY_PRIORITY);

	if (gethostname(my_hostname, MAXHOSTNAMELEN) == -1) {
		syslog(LOG_ERR, "cannot get my hostname");
		exit(1);
	}
	if (!(hent = gethostbyname(my_hostname))) {
		syslog(LOG_ERR, "cannot get my address");
		exit(1);
	}
	
	for (i = 0; hent->h_addr_list[i]; i++)
		memcpy(&addrlist[i], hent->h_addr_list[i], hent->h_length);

	ct_per_sec = sysconf(_SC_CLK_TCK);
	/*
	 * Parse the command line for parameters.
	 */
	parse_args(argc, argv);
	
	if (!srv_name)
		srv_name = my_hostname;
	if (!(hent = gethostbyname(srv_name))) {
		syslog(LOG_ERR, "Host %s was not found", srv_name);
		exit(1);
	}
	srv_addr.s_addr = *(long *) hent->h_addr;

	/*
	 * We need to unregister rpc service before terminating ...
	 */
	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);
}

void connect_server()
{
	struct sockaddr_in inaddr = {AF_INET, 0, {htonl (INADDR_ANY)}};

	socketfd = socket(PF_INET, SOCK_DGRAM, 0);
	if (socketfd == -1) {
		syslog(LOG_ERR, "cannot create UDP client socket");
		exit(1);
	}
	
	inaddr.sin_port = htons(CISD_PORT+monitor_type);
	if (bind (socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
		syslog(LOG_ERR, "cannot bind UDP client socket");
		exit(1);
	}
	
	inaddr.sin_port = htons(CISD_PORT);
	inaddr.sin_addr = srv_addr;
	if (connect(socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
		syslog(LOG_ERR, "cannot connect to CIS server");
		exit(1);
	}
	server_ok = TRUE;
}

void recv_interval()
{
	struct timeval t = {0, 50000};
	struct monitor_reply reply;
        int err;
	
	setfdnoblock(socketfd);
	select(0, NULL, NULL, NULL, &t);

	err = recv(socketfd, &reply, sizeof(reply), 0);

	if (err != sizeof(reply) || reply.code != 1) {
		reply.interval = DEF_INTERVAL;
		server_ok = FALSE;
	}

	interval = reply.interval;
	sectotval(interval_tv, reply.interval);
	
	full_msg = TRUE;
}

int recv_msg()
{
	int err, ret = FALSE;
	struct monitor_reply reply;

	while ((err = recv(socketfd, &reply, sizeof(reply), MSG_NOSIGNAL)) == sizeof(reply)) {

		if (reply.code == -EAGAIN)
			ret = TRUE;

		if (reply.code == -EFAULT)
			full_msg = TRUE;

		server_ok = TRUE;
		buff.seqnum = reply.seqnum;

		interval = reply.interval;
		sectotval(interval_tv, reply.interval);
	}

	if (err == -1 && errno == ECONNREFUSED)
                server_ok = FALSE;
	
	return ret;
}
