/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Monitoring of LM sensors.
 */

#include "cis.h"
#include "mon_common.h"

#include "sensors/sensors.h"

struct lm_sensor {
	struct cis_lmsinfo data;

	const sensors_chip_name *chip;
	int number;
} sensors[] = {
	{{"SYS Temp",   100, 0}, NULL, 0},
	{{"CPU Temp",   100, 0}, NULL, 0},
	{{"CPU Fan",      1, 0}, NULL, 0},
	{{"CPU 1 Temp", 100, 0}, NULL, 0},
	{{"CPU 1 Fan",    1, 0}, NULL, 0},
	{{"CPU 2 Temp", 100, 0}, NULL, 0},
	{{"CPU 2 Fan",    1, 0}, NULL, 0},
};

#define NSENSORS (sizeof(sensors) / sizeof(sensors[0]))

#define for_each_sensor(s) \
	for (s = sensors; s < sensors + NSENSORS; s++)

static int nsensors = NSENSORS;

static char *config_filename = "/etc/sensors.conf";
static FILE *config_file;

static int sensors_ok = FALSE;

static void lm_init (void)
{
	const sensors_chip_name *chip;
	const sensors_feature_data *data;
	struct lm_sensor *s;
	int a = 0, b = 0;
	int chip_nr;
	char *label;
        double value;
	
	/*
	 * Initialize input.
	 */
	if (!(config_file = fopen(config_filename, "r"))) {
		syslog(LOG_ERR, "cannot open %s.", config_filename);
		return;
	}

	if (sensors_init(config_file) != 0) {
		syslog(LOG_ERR, "cannot initialize sensors.");
		return;
	}

        /* Find chips with CPU and System temperature */
	for (chip_nr = 0; (chip = sensors_get_detected_chips(&chip_nr));)
		while((data = sensors_get_all_features(*chip, &a, &b))) {
			if (sensors_get_label(*chip, data->number, &label) != 0)
				continue;
			if (!(data->mode & SENSORS_MODE_R))
				continue;
			if (sensors_get_feature(*chip, data->number, &value) != 0)
				continue;
//			if (data->mapping == SENSORS_NO_MAPPING)
//				continue;

			for_each_sensor(s)
				if (!strcmp(label, s->data.name)) {
					s->chip = chip;
					s->number = data->number;
					s->data.value = value * s->data.divider;
					sensors_ok = TRUE;
				}
			free(label);
		}
}

static void lm_send_init (void)
{
	pack_init(LMMON_INIT);
	put_object(nsensors);
	send_msg();
	recv_interval();
}

static void lm_send_exit (void)
{
	return (send_exit(LMMON_EXIT));
}

static void lm_check(void)
{
	struct lm_sensor *s;
	short old_value;

	if (!sensors_ok)
		return;
	
	if (full_msg) {
		init_msg(LMMON_FULL);
	} else
		init_msg(LMMON_CHANGES);

	for_each_sensor(s) {
		unsigned char id = s - sensors;
		double val;

		if (!s->chip) continue;
		
		old_value = s->data.value;

		if (sensors_get_feature(*s->chip, s->number, &val))
			val = 0.;

		s->data.value = val * s->data.divider;

		if (full_msg) {
			put_object(id);
			put_object(s->data);
		} else if (s->data.value != old_value) {
			if ((bufflen + sizeof(double) + 1) > MON_BUFFLEN)
				buffer_send();

			put_object(id);
			put_object(s->data.value);
		}
	}

	if (full_msg)
		send_msg();
}

struct monitor_desc lmmon = {
	LMMON,
	-1,
	lm_init,
	lm_check,
	0,
	lm_send_init,
	lm_send_exit,
};
