/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Hash table manipulating functions.
 */

#include <stdlib.h>
#include "hash.h"

#define ht_length(a) (a * 3 / 2)

hash_t *hash_create(int length, int key_offset, int key_length, int hcode_offset)
{
	hash_t *ht;

	ht = calloc(1, sizeof (hash_t));
	if (!ht)
		return NULL;

	ht->table = calloc(ht_length(length), sizeof (void *));
	if (!ht->table) {
		free(ht);
		return NULL;
	}

	ht->length       = length;
	ht->key_offset   = key_offset;
	ht->key_length   = key_length;
	ht->hcode_offset = hcode_offset;

	return ht;
}

void hash_destroy(hash_t *ht)
{
	if (!ht)
		return;

	free(ht->table);
	free(ht);
}

void hash_clear(hash_t *ht)
{
	if (!ht)
		return;
	
	memset (ht->table, 0, ht_length(ht->length) * sizeof(void *));
	ht->nitems = 0;
}

#define hash(a)    (((((unsigned int) (a)) * 3) / 2) % length)
#define ht_find(keyptr) \
	idx = hash(*(unsigned short *)((char *) keyptr + ht->hcode_offset)); \
	while (tab[idx] && memcmp(keyptr, (char *) tab[idx] + key_offset, key_length)) \
	       if (++idx == length) \
                       idx = 0;

void *hash_find(hash_t *ht, void *keyptr)
{
	void **tab = ht->table;
	int length = ht_length(ht->length);
	int key_length = ht->key_length;
	int key_offset = ht->key_offset;
	int idx;

	ht_find(keyptr);

	return (tab[idx]);
}

int hash_insert(hash_t *ht, void *item)
{
	void **tab = ht->table;
	int length = ht_length(ht->length);
	int key_length = ht->key_length;
	int key_offset = ht->key_offset;
	void *keyptr = (char *) item + key_offset;
	int idx;

	ht_find(keyptr);

	if (tab[idx]) {
		tab[idx] = item;
		return 1;
	}

	tab[idx] = item;
	ht->nitems++;
	
	if (ht_length(ht->nitems) > length) {
		void **new_tab, **old_tab = tab;

		new_tab = calloc(ht_length(length), sizeof (void *));
		if (!new_tab)
			return 0;

		ht->table  = new_tab;
		ht->length = length;
		ht->nitems = 0;

		/* Rehash the table */
		for (idx = 0; idx < length; idx++)
			if (old_tab[idx])
				hash_insert(ht, old_tab[idx]);

		free(old_tab);
	}
	
	return 1;
}

int hash_remove(hash_t *ht, void *keyptr)
{
	void **tab = ht->table;
	int length = ht_length(ht->length);
	int key_length = ht->key_length;
	int key_offset = ht->key_offset;
	int idx, hole;

	ht_find(keyptr);

	if (!tab[idx])
		return 0;

	hole = idx;
	tab[hole] = NULL;
	
	for(;;) {
		if (++idx == length)
			idx = 0;
		
		if (!tab[idx]) break;
		
		if (!hash_find(ht, (char *) tab[idx] + key_offset))
			tab[hole] = tab[idx], tab[idx] = NULL, hole = idx;
	}
	ht->nitems--;
	return 1;
}

