/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * CIS server daemon - core code with interface to monitors.
 */

#include <unistd.h>
#include <signal.h>
#include <locale.h>
#include <syslog.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/utsname.h>
#include <netinet/in.h>

#include "cis.h"
#include "cis_xdr.h"
#include "cisd.h"

#define DEF_CONFIG                "/etc/cis.conf"

#define DEF_OPEN_MODE             O_CREAT|O_APPEND|O_NOCTTY

#ifndef DEF_INTERVAL
#define DEF_INTERVAL              1.
#endif

#ifndef DEF_MSG_COUNT
#define DEF_MSG_COUNT             60
#endif

#ifndef PROC_HT_LENGTH
#define PROC_HT_LENGTH            16
#endif

#ifndef SOCK_HT_LENGTH
#define SOCK_HT_LENGTH            16
#endif

#ifndef SOCKET_TIMEOUT
#define SOCKET_TIMEOUT            60.    /* in seconds */
#endif


#define SOCKFLAGS     (MSG_DONTWAIT | MSG_NOSIGNAL)

#define allocCPU(a,b,c) (((a) * ((10000 * (b)) / (c))) / 10000)

#ifdef DEBUG
#define syslog(tag, fmt, args...)  printf(fmt"\n" , ## args)
#endif

#define CONSISTENT   1
#define TMP_FLAG     (1<<5)

#define is_marked(a) (a->flags & TMP_FLAG)
#define mark(a)      {a->flags |= TMP_FLAG;}
#define unmark(a)    {a->flags &= ~TMP_FLAG;}

#define for_each_process(host, p) \
	for (p = host->proclist; p; p = p->next)

#define for_each_socket(host, s) \
	for (s = host->socklist; s; s = s->next)


extern void dispatcher(struct svc_req *rqstp, SVCXPRT *transp);
extern void daemonize (void);
extern uint create_auth_code (void);

struct sockaddr_in srv_addr;
struct in_addr myaddr;

char cis_name[CIS_HOSTNAMELEN];
struct cis_info my_info = {cis_name};

int prognum = CIS_PROG;

struct host_info *host_list = NULL;

struct timeval timeout={5,0};
struct timeval current_tv;

struct monitor_msg monbuff;
char *buffptr = monbuff.data;
#define bufflen (buffptr - (char *) &monbuff)

short msglen;
int monitor_socket;
float interval = DEF_INTERVAL;
struct monitor_reply reply;

char *config_file = NULL;

int accept_new_hosts = 0;

char *commands[]   = {"#", "Name", "PerfTypes", "PerfValues", "Reliability", "AcceptNewHosts"};
char *rel_types[]  = {"minimal", "low", "average", "high", "maximal"};
char *yesno[]      = {"no", "yes"};

enum {
        CMD_COMMENT,
        CMD_NAME,
        CMD_PERFTYPES,
        CMD_PERFVALUES,
	CMD_RELIABILITY,
	CMD_ACCEPTNEWHOSTS,
	CMD_TABLEN,
};

char **perf_types = NULL;
int nperf_types = 0;

void parse_args(int argc, char **argv)
{
        int c;

	while ((c = getopt(argc, argv, "i:c:")) != EOF) {

                switch (c) {
                case 'i':
                        interval = atof(optarg);
                        if (interval == 0.) {
                                syslog(LOG_WARNING, "interval cannot be set to zero. reset to default %5.2f [s]", DEF_INTERVAL);
                                interval = DEF_INTERVAL;
                        }
                        break;
                case 'c':
                        config_file = optarg;
			break;
                }
        }
}

int parse_cmd (char *cmds[], int n, char **c)
{
	int i;

	*c += strspn (*c, " \t");

	for (i = 0; i < n; i++)
		if (!strncasecmp (*c, cmds[i], strlen (cmds[i]))) {
			*c += strlen(cmds[i]);
			return (i);
		}
	return -1;
}

int parse_config(void)
{
	struct host_info *host, *h;
        struct hostent *hent;
	FILE *file = NULL;
	int line = 0;
	char name[MAXHOSTNAMELEN];
	char *c, *type;
	int i, cmd;
	int rel = 0;
	float *performance = NULL;
	
	int buflen = 0;
	char *buf  = NULL;

	/*
         * Open the config file.
         */
	if (!(file = fopen(config_file ? config_file : DEF_CONFIG, "r"))) {
		syslog(LOG_ERR, "cannot open config file %s", config_file);
		exit (1);
	}

	*cis_name = 0;
	
	while (getline(&buf, &buflen, file) != -1) {

		line++;
		c = buf;
		
		cmd = parse_cmd(commands, CMD_TABLEN, &c);
		if (cmd == 0) continue;

		c += strspn(c, " \t");       /* strip white spaces */

                switch (cmd) {
                        
		case CMD_NAME:
			sscanf (c, "%s", cis_name);
			break;
			
		case CMD_PERFTYPES:

			c = strdup(c);
			type = strtok(c, ", \t\n");

			do {
				perf_types = realloc(perf_types, (nperf_types+1)* sizeof(char *));
				if (!perf_types)
					goto nomem;

				perf_types[nperf_types++] = type;

			} while ((type = strtok(NULL, ", \t\n")));

			performance = calloc(nperf_types, sizeof(float));

			break;

		case CMD_PERFVALUES:

			for (i = 0; *c; i++, c++) {

				performance[i] = atof(c);

				c = strchr(c, ',');
				if (i == nperf_types || !c)
					break;

				c++;
			}
			break;

		case CMD_RELIABILITY:

			rel = parse_cmd(rel_types, REL_TABLEN, &c);
			if (rel == -1)
				goto error;

			break;

		case CMD_ACCEPTNEWHOSTS:

			accept_new_hosts = TRUE; //parse_cmd(yesno, 2, &c);
			break;

		default:

			if (sscanf(c, "%s", name) != 1 || *name == '\n')
				continue;

			if ((hent = gethostbyname(name)) == NULL) {
				syslog(LOG_ERR, "cannot get the address of host %s", name);
				exit(1);
			}

			host = calloc(1, sizeof (struct host_info));
			if (!host)
				goto nomem;

			if (*cis_name)
				snprintf(host->hostinfo.name, CIS_HOSTNAMELEN, "%s.%s", cis_name, name);
			else
				snprintf(host->hostinfo.name, CIS_HOSTNAMELEN, "%s", name);

			for (i = 0; hent->h_addr_list[i]; i++)
                                memcpy(&host->addrlist[i], hent->h_addr_list[i], hent->h_length);
			host->naddr = i;

			host->hostinfo.reliability = rel;
			host->performance = malloc(nperf_types * sizeof(float));
			if (!host->performance)
				goto nomem;

			host->hostinfo.addr  = host->addrlist[0];
			host->hostinfo.caddr = myaddr;
			
			memcpy (host->performance, performance, nperf_types*sizeof(float));

			if (!host_list)
				host_list = host;
			else {
				for (h = host_list; h->next; h = h->next)
					;
				h->next = host;
			}

			break;
		}
	}
	fclose (file);

       	return 1;
nomem:
	syslog(LOG_ERR, "not enough memory");
       	exit(1);

error:
	syslog(LOG_ERR, "bad entry in config file at line %d", line);
	exit (1);
}


void free_host(struct host_info *host)
{
	hash_destroy(host->proc_ht);
	hash_destroy(host->sock_ht);
        free(host);
}

void delete_host(struct host_info *host)
{
        static struct host_info *h;
        
        if (host == host_list)
                host_list = host->next;
        else {
                for (h = host_list; h->next && h->next != host; h = h->next);
                
                if (h->next != host)
                        return; /* Not found */
                
                h->next = host->next;
        }
        free_host(host);
}

struct host_info *find_host(struct in_addr addr)
{
        struct host_info *h;
        struct in_addr *a;

	if (addr.s_addr == 0x0100007f)
                addr = myaddr;
	
        for (h = host_list; h; h = h->next) {
		if (h->hostinfo.addr.s_addr == addr.s_addr)
                        return h;
                for (a = h->addrlist; a < h->addrlist + h->naddr; a++)
                        if (a->s_addr == addr.s_addr)
                                return h;
        }
        return NULL;
}

int monitor_idle(struct monitor_info *monitor)
{
//	return (monitor_time(monitor) > (monitor->local_time + 2*monitor->clk_interval));

	return (tvaldiff (current_tv, monitor->time_tv) > 3*interval);
}

unsigned long count_rate(struct history *h, struct clnt_req *req)
{
	unsigned long sum = 0;
        unsigned long time  = req->time;
        unsigned long cticks = req->host->hostinfo.ct_per_sec;
        unsigned long bound = time - req->interval * cticks;
        struct history_value *hp;

	if (!h->last)
		return 0;
	
        if (h->last->time < bound)
                return 0;

        if (h->last->time - h->interval > bound)
                return (h->sum / (((float) (time - h->last->time + h->interval))
                        / cticks));

        for (hp = h->last; hp->time >= bound;) {

                sum += hp->change;

                if (--hp < h->history)
                        hp = h->history + HISTORY_LENGTH - 1;

                if (hp == h->last)
                        break;
        }

        return (sum / req->interval);
}


/*
 * Process related functions
 */
/*int slice (struct host_info *host, int niceval)
{
        int slice = niceval;
        
        if (niceval < 0)
                slice = -niceval;
        if (slice > 20)
                slice = 20;
        slice = (slice * host->defprio + 10) / 20 + host->defprio;

        if (niceval >= 0) {
                slice = 2*host->defprio - slice;
                if (!slice)
                        slice = 1;
        }
        return (slice);
        } */

int count_pCPU(struct host_info *host, struct process_info *p, unsigned long current_time)
{
	int pCPU = 0;
	struct history *hist = &p->CPU_history;
	unsigned long interval, sum;

        /*
         * No usage.
         */
        if (!hist->interval)
		goto out;

        history_sum(hist, PROC_HISTORY_LENGTH, &interval, &sum);
	
        /*
         * No change during 2*'interval', process is sleeping.
         */
	if (current_time > hist->last->time + 2*interval)
                goto out;

        /*
         * Hold the last usage during 'interval' after last change.
         */
        pCPU  = (10000 * sum) / interval;

        if (current_time < hist->last->time + interval)
                goto out;

        /*
         * Between, usage will fall down to zero.
         */
        pCPU *= hist->last->time + 2*interval - current_time;
        pCPU /= interval;

    out:
        return pCPU;
}

int availableCPU(struct host_info *host, unsigned long clk)
{
	struct process_info *p;
        struct cis_procinfo *pi;
	struct history *hist;
        struct monitor_info *monitor = host->monitor[PROCMON];
        unsigned long psum_tmp, slsum_tmp, available_tmp;
        unsigned long available = 0;
        unsigned long min_interval;
        unsigned long psum = 0;
        unsigned long slsum = 0;
        int n = 0;

	if (!monitor)
		return 0;
	
	if (clk > 3*monitor->local_time) /* monitor died without message */
		return 0;

	min_interval = 2*monitor->clk_interval;

	for_each_process (host, p) {
		pi = &p->data;
		hist = &p->CPU_history;
                n++;
		unmark(p);
                
                /* What CPU usage process has ? */
                if (hist->interval) {
                        pi->pCPU = count_pCPU(host, p, clk);
                        slsum += pi->priority;
                        psum += pi->pCPU;
			mark(p);
                        
                        /*
                         * Minimal process interval is needed for deciding whether
                         * the CPU was busy all the time or it was idle some time.
                         */
                        if (hist->interval < min_interval)
                                min_interval = hist->interval;
                        continue;
                }
                pi->pCPU = 0;

                if (clk < p->CPU_history.last->time + 2*p->init_interval) {
                        pi->pCPU = 10000;

                        if (clk > p->CPU_history.last->time + p->init_interval) {
                                pi->pCPU *= p->CPU_history.last->time + 2*p->init_interval - clk;
                                pi->pCPU /= p->init_interval;
                        }
                }
        }

        host->hostinfo.procnum = n;

        if (!n)
                return (0);

        if (!psum)
                return (10000);

        if (host->idle_at < monitor->local_time - min_interval && psum < 10000) {
                /*
                 * If CPU was not idle and there was terminated process
                 * (i.e. pCPU sum is < 100 %), we must recompute
                 * needs of all 'greedy' processes.
                 */
                available = 10000 - psum;

                slsum_tmp = 0; psum_tmp = psum;
		for_each_process (host, p) {
			if (!is_marked(p)) continue;

                        if (p->data.pCPU < allocCPU(psum, p->data.priority, slsum)) {
				psum_tmp -= p->data.pCPU;
                                unmark(p);
                        }
                        else
                                slsum_tmp += p->data.priority;

                }
                slsum = slsum_tmp;
                psum = psum_tmp;
                
		for_each_process (host, p) {
			if (!is_marked(p)) continue;

			p->data.pCPU = allocCPU(psum + available, p->data.pCPU, psum);
                        unmark(p);
		}
	}

        /*
         * Estimate how much CPU time a process with default priority
         * can get (default response time).
         */
        slsum = host->defprio + host->slsum;
        available = 20000;
        available_tmp = 10000;

        while (available > available_tmp) {

                available = available_tmp;
                slsum_tmp = host->defprio;

		for_each_process (host, p) {
			if (!p->data.pCPU) continue;

			if (p->data.pCPU < allocCPU(available, p->data.priority, slsum)) {
				available_tmp -= p->data.pCPU;
                                p->data.pCPU = 0;
                        } else
				slsum_tmp += p->data.priority;
                }
                slsum = slsum_tmp;
        }

        available = allocCPU(available, host->defprio, slsum);

        return (available);
}


struct process_info *proc_add(struct host_info *host, struct cis_procinfo *p)
{
	struct process_info *process, *q;
	struct monitor_info *monitor = host->monitor[PROCMON];
	int interval = 0;
        unsigned long time;
        
	process = calloc(1, sizeof (struct process_info));
	if (!process)
		return NULL;

	if (p) memcpy(&process->data, p, sizeof (struct cis_procinfo));

	/*        time = process->start_time;
        if (!time)*/
	time = monitor->local_time - monitor->clk_interval;

        /* Recompute start_time to GMT time. It's ugly but we have to cope
         * with possibly very high unsigned longs */
        process->GMT_start_time = monitor->start_tv.tv_sec;
        if (time < monitor->start_clk)
                process->GMT_start_time -= (monitor->start_clk - time)/monitor->ct_per_sec;
        else
		process->GMT_start_time += (time - monitor->start_clk)/monitor->ct_per_sec;
         
        /* Process history initialization. */
        init_history(&process->CPU_history,    time);
        init_history(&process->majflt_history, time);
        init_history(&process->read_history,   time);
        init_history(&process->write_history,  time);

        process->flags = CONSISTENT;

	list_insert (host->proclist, process);

        /* How much to wait for first usage ? */
        if (host->slsum)
		for_each_process (host, q) {
			q->data.pCPU = count_pCPU(host, q, monitor->local_time);
                        /* Compute time to wait for CPU usage (i.e. the sum of
                         * slices of all CPU intensive processes with higher
                         * priority than the new process has */
                        if (q->data.priority >= process->data.priority &&
                            q->data.pCPU > ((10000 * q->data.priority) / host->slsum))
				interval += q->data.priority;
                }

        if (interval < monitor->clk_interval)
                process->init_interval = monitor->clk_interval;
        else
                process->init_interval = interval;

	host->slsum += process->data.priority;

	hash_insert(host->proc_ht, process);
        
	return process;
}

/*
 * When parent exits, ppid for all children should be set to 1 (init).
 * In order to keep the process tree consistent.
 */
void proc_release_children(struct host_info *host, int pid)
{
	struct process_info *p;

	for_each_process (host, p)
		if (p->data.ppid == pid)
			p->data.ppid = 1;
}

void proc_remove(struct host_info *host, struct process_info *process)
{
	list_remove (host->proclist, process);

	host->slsum -= process->data.priority;
        proc_release_children(host, process->data.pid);
	hash_remove(host->proc_ht, &process->data.pid);

	free(process);
}

void proc_update(struct host_info *host, struct process_info *p, struct cis_procinfo *newp)
{
	struct cis_procinfo *oldp = &p->data;
	struct monitor_info *monitor = host->monitor[PROCMON];
        unsigned short ctps = host->hostinfo.ct_per_sec;
	unsigned long change;

	if (oldp->priority != newp->priority)
		host->slsum += newp->priority - oldp->priority;

        change = newp->utime - oldp->utime + newp->stime - oldp->stime;
	if (change) {
		history_add(&p->CPU_history, monitor->local_time, change);
		
		p->utime_sec = newp->utime/ctps;
		p->stime_sec = newp->stime/ctps;
	}

	history_add(&p->majflt_history, monitor->local_time, newp->majflt   - oldp->majflt);
	history_add(&p->read_history,   monitor->local_time, newp->disk_read - oldp->disk_read);
	history_add(&p->write_history,  monitor->local_time, newp->disk_write - oldp->disk_write);

	memcpy(oldp, newp, PROCINFO_LEN);

	p->flags |= CONSISTENT;
}

static inline struct process_info *proc_find(struct host_info *host, unsigned int pid)
{
	if (!host->monitor[PROCMON])
		return NULL;

	return hash_find(host->proc_ht, &pid);
}


void proc_test(struct host_info *host, struct cis_procinfo *p)
{
	struct process_info *process = proc_find(host, p->pid);

        if (!process)
		proc_add(host, p);
        else
                proc_update(host, process, p);
}


void clear_proclist(struct host_info *host)
{
        while (host->proclist)
		proc_remove(host, host->proclist);
}


/*
 * Socket related functions
 */
struct socket_info *sock_add(struct host_info *host, struct cis_sockinfo *s)
{
	struct socket_info *socket;
	unsigned long time = host->monitor[SOCKMON]->local_time;

	socket = calloc(1, sizeof (struct socket_info));
        if (!socket)
		return NULL;
	memcpy(&socket->data, s, sizeof (struct cis_sockinfo));

	list_insert (host->socklist, socket);

	init_history(&socket->sent_history, time);
        init_history(&socket->rcvd_history, time);

        socket->flags = CONSISTENT;

	hash_insert(host->sock_ht, socket);

        return socket;
}

void sock_update(struct host_info *host, struct socket_info *s, struct cis_sockinfo *news)
{
	struct cis_sockinfo *olds = &s->data;
	unsigned long time = host->monitor[SOCKMON]->local_time;

	history_add(&s->sent_history, time, news->sent - olds->sent);
	history_add(&s->rcvd_history, time, news->rcvd - olds->rcvd);

        memcpy(olds, news, SOCKINFO_LEN);

	s->flags |= CONSISTENT;
}

void sock_remove (struct host_info *host, struct socket_info *s)
{
	s->flags &= ~CONSISTENT;
	hash_remove(host->sock_ht, s);
}

void clear_sockets(struct host_info *host, unsigned long time)
{
	struct socket_info *s, *tmp;

	time -= SOCKET_TIMEOUT * host->hostinfo.ct_per_sec;

	for (s = host->socklist; s;) {
		tmp = NULL;

		if (!(s->flags&CONSISTENT) &&
                    s->sent_history.last->time < time &&
		    s->rcvd_history.last->time < time)
			tmp = s;

		s = s->next;

                if (tmp) {
			list_remove(host->socklist, tmp);
			free(tmp);
		}
	}
}

void clear_socklist(struct host_info *host)
{
	struct socket_info *socket;

	while (host->socklist) {
		socket = host->socklist;
		host->socklist = host->socklist->next;
		hash_remove(host->sock_ht, &socket->data);
		free(socket);
        }
}

static inline struct socket_info *sock_find(struct host_info *host,  struct cis_sockinfo *s)
{
	if (!host->monitor[SOCKMON])
		return NULL;

	return hash_find(host->sock_ht, s);
}

/*
 * Network device related functions
 */
static inline struct netdev_info *netdev_find(struct host_info *host, unsigned char id)
{
	struct netdev_info *d;

        for (d = host->netdevlist; d; d = d->next)
		if (d->data.id == id)
			return d;
	return NULL;
}

struct netdev_info *netdev_add (struct host_info *host, struct cis_netdevinfo *dev)
{
	struct netdev_info *device = calloc(1, sizeof (struct netdev_info));
	unsigned long time = host->monitor[NETMON]->local_time;

	if (!device)
		return NULL;
	memcpy(&device->data, dev, NETDEVINFO_LEN);

	init_history(&device->rxbytes_history, time);
        init_history(&device->txbytes_history, time);
	init_history(&device->collisions_history, time);

	device->flags = CONSISTENT;
	list_insert(host->netdevlist, device);

        return device;
}

void netdev_update(struct host_info *host, struct netdev_info *d, struct cis_netdevinfo *newd)
{
	struct cis_netdevinfo *oldd = &d->data;
	unsigned long time = host->monitor[NETMON]->local_time;
        
	history_add(&d->rxbytes_history, time, newd->receive - oldd->receive);
	history_add(&d->txbytes_history, time, newd->transmit - oldd->transmit);
	history_add(&d->collisions_history, time, newd->collisions - oldd->collisions);

	memcpy (oldd, newd, NETDEVINFO_LEN);
        
	d->flags |= CONSISTENT;
}

void clear_netdevlist(struct host_info *host)
{
	struct netdev_info *device;
        
        while (host->netdevlist) {
		device = host->netdevlist;
		host->netdevlist = host->netdevlist->next;
		free(device);
	}
}

/*
 * Functions for parsing message from monitors
 */
#define get(ptr, len) \
        {                                            \
               if ((bufflen + len) > msglen)         \
                      goto next;                     \
               memcpy(ptr, buffptr, len);            \
                      buffptr += len;                \
        }

#define get_object(obj) get(&obj, sizeof(obj))
#define get_change(obj) \
        {                                    \
               get_object(change);           \
               obj += change;                \
        }

#define buffer_get(obj) \
        {                                                                    \
               if ((bufflen + sizeof(*(obj))) > len)                         \
                      goto next;                                             \
               else {                                                        \
                      memcpy(obj, buffptr, sizeof(*(obj)));                  \
                      buffptr += sizeof(*(obj));                             \
               }                                                             \
        }

#define sys_code(el)   OFFSET(cis_hostinfo, el)

void sys_change(struct cis_hostinfo *hi)
{
	unsigned char code;
        int change;

        get_object(code);

	while (code != 0xff) {
                switch (code) {
                case sys_code(procnum):       get_change(hi->procnum);       break;
                case sys_code(loads[0]):      get_change(hi->loads[0]);      break;
                case sys_code(loads[1]):      get_change(hi->loads[1]);      break;
                case sys_code(loads[2]):      get_change(hi->loads[2]);      break;
                case sys_code(totalswap):     get_change(hi->totalswap);     break;
                case sys_code(freeswap):      get_change(hi->freeswap);      break;
                case sys_code(freeram):       get_change(hi->freeram);       break;
                case sys_code(sharedram):     get_change(hi->sharedram);     break;
                case sys_code(bufferram):     get_change(hi->bufferram);     break;
                case sys_code(cachedram):     get_change(hi->cachedram);     break;
                case sys_code(idle):          get_change(hi->idle);          break;
                case sys_code(ctx_swtch):     get_change(hi->ctx_swtch);     break;
                case sys_code(swpin):         get_change(hi->swpin);         break;
                case sys_code(swpout):        get_change(hi->swpout);        break;
		case sys_code(disk_read[0]):  get_change(hi->disk_read[0]);  break;
                case sys_code(disk_read[1]):  get_change(hi->disk_read[1]);  break;
                case sys_code(disk_read[2]):  get_change(hi->disk_read[2]);  break;
                case sys_code(disk_read[3]):  get_change(hi->disk_read[3]);  break;
		case sys_code(disk_write[0]): get_change(hi->disk_write[0]); break;
                case sys_code(disk_write[1]): get_change(hi->disk_write[1]); break;
                case sys_code(disk_write[2]): get_change(hi->disk_write[2]); break;
                case sys_code(disk_write[3]): get_change(hi->disk_write[3]); break;
		}
		get_object(code);
	}
	next:
}

#define proc_code(el)   OFFSET(cis_procinfo, el)

void proc_change(struct host_info *host, unsigned int pid)
{
        struct cis_procinfo proc;
	struct process_info *process = proc_find(host, pid);
        char code;
	int change;

	get_object(code);

	if (code == 1) {/* New process */
		get(&proc, PROCINFO_LEN);
		proc_test(host, &proc);
		return;
        }
        if (code == 2) {/* Terminated process */
                get(&proc, PROCINFO_LEN);
		proc_remove(host, process);
                return;
        }
        
        if (process)
		proc = process->data;
	else {
		/* Change of nonexisting process */
                memset(&proc, 0, sizeof(proc));
                proc.pid = pid;
                sprintf(proc.cmd, "unknown");
                proc.priority = host->defprio;
                process = proc_add(host, &proc);
        }

        while (code) {
                switch (code) {
                case proc_code(utime):    get_change(proc.utime);    break;
                case proc_code(stime):    get_change(proc.stime);    break;
                case proc_code(minflt):   get_change(proc.minflt);   break;
                case proc_code(majflt):   get_change(proc.majflt);   break;
                case proc_code(disk_read): get_change(proc.disk_read); break;
                case proc_code(disk_write): get_change(proc.disk_write); break;
                case proc_code(priority): get_change(proc.priority); break;
                case proc_code(rss):      get_change(proc.rss);      break;
                case proc_code(vm):       get_change(proc.vm);       break;
		case proc_code(ppid):     get_object(proc.ppid);     break;
		case proc_code(uid):      get_object(proc.uid);      break;
		case proc_code(cmd):      get_object(proc.cmd);      break;
		}
                get_object(code);
        }
        proc_update(host, process, &proc);
    next:
}


void sock_test(struct host_info *host, struct cis_sockinfo *s)
{
	struct socket_info *socket = sock_find(host, s);

	if (!socket)
                sock_add(host, s);
        else
                sock_update(host, socket, s);
}

#define sock_code(el)   OFFSET(cis_sockinfo, el)

void sock_change(struct host_info *host, struct cis_sockinfo *s)
{
	struct cis_sockinfo sock;
	struct socket_info *socket = sock_find(host, s);
	char code;
        int change;

        get_object(code);

        if (socket)
		sock = socket->data;
        else {
                /* Change of nonexisting socket */
		memset(&sock, 0, sizeof(sock));
		memcpy(&sock, s, SOCKINFO_HDRLEN);
		socket = sock_add(host, &sock);
        }

        if (code == 1) {/* New socket */
                get(&sock + SOCKINFO_HDRLEN, SOCKINFO_LEN-SOCKINFO_HDRLEN);
                sock_update(host, socket, &sock);
                return;
        }
        if (code == 2) {/* Closed socket */
                get(&sock + SOCKINFO_HDRLEN, SOCKINFO_LEN-SOCKINFO_HDRLEN);
		sock_update(host, socket, &sock);
                sock_remove(host, socket);
                return;
        }

	while (code) {
                switch (code) {
                case sock_code(pid):      get_object(sock.pid);      break;
                case sock_code(uid):      get_object(sock.uid);      break;
                case sock_code(sent):     get_change(sock.sent);     break;
                case sock_code(rcvd):     get_change(sock.rcvd);     break;
                }
                get_object(code);
        }
        sock_update(host, socket, &sock);
    next:
}

void netdev_test(struct host_info *host, struct cis_netdevinfo *d)
{
	struct netdev_info *device = netdev_find(host, d->id);

	if (!device)
                netdev_add(host, d);
        else
                netdev_update(host, device, d);
}

#define netdev_code(el)   OFFSET(cis_netdevinfo, el)

void netdev_change(struct host_info *host, unsigned char id)
{
	struct cis_netdevinfo dev;
	struct netdev_info *device = netdev_find(host, id);
	char code;
	int change;

        get_object(code);

	if (code == 1) {/* New device */
                get(&dev, NETDEVINFO_LEN);
                netdev_test(host, &dev);
                return;
        }
        
        if (device)
		dev = device->data;
        else {
                /* Change of nonexisting device */
                memset(&dev, 0, sizeof(dev));
		dev.id = id;
		sprintf(dev.name, "unknown");
		device = netdev_add(host, &dev);
        }

        while (code) {
                switch (code) {
                case netdev_code(transmit):   get_change(dev.transmit);    break;
                case netdev_code(receive):    get_change(dev.receive);    break;
                case netdev_code(collisions): get_change(dev.collisions);  break;
                case netdev_code(status):     get_object(dev.status);      break;
                }
                get_object(code);
        }
        netdev_update(host, device, &dev);
    next:
}

int parse_init_msg(struct host_info *host, char tag)
{
        unsigned long  start_clk;
        int naddr, i;
        struct in_addr addrlist[MAX_ADDR_NUM];
        struct utsname uname;
        int ct_per_sec, defprio;
	unsigned long totalram, totalswap;
	unsigned char nsensors;
	struct cis_lmsinfo *newtab;

	struct monitor_info *monitor = NULL;
	int mtype = monitor_type(tag);

	/*
         * Parse init message
         */
        get_object(start_clk);
        get_object(ct_per_sec);
        get_object(naddr);
        for (i = 0; i < naddr; i++)
                get_object(addrlist[i]);

	if (mtype == SYSMON) {
		get_object(uname);
                get_object(defprio);
                get_object(totalram);
                get_object(totalswap);
	}
	if (mtype == LMMON)
		get_object(nsensors);
	/*
         * Init message was ok.
         */
	monitor = host->monitor[mtype];

        monitor->clk_interval = interval*ct_per_sec;

        monitor->start_clk  = start_clk;
        monitor->ct_per_sec = ct_per_sec;
        monitor->start_tv   = current_tv;

	monitor->seqnum     = monbuff.seqnum;
        monitor->flags      = CONSISTENT;
        
        switch (mtype) {

	case SYSMON:
		if(*cis_name)
			snprintf(host->hostinfo.name, CIS_HOSTNAMELEN, "%s.%s", cis_name, uname.nodename);
		else
			snprintf(host->hostinfo.name, CIS_HOSTNAMELEN, "%s", uname.nodename);
		host->defprio            = defprio;
		host->hostinfo.totalram  = totalram;
                host->hostinfo.totalswap = totalswap;
		init_history(&host->ctx_swtch_history, monbuff.time);
		init_history(&host->swpin_history, monbuff.time);
		init_history(&host->swpout_history, monbuff.time);
		for (i = 0; i < 4; i++)
			init_history(&host->disk_read_history[i], monbuff.time);
		for (i = 0; i < 4; i++)
			init_history(&host->disk_write_history[i], monbuff.time);
		break;
		
        case PROCMON:
                if (host->proc_ht) {
			clear_proclist(host);
			break;
		}
		host->proc_ht = hash_create(PROC_HT_LENGTH,
					    OFFSET(process_info, data.pid),
					    sizeof (int), 0);
		if (!host->proc_ht){
			free(monitor);
			return -ENOMEM;
		}
		host->slsum = 0;
		break;

        case SOCKMON:
		if (host->sock_ht) {
			clear_socklist(host);
			break;
		}
		host->sock_ht = hash_create(SOCK_HT_LENGTH,
					    OFFSET(socket_info, data),
					    SOCKINFO_HDRLEN,
					    OFFSET(cis_sockinfo, sport));
		if (!host->sock_ht) {
			free(monitor);
			return -ENOMEM;
		}
		break;
                
        case NETMON:

                clear_netdevlist(host);
		break;

	case LMMON:
		
		newtab = realloc(host->lmstab, nsensors * sizeof(struct cis_lmsinfo));
		if (!newtab) {
			free(monitor);
			return -ENOMEM;
		}
		host->lmstab = newtab;
		host->lmstab_len = nsensors;
		memset(host->lmstab, 0, nsensors * sizeof(struct cis_lmsinfo));
	}

	host->hostinfo.addr  = host->addrlist[0];
	host->hostinfo.caddr = myaddr;

	memcpy(&host->uname, &uname, sizeof (struct utsname));
	host->hostinfo.ct_per_sec = ct_per_sec;

        return 1;
    next:
        return -E2BIG;
}

int add_monitor(struct host_info *host, struct sockaddr_in *mon_addr, int mtype)
{
        struct in_addr addr = mon_addr->sin_addr;
        unsigned short port = ntohs(mon_addr->sin_port);

        int new_host = FALSE;
	struct monitor_info *monitor = NULL;

	if (host)
		monitor = host->monitor[mtype];

	if (!monitor && !(monitor = calloc(1, sizeof (struct monitor_info)))) {
		syslog(LOG_WARNING, "not enough memory for monitor");
		return -ENOMEM;
	}

	monitor->port   = port;
        monitor->seqnum = monbuff.seqnum-1;
        monitor->flags  = 0;

	if (!host) {
		if (!(host = calloc(1, sizeof (struct host_info)))) {
                        syslog(LOG_WARNING, "not enough memory for monitor");
			return -ENOMEM;
		}
		new_host = TRUE;
	}

	if (!host->monitor[mtype])
		host->monitor[mtype] = monitor;
        
	if (new_host) {
		host->hostinfo.addr = addr;
		
		host->next = host_list;
		host_list  = host;
	}

        return 1;
}

void monitor_dispatch(void)
{
	struct in_addr addr;
	unsigned short port;
	char tag;
	short end;
	int mtype;
	struct host_info *host;
	struct monitor_info *monitor = NULL;
	unsigned long msg_time;
	struct sockaddr_in mon_addr = {AF_INET};
	socklen_t addrlen;
	struct cis_procinfo proc;
	struct cis_sockinfo sock;
	struct cis_netdevinfo dev;
	int i;

next:
	addrlen = sizeof(mon_addr);
	msglen = recvfrom(monitor_socket, &monbuff, sizeof(struct monitor_msg),
			  MSG_NOSIGNAL, &mon_addr, &addrlen);
	if (msglen == -1)
		return;

	addr = mon_addr.sin_addr;
	port = ntohs(mon_addr.sin_port);
	mon_addr.sin_family = AF_INET;

#ifndef DEBUG
	if (port >= 1024)          /* Authorization check. Only root may bind */
		goto next;         /* ports below 1024 */
#endif
        
        gettimeofday(&current_tv, NULL);
	msg_time = monbuff.time;

	host = find_host(addr);
	if (!host && !accept_new_hosts)
		goto system_error;

	if (!host || !host->monitor[port - CISD_PORT])
		if (add_monitor(host, &mon_addr, port - CISD_PORT) != 1)
			goto next;

	monitor = host->monitor[port - CISD_PORT];
	
	/* Wait for init message */
	if (!(monitor->flags&CONSISTENT) && !(monbuff.flags&MSG_INIT)) {
		reply.code = -EAGAIN;
		goto send_reply;
	}

	/* Some messages are lost. Ask for full message */
	if (++(monitor->seqnum) != monbuff.seqnum /*|| !monbuff.seqnum*/) {
		reply.code = -EFAULT;
		goto send_reply;
	}

	buffptr = (char *) monbuff.data;

nexttag:
        get_object(tag);
        get_object(end);

	mtype = monitor_type(tag);
	if (!host->monitor[mtype])
		if (add_monitor(host, &mon_addr, mtype) != 1)
			goto next;
	
	if (is_init_msg(tag)) {
		reply.code = parse_init_msg(host, tag);
		goto send_reply;
	}
	monitor = host->monitor[mtype];

	/* Messages from monitors */
	if (monitor->port != port)
		goto system_error;

        if (tag != PROCMON_NEW &&
	    tag != PROCMON_CHANGES_ASYNC) {
                monitor->local_time = msg_time;
		monitor->time_tv    = current_tv;
	}

        switch (tag) {
                unsigned int  pid;
		unsigned char id;
                
                /************************
                 * Messages from cismon *
                 ************************/

	case MONITOR_EXIT:

		free(host->monitor[MONITOR]);
                host->monitor[MONITOR] = NULL;
		break;

		/************************
                 * Messages from sysmon *
                 ************************/

	case SYSMON_EXIT:

                free(host->monitor[SYSMON]);
                host->monitor[SYSMON] = NULL;
		break;

	case SYSMON_FULL:
	case SYSMON_CHANGES:
		{
			struct cis_hostinfo *hi = &host->hostinfo;
			struct cis_hostinfo oldhi;
			
			monitor->flags |= CONSISTENT;

			oldhi = *hi;

			if (tag == SYSMON_FULL) {
				get(hi, HOSTINFO_LEN);
			} else
				sys_change(hi);

			if (oldhi.idle != hi->idle)
				host->idle_at = msg_time;
			if (oldhi.ctx_swtch)
				history_add(&host->ctx_swtch_history,
					    monitor->local_time,
					    hi->ctx_swtch - oldhi.ctx_swtch);

			if (oldhi.swpin)
				history_add(&host->swpin_history,
					    monitor->local_time,
					    hi->swpin - oldhi.swpin);

			if (oldhi.swpout)
				history_add(&host->swpout_history,
					    monitor->local_time,
					    hi->swpout - oldhi.swpout);

			for (i = 0; i < 4; i++)
				if (oldhi.disk_read[i])
					history_add(&host->disk_read_history[i],
						    monitor->local_time,
						    hi->disk_read[i] - oldhi.disk_read[i]);
			for (i = 0; i < 4; i++)
				if (oldhi.disk_write[i])
					history_add(&host->disk_write_history[i],
						    monitor->local_time,
						    hi->disk_write[i] - oldhi.disk_write[i]);
		}
		break;

                /*************************
                 * Messages from procmon *
                 *************************/
                
        case PROCMON_EXIT:

                clear_proclist(host);
                free(host->monitor[PROCMON]);
                host->monitor[PROCMON] = NULL;
                break;
                
        case PROCMON_FULL:         /* The information about all processes */

		{
			struct process_info *process, *p;

			monitor->flags |= CONSISTENT;
			
			if (monbuff.flags & MSG_FIRST)
				for (p = host->proclist; p; p = p->next)
					p->flags &= ~CONSISTENT;

			while (bufflen < end) {
				get(&proc, PROCINFO_LEN);
				proc_test(host, &proc);
			}

			if (!(monbuff.flags & MSG_LAST))
				break;

			/*
			 * Free all terminated processes.
			 */
			p = host->proclist;
			while (p) {
				process = p;
				p = p->next;
				if (process->flags&CONSISTENT) continue;

				proc_remove(host, process);
			}
		}
                break;
                
        case PROCMON_NEW:   /* New process */

                get(&proc, PROCINFO_LEN);
                proc_test(host, &proc);
                break;
                
        case PROCMON_CHANGES:
        case PROCMON_CHANGES_ASYNC:

		while (bufflen < end) {
			get_object(pid);
			proc_change(host, pid);
		}
		
		break;

                /*************************
                 * Messages from sockmon *
                 *************************/
                
        case SOCKMON_EXIT:

                clear_socklist(host);
                free(host->monitor[SOCKMON]);
                host->monitor[SOCKMON] = NULL;
                break;
                
        case SOCKMON_FULL:    /* The information about all sockets */
		{
			struct socket_info *s;
			
			if (monbuff.flags & MSG_FIRST)
				for (s = host->socklist; s; s = s->next)
					s->flags &= ~CONSISTENT;

			while (bufflen < end) {
				get(&sock, SOCKINFO_LEN);
				sock_test(host, &sock);
			}

			if (!(monbuff.flags & MSG_LAST))
				break;
			/*
			 * Free all closed sockets.
			 */
			clear_sockets(host, msg_time);

			for (s = host->socklist; s; s = s->next) {
				if (s->flags&CONSISTENT) continue;
				sock_remove(host, s);
			}
		}
                break;

        case SOCKMON_NEW:     /* New sockets */
                
                get(&sock, SOCKINFO_LEN);
                sock_test(host, &sock);
                break;

        case SOCKMON_CHANGES:

		while (bufflen < end) {
			get(&sock, SOCKINFO_HDRLEN);
			sock_change(host, &sock);
		}

		clear_sockets(host, msg_time);

                break;

                /************************
                 * Messages from netmon *
                 ************************/

        case NETMON_EXIT:         /* Netmon terminated */

                clear_netdevlist(host);
                free(host->monitor[NETMON]);
                host->monitor[NETMON] = NULL;
                break;
                
        case NETMON_FULL:         /* The information about all devices */
		{
			struct netdev_info *d;

			if (monbuff.flags & MSG_FIRST) {
				for (d = host->netdevlist; d; d = d->next)
					d->flags &= ~CONSISTENT;
			}

			while (bufflen < end) {
				get(&dev, NETDEVINFO_LEN);
				netdev_test(host, &dev);
			}

			if (!(monbuff.flags & MSG_LAST))
				break;

			/*
			 * Clear status for stopped devices.
			 */
			for (d = host->netdevlist; d; d = d->next) {
				if (d->flags&CONSISTENT) continue;
				d->data.status = 0;
			}
		}
		break;

        case NETMON_NEW:          /* New devices */

                get(&dev, NETDEVINFO_LEN);
                netdev_test(host, &dev);
                break;

        case NETMON_CHANGES:

		while (bufflen < end) {
			get_object(id);
			netdev_change(host, id);
		}
		break;

                /***********************
                 * Messages from lmmon *
                 ***********************/

	case LMMON_EXIT:         /* lmmon terminated */

		free(host->lmstab);
		host->lmstab = NULL;
		host->lmstab_len = 0;
		free(host->monitor[LMMON]);
                host->monitor[LMMON] = NULL;
                break;
                
        case LMMON_FULL:         /* The information about all devices */

		while (bufflen < end) {
			get_object(id);
			if (id >= host->lmstab_len)
				continue;
			get_object(host->lmstab[(int) id]);
		}
		break;

	case LMMON_CHANGES:

		while (bufflen < end) {
			get_object(id);
			if (id >= host->lmstab_len)
				continue;
			get_object(host->lmstab[(int) id].value);
		}
		break;
                
        default:
                goto next;
        }
        
        goto nexttag;

    send_reply:
        reply.seqnum = monitor->seqnum;
	sendto(monitor_socket, &reply, sizeof(reply), SOCKFLAGS, &mon_addr, addrlen);
	goto next;

    system_error:
	{
		int return_code = -EPERM;
		
		sendto(monitor_socket, &return_code, sizeof(return_code),
		       SOCKFLAGS, &mon_addr, addrlen);
	}
        goto next;
}

void cleanup (int sig)
{
        (void) pmap_unset(prognum, CIS_VER);
        
//        syslog(LOG_INFO, "terminated");
        closelog();
        exit(0);
}

int main (int argc,char *argv[])
{
        struct timeval t;
        
	register SVCXPRT *transp_tcp;
	int sock;
	struct sockaddr_in addr;
        int flags;
	fd_set rfds;
	int err;
	char myhostname[MAXHOSTNAMELEN];
	struct hostent *hent;

	get_myaddress(&srv_addr);
	if (gethostname(myhostname, MAXHOSTNAMELEN) == -1) {
		syslog(LOG_ERR, "cannot get my hostname");
		exit(1);
	}

	myaddr = srv_addr.sin_addr;

	if (myaddr.s_addr == 0x0100007f) {
		if (!(hent = gethostbyname(myhostname))) {
			syslog(LOG_ERR, "cannot get my address");
			exit(1);
		}
		myaddr = *(struct in_addr *) hent->h_addr;
	}
	
#ifndef DEBUG
        if (getuid()) {
                fprintf(stderr, "You don't have permission to run cis server.\n");
                exit (1);
        }

        openlog("cis", LOG_CONS, LOG_DAEMON);

        if (!getuid()) {
                /*
                 * Look at the portmapper whether there is a cis daemon already registered.
                 */

		if (pmap_getport(&srv_addr, CIS_PROG, CIS_VER, IPPROTO_TCP)) {
			syslog(LOG_ERR, "cis daemon is already registered");
                        exit(1);
		}
	}
#endif

	setlocale(LC_ALL,"");
	setlocale (LC_NUMERIC, "C");

	/*
         * Parse command line for parameters.
	 */
	parse_args(argc, argv);
	reply.interval = interval;

	parse_config();
	/*
	 * We need to unregister rpc service before terminating ...
	 */
	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGPIPE, SIG_IGN);

	/*
	 * Initialize random number generator for authentication.
	 */
	gettimeofday(&t, NULL);
        srand((unsigned int) t.tv_usec);

	monitor_socket = socket(PF_INET, SOCK_DGRAM, 0);
	if (monitor_socket == -1) {
		syslog(LOG_ERR, "cannot create UDP server socket");
		exit(1);
        }

	addr.sin_family = AF_INET;
	addr.sin_port = htons(CISD_PORT);
        addr.sin_addr.s_addr = htonl (INADDR_ANY);
	err = bind (monitor_socket, (struct sockaddr *) &addr, sizeof (addr));
	if (err < 0) {
		syslog(LOG_ERR, "cannot bind UDP server socket");
		exit(1);
        }
	/*
	 * Enable nonblocking mode on socket.
	 */
	flags = fcntl(monitor_socket, F_GETFL, 0);
	if (flags == -1) {
		syslog(LOG_ERR, "cannot get UDP server socket flags");
		exit(1);
	}

	if (fcntl(monitor_socket, F_SETFL, (flags|FNDELAY)) == -1) {
		syslog(LOG_ERR, "cannot set UDP server socket flags");
		exit(1);
	};


#ifdef DEBUG
        (void) pmap_unset(CIS_PROG, CIS_VER);
#endif

        /*
         * Make a server point and register it with rpcbinder.
         */
	sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock == -1) {
		syslog(LOG_ERR, "cannot create TCP server socket");
		exit(1);
	}


	addr.sin_family = AF_INET;
        addr.sin_port = htons(CISD_PORT-1);
        addr.sin_addr.s_addr = htonl (INADDR_ANY);
        err = bind (sock, (struct sockaddr *) &addr, sizeof (addr));
        if (err < 0) {
                syslog(LOG_ERR, "cannot bind TCP server socket");
                exit(1);
        }
	
        transp_tcp = svctcp_create(sock, 0, 0);
        if (transp_tcp == NULL) {
                syslog(LOG_ERR, "cannot create tcp service");
                exit(1);
        }

        if (!svc_register(transp_tcp, prognum, CIS_VER, dispatcher, IPPROTO_TCP)) {
                syslog(LOG_ERR, "cannot register tcp service with the rpcbinder");
                exit(1);
        }

#ifndef DEBUG
        /*
         * It's time to do the magic to become a daemon.
         */
        daemonize();
#endif

        syslog(LOG_INFO, "started");

        for(;;) {

	        rfds = svc_fdset;
                FD_SET(monitor_socket, &rfds);
                
                switch (select (FD_SETSIZE, &rfds, NULL, NULL, NULL)) {

                case -1:
                        continue;

                default:
			if (FD_ISSET(monitor_socket, &rfds))
				monitor_dispatch();

                        FD_CLR(monitor_socket, &rfds);
                        svc_getreqset (&rfds);
                }
	}
}
