/*
 * Cluster Information Service record file processing library.
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>

#include "cis.h"
#include "cis_rec.h"

#undef TRUE
#define TRUE 1

#undef FALSE
#define FALSE 0

#define SIZE(str, el)    sizeof (((struct str *) NULL)->el)
#define ELEMENT(str, el) OFFSET(str,el), SIZE(str,el)

struct description cis_hostinfo_desc[] = {
        {0, 0, sizeof(struct cis_hostinfo)},
        {ELEMENT(cis_hostinfo, procnum),       1},
        {ELEMENT(cis_hostinfo, loads[0]),      1},
        {ELEMENT(cis_hostinfo, loads[1]),      1},
        {ELEMENT(cis_hostinfo, loads[2]),      1},
        {ELEMENT(cis_hostinfo, totalram),      1},
        {ELEMENT(cis_hostinfo, freeram),       1},
        {ELEMENT(cis_hostinfo, sharedram),     1},
        {ELEMENT(cis_hostinfo, bufferram),     1},
        {ELEMENT(cis_hostinfo, cachedram),     1},
        {ELEMENT(cis_hostinfo, totalswap),     1},
        {ELEMENT(cis_hostinfo, freeswap),      1},
        {ELEMENT(cis_hostinfo, ctx_swtch),     1},
        {ELEMENT(cis_hostinfo, swpin),         1},
        {ELEMENT(cis_hostinfo, swpout),        1},
	{ELEMENT(cis_hostinfo, disk_read[0]),  1},
	{ELEMENT(cis_hostinfo, disk_read[1]),  1},
	{ELEMENT(cis_hostinfo, disk_read[2]),  1},
	{ELEMENT(cis_hostinfo, disk_read[3]),  1},
	{ELEMENT(cis_hostinfo, disk_write[0]), 1},
	{ELEMENT(cis_hostinfo, disk_write[1]), 1},
	{ELEMENT(cis_hostinfo, disk_write[2]), 1},
	{ELEMENT(cis_hostinfo, disk_write[3]), 1},
        {ELEMENT(cis_hostinfo, status),        1},
        {ELEMENT(cis_hostinfo, CPU_available), 1},
        {ELEMENT(cis_hostinfo, reliability),   1},
        {ELEMENT(cis_hostinfo, performance),   1},
};

#define cis_hostinfo_desclen (sizeof (cis_hostinfo_desc)/sizeof (struct description))

struct description cis_procinfo_desc[] = {
        {ELEMENT(cis_procinfo, pid),        PROCINFO_LEN},
        {ELEMENT(cis_procinfo, ppid),       SIZE(cis_procinfo, ppid)},
        {ELEMENT(cis_procinfo, cmd),        SIZE(cis_procinfo, cmd)},
        {ELEMENT(cis_procinfo, priority),   1},
        {ELEMENT(cis_procinfo, utime),      1},
        {ELEMENT(cis_procinfo, stime),      1},
        {ELEMENT(cis_procinfo, rss),        1},
        {ELEMENT(cis_procinfo, vm),         1},
        {ELEMENT(cis_procinfo, pCPU),       1},
        {ELEMENT(cis_procinfo, majflt),     1},
        {ELEMENT(cis_procinfo, disk_read),  1},
        {ELEMENT(cis_procinfo, disk_write), 1},
};

#define cis_procinfo_desclen (sizeof (cis_procinfo_desc)/sizeof (struct description))

struct description cis_sockinfo_desc[] = {
        {0, OFFSET(cis_sockinfo ,uid),    SOCKINFO_LEN},
        {ELEMENT(cis_sockinfo ,sent),      1},
        {ELEMENT(cis_sockinfo ,rcvd),      1},
};

#define cis_sockinfo_desclen (sizeof (cis_sockinfo_desc)/sizeof (struct description))

struct description cis_netdevinfo_desc[] = {
        {ELEMENT(cis_netdevinfo, name),       NETDEVINFO_LEN},
        {ELEMENT(cis_netdevinfo ,status),     1},
        {ELEMENT(cis_netdevinfo ,receive),    1},
        {ELEMENT(cis_netdevinfo ,transmit),   1},
        {ELEMENT(cis_netdevinfo ,collisions), 1},
};

#define cis_netdevinfo_desclen (sizeof (cis_netdevinfo_desc)/sizeof (struct description))

#define LAST_1 (1 << (sizeof (int)*8 - 1))

static int proc_cmp(const void *l, const void *r)
{
        struct cis_procinfo *left  = (struct cis_procinfo *) l;
        struct cis_procinfo *right = (struct cis_procinfo *) r;

        return (left->pid - right->pid);
}

static int sock_cmp(const void *l, const void *r)
{
        struct cis_sockinfo *left  = (struct cis_sockinfo *) l;
        struct cis_sockinfo *right = (struct cis_sockinfo *) r;
        int res;
        
        if ((res = left->saddr.s_addr - right->saddr.s_addr))
                return (res);
        if ((res = left->type - right->type))
                return (res);
        if ((res = left->sport - right->sport))
                return (res);
        if ((res = left->daddr.s_addr - right->daddr.s_addr))
                return (res);
        if ((res = left->dport - right->dport))
                return (res);

        return (0);
}

static int netdev_cmp(const void *l, const void *r)
{
        struct cis_netdevinfo *left  = (struct cis_netdevinfo *) l;
        struct cis_netdevinfo *right = (struct cis_netdevinfo *) r;

        return (strncmp(left->name, right->name, CIS_DEVNAMELEN));
}

/*
 * Record file manipulation functions.
 */
// internal routines.

static int read_record_header(FILE *file, struct record_info *rinfo)
{
        struct record_info r;
	
	if (file != stdin && (r.offset = ftell(file)) < 0)
		return -1;

	if (!fread(&r.type, 1, 1, file) || !fread(&r.length, sizeof (short), 1, file)) {
		if (file != stdin)
			fseek(file, r.offset, SEEK_SET);
		return -1;
	}

	rinfo->type   = r.type;
        rinfo->length = r.length;
        rinfo->offset = r.offset;

        return 0;
}

static int read_record_data(FILE *file, struct record_info *rinfo)
{
        char *buf = NULL;
        
        if (!rinfo->length)
		goto end;

	buf = malloc(rinfo->length);
	if (!buf)
		return -1;

	if (fread(buf, rinfo->length, 1, file) != 1) {
		free(buf);
		return -1;
	}
end:

	rinfo->data = buf;
	return 0;
}

static inline int objcmp(char *a, char *b, int len)
{
        for (a += len-1, b += len-1; len; len--, a--, b--)
                if (*a != *b)
                        return len;
        return 0;
}

static inline int changed(void *oldp, void *newp, struct description *desc, int desclen)
{
        struct description *d;

        if ((!oldp || !newp) && (oldp || newp))
                return TRUE;

        for (d = desc+1; d < desc+desclen; d++)
                if (objcmp((char *) oldp + d->offset,
                           (char *) newp + d->offset,
                           d->length))
                        return TRUE;
        return FALSE;
}


/*
 * The implementation is done for Linux endian.
 */
static inline int add_changes(char **buffp, int *bufflen, void *oldp, void *newp, struct description *desc, int desclen)
{
        int len = *bufflen;
        char *buff = *buffp;
        char *tmp;
        int flags, bits;
        int flagsp, nflags, nbits;
        int fragment;

        struct description *d;

        if (!oldp) { /* New object, save whole object */
                if (!(tmp = realloc(buff, len + desc->fragment + 1)))
                        goto error;
                buff = tmp;
                *(buff+len) = 1; /* code */
                memcpy(buff+len+1, (char *) newp, desc->fragment);
                len += desc->fragment + 1;
                goto end;
        }
                
        if (!newp) { /* Destroyed object, save its identifier */
                if (!(tmp = realloc(buff, len + desc->length + 1)))
                        goto error;
                buff = tmp;
                *(buff+len) = 2; /* code */
                memcpy(buff+len+1, (char *) oldp + desc->offset, desc->length);
                len += desc->length + 1;
                goto end;
        }

        flagsp = len;
        flags  = 0;
        nflags = 2;
        /* Store the identifier */
        if (!(tmp = realloc(buff, len + desc->length + 1)))
                goto error;
        buff = tmp;
        memcpy(buff+len+1, (char *) oldp + desc->offset, desc->length);
        len += desc->length+1;

        for (d = desc+1; d < desc+desclen; d++) {

                fragment = objcmp((char *) oldp + d->offset,
                                  (char *) newp + d->offset,
                                  d->length);
                nbits = 1;
                if (!fragment)
                        bits = 0;
                else if (fragment <= d->fragment)
                        bits = 1, nbits = 2, fragment = d->fragment;
                else if (fragment <= 2*d->fragment && 2*d->fragment < d->length)
                        bits = 3, nbits = 3, fragment = 2*d->fragment;
                else
                        bits = 7, nbits = 3, fragment = d->length;
                        
                if (nflags + nbits > 8) {
                        if (!(tmp = realloc(buff, len+1)))
                                goto error;
                        buff = tmp;

                        flags >>= 8 - nflags;        /* Allocate space for bits */
                        flags |= bits << nflags;     /* Add bits to the flags */
                        buff[flagsp] = flags;        /* Store lower byte of flags */
                        nflags += nbits - 8;         /* Compute length of remainder */
                        flags >>= nflags;
                        flagsp = len++;              /* Get the pointer to the next flags field */
                } else {
                        flags >>= nbits;             /* Allocate space for bits */
                        flags |= bits << (8 - nbits);/* Add bits */
                        nflags += nbits;
                }

                if (!bits) /* no change */
                        continue;

                if (!(tmp = realloc(buff, len+fragment)))
                        goto error;
                buff = tmp;
                memcpy(buff + len, (char *) newp + d->offset, fragment);
                len += fragment;
        }

        if (nflags)
                buff[flagsp] = flags >> (8-nflags);

    end:
        *buffp   = buff;
        *bufflen = len;

        return TRUE;
    error:
        return FALSE;
}

void apply_changes(void *str, char **changes, struct description *desc, int desclen)
{
        char *c = *changes + 1;
        unsigned short flags = ((short)1 << 15) | **changes;
        struct description *d;

        if (flags & 1) {
                memcpy((char *) str, c, desc->fragment);
                c += desc->fragment;
                goto end;
        }
        if (flags & 2) {
                memcpy((char *) str + desc->offset, c, desc->length);
                c += desc->length;
                goto end;
        }

        c += desc->length; /* Skip the identifier, str should point to */
                           /* correct object */
        flags >>= 2;       /* Skip new and terminated flag */

        for (d = desc+1; d < desc+desclen; d++, flags >>= 1) {

                if (!(flags >> 8))
                        flags = (1 << 15) | *(c++);

                if (!(flags & 1)) /* no change */
                        continue;

                flags >>= 1;
                if (!(flags >> 8))
                        flags = (1 << 15) | *(c++);

                if (!(flags & 1)) {
                        memcpy((char *) str + d->offset, c, d->fragment);
                        c += d->fragment;
                        continue;
                }

                flags >>= 1;
                if (!(flags >> 8))
                        flags = (1 << 15) | *(c++);
                        
                if (!(flags & 1)) {
                        memcpy((char *) str + d->offset, c, 2*d->fragment);
                        c += 2*d->fragment;
                        continue;
                }

                memcpy((char *) str + d->offset, c, d->length);
                c += d->length;
        }

    end:
        *changes = c;
}

static struct cis_header header = {CIS_RECORD_VERSION, {0,0}, 0., 0, NULL};

/*
 * Library routines.
 */
int cisSaveHeader(FILE *file, struct timeval *time, int n, struct cis_addr *hosts, float interval)
{
	int err, len;
	
	if (!n || n > 255 || !hosts)
		return -1;
	
	if (file != stdout)
		ftruncate(fileno(file),0);

	if (time)
		header.time = *time;
	else
		gettimeofday(&header.time, NULL);


	header.time.tv_usec /= 10000;
	header.time.tv_usec *= 10000;
	
	header.nhosts = n;
        header.interval = interval;
	
	err = fwrite(&header, CIS_HDRLEN, 1, file);
	if (err != 1)
		return -1;

	len = n*sizeof (struct cis_addr);
	header.hosts = realloc(header.hosts, len);
	if (!header.hosts)
		return -1;
	memcpy(header.hosts, hosts, len);

	err = fwrite(header.hosts, sizeof(struct cis_addr), n, file);
	if (err != n)
		return -1;

	return 0;
}

int cisReadHeader(FILE *file, struct cis_header **hdr)
{
	int err, len;

	if (file != stdin && fseek(file, 0L, SEEK_SET) < 0)
		return -1;

	err = fread(&header, CIS_HDRLEN, 1, file);
	if (err != 1)
		return -1;
	
	len = header.nhosts*sizeof (struct cis_addr);
	header.hosts = realloc(header.hosts, len);
	if (!header.hosts)
		return -1;
	err = fread(header.hosts, sizeof(struct cis_addr), header.nhosts, file);
	if (err != header.nhosts)
		return -1;

	*hdr = &header;
	return 0;
}

int cisHostinfoPrepareChanges(struct cis_hostinfo *oldhi, struct cis_hostinfo *newhi, char **changes)
{
        char *buff = NULL;
        int len = 0;
        int desclen = sizeof (cis_hostinfo_desc)/sizeof (struct description);

        if (!oldhi && !newhi) {
                *changes = NULL;
                return 0;
        }

        if (changed(oldhi, newhi, cis_hostinfo_desc, cis_hostinfo_desclen))
                if (!add_changes(&buff, &len, oldhi, newhi, cis_hostinfo_desc, desclen))
                        goto error;

        *changes = buff;
        
        return len;

    error:
        free(buff);
        return -1;
}

int cisHostinfoApplyChanges(struct cis_hostinfo *hi, char *changes, int chlen, struct cis_hostinfo **newhi)
{
        *newhi = calloc(1, sizeof (struct cis_hostinfo));

        if (!*newhi)
                return -1;

        **newhi = *hi;
        
        if (!changes)
                return 0;

        if (*changes & 2) {
                (*newhi)->status = HOST_NOT_AVAILABLE;
                return 1;
        }

        apply_changes(*newhi, &changes, cis_hostinfo_desc, cis_hostinfo_desclen);

        return 1;
}

int cisProclistPrepareChanges(struct cis_procinfo *oldtab, int nold, struct cis_procinfo *newtab, int nnew, char **changes)
{
        struct cis_procinfo *p, *q;
        char *buff = NULL;
        int len = 0;

        qsort(oldtab, nold, sizeof (struct cis_procinfo), proc_cmp);
        qsort(newtab, nnew, sizeof (struct cis_procinfo), proc_cmp);

        p = oldtab;
        q = newtab;

        while (p < oldtab+nold || q < newtab+nnew) {

                if (q < newtab+nnew && (p == oldtab+nold || p->pid > q->pid)) {
                        if (!add_changes(&buff, &len, NULL, q,      /* new process */
                                         cis_procinfo_desc, cis_procinfo_desclen))
                                goto error;
                        q++;
                        continue;
                }
                if (p < oldtab+nold && (q == newtab+nnew || p->pid < q->pid)) {
                        if (!add_changes(&buff, &len, p, NULL,    /* terminated process */
                                         cis_procinfo_desc, cis_procinfo_desclen))
                                goto error;
                        p++;
                        continue;
                }

                if (changed(p, q, cis_procinfo_desc, cis_procinfo_desclen))
                        add_changes(&buff, &len, p, q,
                                    cis_procinfo_desc, cis_procinfo_desclen);
                
                p++;
                q++;
        }
        *changes = buff;
        
        return len;

    error:
        free(buff);
        return -1;
}

int cisProclistApplyChanges(struct cis_procinfo *oldtab, int nold, char *changes, int chlen, struct cis_procinfo **newtab)
{
        struct cis_procinfo *p, *q, *tab = NULL;
        char *c;
        int len = 0;
        int pid;
        int flags;

        *newtab = NULL;

        qsort(oldtab, nold, sizeof (struct cis_procinfo), proc_cmp);

        p = oldtab;
        q = NULL;

        c = changes;
        while (c < changes + chlen) {
                flags = *c;
                if (flags & 1)
                        memcpy(&pid, c+1+cis_procinfo_desc->offset, sizeof (int));
                else
                        memcpy(&pid, c+1, sizeof (int));
                
                /*
                 * Copy all processes up to new/updated one. If the process is terminated
                 * skip it.
                 */
                while (p < oldtab + nold && p->pid < pid) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_procinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        q = tab+len;
                        *q = *p;
                        len++, p++;
                }
                if (!(flags & 1) && p == oldtab+nold)
                        goto error;

                if (flags & 2) {
                        if (p->pid != pid)
                                goto error;
                        c += sizeof(int)+1;
                        p++;
                        continue;
                }

                /*
                 * If the pid was changed, copy next process.
                 */
                if (!q || q->pid != pid) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_procinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        q = tab+len;
                        len++;
                        if (flags & 1) {
                                memset(q, 0, sizeof (struct cis_procinfo));
                                q->pid = pid;
                        }
                        else if (p < oldtab+nold)
                                memcpy(q, p++, sizeof (struct cis_procinfo));
                        else
                                goto error;
                }

                if (q->pid != pid)
                        goto error;

                apply_changes(q, &c, cis_procinfo_desc, cis_procinfo_desclen);
        }
        /*
         * Copy the rest of oldtab.
         */
        for (;p < oldtab + nold; p++) {
                tab = realloc(*newtab, (len+1)*sizeof (struct cis_procinfo));
                if (!tab)
                        goto error;
                *newtab = tab;
                q = tab+len;
                *q = *p;
                len++;
        }

        return len;

    error:
        free(tab);
        *newtab = NULL;
        return -1;
}

int cisSocklistPrepareChanges(struct cis_sockinfo *oldtab, int nold, struct cis_sockinfo *newtab, int nnew, char **changes)
{
        struct cis_sockinfo *s, *t;
        char *buff = NULL;
        int len = 0;

        qsort(oldtab, nold, sizeof (struct cis_sockinfo), sock_cmp);
        qsort(newtab, nnew, sizeof (struct cis_sockinfo), sock_cmp);

        s = oldtab;
        t = newtab;

        while (s < oldtab+nold || t < newtab+nnew) {

                if (t < newtab+nnew &&
                    (s == oldtab+nold || sock_cmp(s, t) > 0)) {
                        if (!add_changes(&buff, &len, NULL, t,      /* new socket */
                                         cis_sockinfo_desc, cis_sockinfo_desclen))
                                goto error;
                        t++;
                        continue;
                }
                if (s < oldtab+nold && (t == newtab+nnew || sock_cmp(s, t) < 0)) {
                        if (!add_changes(&buff, &len, s, NULL,    /* closed socket */
                                         cis_sockinfo_desc, cis_sockinfo_desclen))
                                goto error;
                        s++;
                        continue;
                }

                if (changed(s, t, cis_sockinfo_desc, cis_sockinfo_desclen))
                        add_changes(&buff, &len, s, t,
                                    cis_sockinfo_desc, cis_sockinfo_desclen);
                
                s++;
                t++;
        }
        *changes = buff;
        
        return len;

    error:
        free(buff);
        return -1;
}

int cisSocklistApplyChanges(struct cis_sockinfo *oldtab, int nold, char *changes, int chlen, struct cis_sockinfo **newtab)
{
        struct cis_sockinfo *s, *t, *tab = NULL, tmp;
        char *c;
        int len = 0;
        int flags;

        *newtab = NULL;

        qsort(oldtab, nold, sizeof (struct cis_sockinfo), sock_cmp);

        s = oldtab;
        t = NULL;

        c = changes;
        while (c < changes + chlen) {
                flags = *c;
                if (flags & 1)
                        memcpy(&tmp.saddr, c+1+cis_sockinfo_desc->offset, cis_sockinfo_desc->length);
                else
                        memcpy(&tmp.saddr, c+1, cis_sockinfo_desc->length);

                /*
                 * Copy all sockets up to new/updated one. If the socket is closed
                 * skip it.
                 */
                while (s < oldtab + nold && sock_cmp(s, &tmp) < 0) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_sockinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        t = tab+len;
                        *t = *s;
                        len++, s++;
                }
                if (!(flags & 1) && s == oldtab+nold)
                        goto error;

                if (flags & 2) {
                        if (sock_cmp(s, &tmp) != 0)
                                goto error;
                        c += cis_sockinfo_desc->length+1;
                        s++;
                        continue;
                }

                if (!t || sock_cmp(t, &tmp) != 0) {
                        /* Copy next socket.  */
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_sockinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        t = tab+len;
                        len++;
                        if (flags & 1) {
                                memset(t, 0, sizeof (struct cis_sockinfo));
                                memcpy(&t->saddr, &tmp.saddr, cis_sockinfo_desc->length);
                        }
                        else if (s < oldtab+nold)
                                memcpy(t, s++, sizeof (struct cis_sockinfo));
                        else
                                goto error;
                }

                if (sock_cmp(t, &tmp) != 0)
                        goto error;

                apply_changes(t, &c, cis_sockinfo_desc, cis_sockinfo_desclen);
        }
        /*
         * Copy the rest of oldtab.
         */
        for (;s < oldtab + nold; s++) {
                tab = realloc(*newtab, (len+1)*sizeof (struct cis_sockinfo));
                if (!tab)
                        goto error;
                *newtab = tab;
                t = tab+len;
                *t = *s;
                len++;
        }

        return len;

    error:
        free(tab);
        *newtab = NULL;
        return -1;
}

int cisNdevlistPrepareChanges(struct cis_netdevinfo *oldtab, int nold, struct cis_netdevinfo *newtab, int nnew, char **changes)
{
        struct cis_netdevinfo *nd, *od;
        char *buff = NULL;
        int len = 0;

        qsort(oldtab, nold, sizeof (struct cis_netdevinfo), netdev_cmp);
        qsort(newtab, nnew, sizeof (struct cis_netdevinfo), netdev_cmp);

        od = oldtab;
        nd = newtab;

        while (od < oldtab+nold || nd < newtab+nnew) {

                if (nd < newtab+nnew && (od == oldtab+nold || netdev_cmp(od->name, nd->name) > 0)) {
                        if (!add_changes(&buff, &len, NULL, nd,      /* new device */
                                         cis_netdevinfo_desc, cis_netdevinfo_desclen))
                                goto error;
                        nd++;
                        continue;
                }
                if (od < oldtab+nold && (nd == newtab+nnew || netdev_cmp(od->name, nd->name) < 0)) {
                        if (!add_changes(&buff, &len, od, NULL,      /* removed device */
                                         cis_netdevinfo_desc, cis_netdevinfo_desclen))
                                goto error;
                        od++;
                        continue;
                }

                if (changed(od, nd, cis_netdevinfo_desc, cis_netdevinfo_desclen))
                        add_changes(&buff, &len, od, nd,
                                    cis_netdevinfo_desc, cis_netdevinfo_desclen);

                od++;
                nd++;
        }
        *changes = buff;

        return len;

    error:
        free(buff);
        return -1;
}

int cisNdevlistApplyChanges(struct cis_netdevinfo *oldtab, int nold, char *changes, int chlen, struct cis_netdevinfo **newtab)
{
        struct cis_netdevinfo *od, *nd, *tab = NULL;
	char *c, name[CIS_DEVNAMELEN];
        int len = 0;
        int flags;

        *newtab = NULL;

        qsort(oldtab, nold, sizeof (struct cis_netdevinfo), netdev_cmp);

        od = oldtab;
        nd = NULL;

        c = changes;
        while (c < changes + chlen) {
                flags = *c;
                if (flags & 1)
			memcpy(name, c+1+cis_netdevinfo_desc->offset, CIS_DEVNAMELEN);
                else
                        memcpy(name, c+1, CIS_DEVNAMELEN);

                /*
                 * Copy all devices up to new/updated one. If the device is removed
                 * skip it.
                 */
                while (od < oldtab + nold && netdev_cmp(od->name, name) <0) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_netdevinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        nd = tab+len;
                        *nd = *od;
                        len++, od++;
                }
                if (!(flags & 1) && od == oldtab+nold)
                        goto error;

                if (flags & 2) {
                        if (netdev_cmp(od->name, name))
                                goto error;
			c += CIS_DEVNAMELEN+1;
                        od++;
                        continue;
                }

                /*
                 * If the name was changed, copy next device.
                 */
                if (!nd || netdev_cmp(nd->name, name)) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_netdevinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        nd = tab+len;
                        len++;
                        if (flags & 1) {
                                memset(nd, 0, sizeof (struct cis_netdevinfo));
				strncpy(nd->name, name, CIS_DEVNAMELEN);
                        }
                        else if (od < oldtab+nold)
                                memcpy(nd, od++, sizeof (struct cis_netdevinfo));
                        else
                                goto error;
                }

                if (netdev_cmp(nd->name, name))
                        goto error;

                apply_changes(nd, &c, cis_netdevinfo_desc, cis_netdevinfo_desclen);
        }
        /*
         * Copy the rest of oldtab.
         */
        for (;od < oldtab + nold; od++) {
                tab = realloc(*newtab, (len+1)*sizeof (struct cis_netdevinfo));
                if (!tab)
                        goto error;
                *newtab = tab;
                nd = tab+len;
                *nd = *od;
                len++;
        }

        return len;

    error:
        free(tab);
        *newtab = NULL;
        return -1;
}

int cisSaveRecord(FILE *file, int type, char *data, int block, int num, int save)
{
        short length;
        long offset;

	if (file != stdout && (offset = ftell(file)) < 0)
		return -1;

	if (num < 0)
		num = 0;

	length = num * save;

        if (!data)
                num = 0;

        if (!fwrite(&type,   sizeof (char),  1, file) ||
            !fwrite(&length, sizeof (short), 1, file))
                goto error;

        for (; num; num--, data += block)
                if (fwrite(data, 1, save, file) < save)
                        goto error;

        return 0;

error:
	if (file != stdout)
		ftruncate(fileno(file), offset);
        return -1;
}

int cisRecordInfo(FILE *file, struct record_info *rinfo)
{
        int ret;

        ret = read_record_header(file, rinfo);
        if (ret < 0)
                return ret;

        fseek(file, rinfo->offset, SEEK_SET);
        
        return ret;
}

int cisReadRecordHeader(FILE *file, struct record_info *rinfo)
{
	return(read_record_header(file, rinfo));
}

int cisReadRecordData(FILE *file, struct record_info *rinfo)
{
	return(read_record_data(file, rinfo));
}

int cisReadRecord(FILE *file, struct record_info *rinfo)
{
        int ret;

        ret = read_record_header(file, rinfo);
        if (ret < 0)
                return ret;

	return (read_record_data(file, rinfo));
}

int cisParseFile(FILE *file, struct record_info **rec, void callback(int))
{
        struct record_info *array = NULL, *new_array, tmp;
        long offset, total_length;
        int n = 0, i;
        int ret;
	struct stat fst;
	struct timeval time;
	struct cis_addr addr;

        if (fstat (fileno(file), &fst) == -1)
                return -1;

        total_length = fst.st_size;

        while (!feof(file)) {

                offset = ftell(file);

		ret = read_record_header(file, &tmp);
		if (ret < 0)
			return ret;

		if (tmp.type == CIS_REC_TIME) {
			read_record_data(file, &tmp);
			i = *(int *) tmp.data;
			time.tv_sec  = header.time.tv_sec + i/100;
			time.tv_usec = header.time.tv_usec + (i%100)*10000;
			time.tv_sec  += time.tv_usec % 1000000L;
			free(tmp.data);
			goto out;
		}
		if (tmp.type == CIS_REC_HOST) {
			read_record_data(file, &tmp);
			addr = header.hosts[(int) *(char*) tmp.data];
			free(tmp.data);
			goto out;
		}

		if (fseek(file, tmp.length, SEEK_CUR) == -1)
                        break;

		new_array = realloc(array, (n+1) * sizeof (struct record_info));
                if (!new_array) {
                        errno = ENOMEM;
                        goto error;
                }
                tmp.host = addr;
		tmp.time = time;

		array = new_array;
                memcpy(&array[n], &tmp, sizeof (struct record_info));
                n++;
	out:
                if (callback)
                        callback(((total_length - offset)*100)/total_length);
        } 

        *rec = array;
        return n;

    error:
        free(array);
        return -1;
}

int cisSaveBreak(FILE *file, int length)
{
	return cisSaveRecord(file, CIS_REC_BREAK, (char *) &length, sizeof (int), 1, sizeof (int));
}

int cisSaveTime(FILE *file, struct timeval time)
{
	int tmp;

	time.tv_usec /= 10000;
	time.tv_usec *= 10000;
	
	time.tv_sec -= header.time.tv_sec;
        tmp = time.tv_sec * 100 + (time.tv_usec - header.time.tv_usec)/10000;
	
        return cisSaveRecord(file, CIS_REC_TIME, (char *) &tmp, sizeof (int), 1, sizeof (int));
}

static int find_host(struct cis_hostinfo *hi)
{
	struct cis_addr addr = {hi->caddr, hi->addr};
	int i;

	for (i = 0; i < header.nhosts; i++)
		if (!memcmp(&header.hosts[i], &addr, sizeof(struct cis_addr)))
			return i;
	return -1;
}

int cisSaveHostID(FILE *file, struct cis_hostinfo *hi)
{
	int tmp = find_host(hi);
	unsigned char idx;
	
	if (tmp < 0)
		return -1;

	idx = tmp;

	return cisSaveRecord(file, CIS_REC_HOST, &idx, 1, 1, 1);
}

int cisSaveHostinfo(FILE *file, struct cis_hostinfo *data)
{
        return cisSaveRecord(file,
                             CIS_REC_SYSINFO,
                             (char *) data,
                             sizeof (struct cis_hostinfo),
                             1,
                             sizeof (struct cis_hostinfo));
}

int cisSaveHostinfoChanges(FILE *file, char *changes, int length)
{
        return cisSaveRecord(file,
			     CIS_REC_SYSINFO,
			     changes,
                             length,
                             1,
                             length);
}

int cisSaveProclist(FILE *file, struct cis_procinfo *data, int number)
{
        return cisSaveRecord(file,
                             CIS_REC_PROCESSES,
                             (char *) data,
                             sizeof (struct cis_procinfo),
                             number,
                             sizeof (struct cis_procinfo));
}

int cisSaveProclistChanges(FILE *file, char *changes, int length)
{
        return cisSaveRecord(file,
			     CIS_REC_PROCESSES,
			     changes,
                             length,
                             1,
                             length);
}

int cisSaveSocklist(FILE *file, struct cis_sockinfo *data, int number)
{
        return cisSaveRecord(file,
			     CIS_REC_SOCKETS,
                             (char *) data,
                             sizeof (struct cis_sockinfo),
                             number,
                             sizeof (struct cis_sockinfo));
}

int cisSaveSocklistChanges(FILE *file, char *changes, int length)
{
        return cisSaveRecord(file,
                             CIS_REC_SOCKETS,
                             changes,
                             length,
                             1,
                             length);
}

int cisSaveNdevlist(FILE *file, struct cis_netdevinfo *data, int number)
{
        return cisSaveRecord(file,
                             CIS_REC_IFACES,
                             (char *) data,
                             sizeof (struct cis_netdevinfo),
                             number,
                             sizeof (struct cis_netdevinfo));
}

int cisSaveNdevlistChanges(FILE *file, char *changes, int length)
{
        return cisSaveRecord(file,
			     CIS_REC_IFACES,
                             changes,
                             length,
                             1,
                             length);
}

int cisApplyChanges(int type, void *oldtab, int nold, char *changes, int chlen, void **newtab)
{
        switch (type) {

        case CIS_REC_SYSINFO:
                return (cisHostinfoApplyChanges((struct cis_hostinfo *) oldtab,
                                                changes, chlen,
                                                (struct cis_hostinfo **) newtab));

        case CIS_REC_PROCESSES:
                return (cisProclistApplyChanges((struct cis_procinfo *) oldtab, nold,
                                            changes, chlen,
                                            (struct cis_procinfo **) newtab));

        case CIS_REC_SOCKETS:
                return (cisSocklistApplyChanges((struct cis_sockinfo *) oldtab, nold,
                                            changes, chlen,
                                            (struct cis_sockinfo **) newtab));

        case CIS_REC_IFACES:
                return (cisNdevlistApplyChanges((struct cis_netdevinfo *) oldtab, nold,
                                              changes, chlen,
                                              (struct cis_netdevinfo **) newtab));
        default:
                return -1;
        }
}