/*
 * Cluster Information Service client library - client RPC calls to CIS server.
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _cis_clnt_h
#define _cis_clnt_h

#include <rpc/rpc.h>

#include "cis.h"

extern char *rel_types[];

enum {
        CIS_NOMEM = 200,
        CIS_NOCONN,
        CIS_NOHOST,
};


#define CIS_PROCNUM  1
#define CIS_LOADS    (1<<1)
#define CIS_RAM      (1<<2)
#define CIS_SWAP     (1<<3)
#define CIS_CPU      (1<<4)
#define CIS_IDLE     (1<<5)
#define CIS_CTXSWTCH (1<<6)
#define CIS_DISK     (1<<7)
#define CIS_ALL      (-1)


extern int cis_errno;

/* Client -> CIS server RPC calls */

/* Function connects to specific CIS server and returns RPC CLIENT handle */
CLIENT *cisConnect (char *hostname);

/* Close RPC handle */
void cisDisconnect (CLIENT *handle);

/* Set RPC timeout */
void cisTimeout    (CLIENT *handle,
                    struct timeval tout);

/*
 * Get cluster information (name).
 */
struct cis_info *cisInfo(CLIENT *handle);

/*
 * Get the info about host. Allocated structure cis_hostinfo is returned
 * If the host is not monitored or an error occured, NULL is returned and
 * cis_errno contains RPC error code.
 */
struct cis_hostinfo *cisHostinfo(CLIENT *handle,
				 struct in_addr addr,
				 char *perf_type,
				 float interval);

/*
 * Get the list of hosts monitored by CIS server with >= reliability.
 * An array is allocated dynamicaly and return code contains its length.
 * Dynamic statistics are computed for given interval.
 */
int cisHostlist(CLIENT *handle,
		struct cis_hostinfo **list,
		unsigned char reliability,
		char *perf_type,
		float interval);

/*
 * Update host information. Code identifies fields that should be updated.
 * Status of host is updated automatically. If code is equal to zero, only
 * status is updated. If it's -1, all fields will be updated. hosttab must
 * contain array of pointers to structures returned by cisHostInfo() call.
 * Return code is zero if call was successful or -1 if some error occured.
 * In that case cis_errno contains RPC error code.
 */
int cisUpdateHostsinfo(CLIENT *handle,
                       int code,
                       struct cis_hostinfo **hosttab,
                       int nhosts,
                       float interval);

/*
 * Update info for only one host.
 */
int cisUpdateHostinfo (CLIENT *handle,
                       int code,
                       struct cis_hostinfo *hinfo,
                       float interval);

/*
 * Get a list of processes. If uid is specified, only processes of given
 * user are returned. Array is allocated dynamicaly and return code contains
 * its length. If some error has occured, return code is less than zero and
 * contains RPC error code (rpc/clnt.h). The cis_procinfo structure is defined
 * in cis.h.
 */
int cisProclist (CLIENT *handle,
		 struct cis_procinfo **list,
		 struct in_addr addr,
		 unsigned short uid,
		 float interval);

/*
 * Get a list of sockets. It's similar to cisProcessList() function.
 * Send and receive rates (in bytes per second) are counted for specified
 * interval (in seconds). If it is not specified or it is longer than CIS
 * server's archive, the length of archive is used instead.
 */
int cisSocklist (CLIENT *handle,
		 struct cis_sockinfo **list,
		 struct in_addr addr,
		 unsigned short uid,
		 float interval);

/*
 * Get a list of network devices. It's similar to cisSocketList() function.
 */
int cisNdevlist (CLIENT *handle,
		 struct cis_netdevinfo **list,
		 struct in_addr addr,
		 float interval);

int cisLMsensors(CLIENT *handle, struct cis_lmsinfo **list, struct in_addr addr);
/* end of client calls */
#endif
