/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_merge: merges multiple record files into one.
 */

#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>
#include <sys/time.h>

#include "cis.h"
#include "cis_rec.h"

struct host_info {
        int flag;
	struct cis_addr addr;
        struct file_info *finfo;
	
	struct cis_hostinfo *hostinfo;
	struct cis_hostinfo *hostinfo_old;

        int nproc;
        struct cis_procinfo *proctab;

	int nproc_old;
        struct cis_procinfo *proctab_old;
        
        int nsock;
	struct cis_sockinfo *socktab;

	int nsock_old;
	struct cis_sockinfo *socktab_old;

        int ndev;
        struct cis_netdevinfo *devtab;

	int ndev_old;
        struct cis_netdevinfo *devtab_old;

	struct host_info *next;
	
} *hostlist = NULL;

int nhosts = 0;

int nfiles;

struct file_info {
        int available;
        int active;
        char *name;
        FILE *fp;
        struct record_info rec;
        int nhosts;
	struct host_info **hosts;
	struct cis_header hdr;
        struct timeval time;
	
} *filetab;

struct timeval current_time;
float interval;

int tvalcmp (struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

struct host_info *find_host (struct cis_addr addr)
{
	struct host_info *h;

	for (h = hostlist; h; h = h->next)
		if (!memcmp(&h->addr, &addr, sizeof(struct cis_addr)))
			return h;
	return NULL;
}

struct host_info *add_host (struct cis_addr addr)
{
	struct host_info *h;

	h = calloc (1, sizeof (struct host_info));
	if (!h) {
		printf ("not enough memory\n");
		exit (1);
	}

	h->addr = addr;
	h->flag = 1;

	h->next = hostlist;
	hostlist = h;
	nhosts++;

        return h;
}

void record_time(struct file_info *fi, struct record_info r, struct timeval *t)
{
	int i = *(int *) r.data;
	
	t->tv_sec  = fi->hdr.time.tv_sec + i/100;
	t->tv_usec = fi->hdr.time.tv_usec + (i%100)*10000;
	if (t->tv_usec >= 1000000L) {
		t->tv_sec   += t->tv_usec / 1000000L;
		t->tv_usec  %= 1000000L;
	}
}

void get_full_info (struct file_info *fi)
{
	struct record_info r;
	struct host_info *h;
	int i;

	for (i = 0; i < fi->hdr.nhosts; i++) {
		h = fi->hosts[i];
		h->hostinfo_old = h->hostinfo;
		h->proctab_old  = h->proctab;
		h->nproc_old    = h->nproc;
		h->socktab_old  = h->socktab;
		h->nsock_old    = h->nsock;
		h->devtab_old   = h->devtab;
		h->ndev_old     = h->ndev;
	}
	
	while (cisReadRecord (fi->fp, &r) > -1 && r.type != CIS_REC_BREAK) {

		switch (r.type) {

		case CIS_REC_HOST:
			h = fi->hosts[(int) *(char*) r.data];
			free (r.data);
			break;

		case CIS_REC_SYSINFO:
			h->hostinfo = (struct cis_hostinfo *) r.data;
			break;

		case CIS_REC_PROCESSES:
			h->proctab = (struct cis_procinfo *) r.data;
                        h->nproc = r.length / sizeof (struct cis_procinfo);
			break;

		case CIS_REC_SOCKETS:
			h->socktab = (struct cis_sockinfo *) r.data;
                        h->nsock = r.length / sizeof (struct cis_sockinfo);
			break;

		case CIS_REC_IFACES:
			h->devtab = (struct cis_netdevinfo *) r.data;
                        h->ndev = r.length / sizeof (struct cis_netdevinfo);
			break;
		}
	}

	current_time = fi->time;

	/*
	 * Get time record
	 */
	cisReadRecord (fi->fp, &r);
	record_time(fi, r, &fi->time);
	free(r.data);

	for (i = 0; i < fi->hdr.nhosts; i++)
		fi->hosts[i]->finfo = fi;

	fi->active = 1;
}


void disable_host (struct cis_addr addr)
{
        struct file_info *fi;
        struct host_info *h;

        for (fi = filetab; fi < filetab+nfiles; fi++)
		if ((h = find_host (addr)))
                        h->flag = 0;
}

struct file_info *next_record (void)
{
	struct file_info *fi, *oldest = NULL;

	for (fi = filetab; fi < filetab+nfiles; fi++) {
		if (!fi->available) continue;
		if (!oldest || tvalcmp (oldest->time, fi->time) > 0 ||
		    (tvalcmp (oldest->time, fi->time) == 0 && !fi->active))
			oldest = fi;
	}

        return oldest;
}

void rec_copy (FILE *f1, FILE *f2, int len)
{
        static char buff[1024];

        while (len) {
                if (len < 1024) {
                        fread  (buff, len, 1, f2);
                        fwrite (buff, len, 1, f1);
                        break;;
                }
                fread  (buff, 1024, 1, f2);
                fwrite (buff, 1024, 1, f1);
                len -= 1024;
        }
}

void rec_skip (FILE *f, int len)
{
        fseek (f, len, SEEK_CUR);
}

FILE *outfile;
int gap = 0;
struct cis_hostinfo hinfo;

void save_changes (struct file_info *fi)
{
	char *hchanges = NULL;
        char *pchanges = NULL;
        char *schanges = NULL;
        char *dchanges = NULL;
	int hchlen, pchlen, schlen, dchlen;
	int i, time_saved = 0;

	for (i = 0; i < fi->hdr.nhosts; i++) {
		struct host_info *h = fi->hosts[i];

		hchlen = cisHostinfoPrepareChanges (h->hostinfo_old, h->hostinfo, &hchanges);
		pchlen = cisProclistPrepareChanges (h->proctab_old, h->nproc_old, h->proctab, h->nproc, &pchanges);
		schlen = cisSocklistPrepareChanges (h->socktab_old, h->nsock_old, h->socktab, h->nsock, &schanges);
		dchlen = cisNdevlistPrepareChanges (h->devtab_old, h->ndev_old, h->devtab, h->ndev, &dchanges);

		if (h->hostinfo_old != h->hostinfo)
			free(h->hostinfo_old);
		if (h->proctab_old  != h->proctab)
			free(h->proctab_old);
		if (h->socktab_old  != h->socktab)
			free(h->socktab_old);
		if (h->devtab_old   != h->devtab)
			free(h->devtab_old);

		if (!(hchlen || pchlen || schlen || dchlen))
			continue;

		if (!time_saved) {
			cisSaveTime(outfile, current_time);
			time_saved = 1;
		}
		cisSaveHostID(outfile, h->hostinfo);

		if (hchlen) {
			cisSaveHostinfoChanges (outfile, hchanges, hchlen);
			free (hchanges);
		}
		if (pchlen) {
			cisSaveProclistChanges (outfile, pchanges, pchlen);
			free (pchanges);
		}
		if (schlen) {
			cisSaveSocklistChanges (outfile, schanges, schlen);
			free (schanges);
		}
		if (dchlen) {
			cisSaveNdevlistChanges (outfile, dchanges, dchlen);
			free (dchanges);
		}
	}
}

int main (int argc, char *argv[])
{
        struct file_info *fi;
        char **arg;
        struct cis_header *header;
	struct host_info *h;
	struct cis_addr *addrtab;
	struct record_info r;
	int i, len, err;
	int time_saved;

	gettimeofday(&current_time, NULL);
	interval = 1e100;
	
        if (argc < 4) {
                printf ("usage: cis_merge outputfile file1 file2 [file3] ...\n");
                exit (1);
        }

        if (!(outfile = fopen (argv[1], "w+"))) {
                printf ("cannot open file %s for writing.\n", argv[1]);
                exit (1);
        }
        
        nfiles  = argc - 2;
        filetab = calloc (nfiles, sizeof (struct file_info));

	memset (&hinfo, 0, sizeof (struct cis_hostinfo));
	hinfo.status = HOST_NOT_AVAILABLE;

	for (fi = filetab, arg = argv+2; fi < filetab + nfiles; fi++, arg++) {
                if (!(fi->fp = fopen (*arg, "r"))) {
			printf ("cannot open file %s for reading.\n", *arg);
			exit (1);
		}
                fi->name = *arg;

		if (cisReadHeader (fi->fp, &header) < 0 || header->version != CIS_RECORD_VERSION) {
			printf ("incorrect record file %s \n", *arg);
			exit (1);
		}

		fi->hdr = *header;
		fi->hdr.hosts = malloc(header->nhosts*sizeof(struct cis_addr));
		memcpy(fi->hdr.hosts, header->hosts, header->nhosts*sizeof(struct cis_addr));
		fi->hdr.nhosts = header->nhosts;
		fi->hosts = malloc(header->nhosts*sizeof(struct host_info *));

		for (i = 0; i < fi->hdr.nhosts; i++) {
			if (!(h = find_host (fi->hdr.hosts[i])))
				h = add_host (fi->hdr.hosts[i]);
			
			h->hostinfo = calloc(1, sizeof(struct cis_hostinfo));
			h->hostinfo->status = HOST_NOT_AVAILABLE;

			fi->hosts[i] = h;
		}

		if (tvalcmp (current_time, fi->hdr.time) > 0)
			current_time = fi->hdr.time;
		if (interval > fi->hdr.interval)
			interval = fi->hdr.interval;

		fi->time = fi->hdr.time;
		fi->available = 1;
		fi->active = 0;

//              fi->available = cisRecordInfo (fi->fp, &fi->rec) >= 0;
        }

	addrtab = malloc(nhosts*sizeof(struct cis_addr));
	for (i = 0, h = hostlist; i < nhosts && i < 256; i++, h = h->next)
		addrtab[i] = h->addr;

	if (i == 256) {
		printf("reached maximal number of hosts in record file (256)");
		exit(1);
	}
	
	if (cisSaveHeader(outfile, &current_time, nhosts, addrtab, interval) == -1) {
		printf("error while saving record file header\n");
		exit(1);
	}

	for (fi = filetab; fi < filetab + nfiles; fi++)
		if (tvalcmp (current_time, fi->hdr.time) == 0)
			break;

	/*
	 * Copy full info from oldest record file.
	 */
	get_full_info(fi);

	for (h = hostlist; h; h = h->next) {
		cisSaveHostID(outfile, h->hostinfo);
		cisSaveHostinfo (outfile, h->hostinfo);

		if (h->hostinfo->status != HOST_AVAILABLE)
			continue;

		cisSaveProclist (outfile, h->proctab, h->nproc);
		cisSaveSocklist (outfile, h->socktab, h->nsock);
		cisSaveNdevlist (outfile, h->devtab,  h->ndev);
	}
	cisSaveBreak(outfile, 0);

	while ((fi = next_record ())) {

		if (!fi->active) {
			get_full_info(fi);
			save_changes(fi);
			continue;
		}

		time_saved = 0;
		while ((err = cisRecordInfo (fi->fp, &r)) > -1 && r.type != CIS_REC_TIME) {
			struct cis_hostinfo *hi;
			struct cis_procinfo *ptab;
			struct cis_sockinfo *stab;
			struct cis_netdevinfo *dtab;

			if (r.type == CIS_REC_HOST || h)
				cisReadRecord (fi->fp, &r);
			else {
				fseek (fi->fp, r.length + CIS_RECHDRLEN, SEEK_CUR);
				continue;
			}

			switch (r.type) {

			case CIS_REC_HOST:
				h = fi->hosts[(int) *(char*) r.data];
				if (h->finfo != fi)
					h = NULL;
				break;

			case CIS_REC_SYSINFO:
				cisHostinfoApplyChanges (h->hostinfo, r.data, r.length, &hi);
				if (h->hostinfo)
					free(h->hostinfo);
				h->hostinfo = hi;
				break;

			case CIS_REC_PROCESSES:
				len = cisProclistApplyChanges (h->proctab, h->nproc, r.data, r.length, &ptab);
				if (h->proctab)
					free(h->proctab);
				h->proctab = ptab;
				h->nproc = len;
				break;

			case CIS_REC_SOCKETS:
				len = cisSocklistApplyChanges (h->socktab, h->nsock, r.data, r.length, &stab);
				if (h->socktab)
					free(h->socktab);
				h->socktab = stab;
				h->nsock = len;
				break;

			case CIS_REC_IFACES:
				len = cisNdevlistApplyChanges (h->devtab, h->ndev, r.data, r.length, &dtab);
				if (h->devtab)
					free(h->devtab);
				h->devtab = dtab;
				h->ndev = len;
				break;
			}

			if (!h) {
				free (r.data);
				continue;
			}
			
			if (!time_saved) {
				cisSaveTime(outfile, fi->time);
				time_saved = 1;
			}
			if (r.type == CIS_REC_HOST) {
				cisSaveHostID(outfile, h->hostinfo);
				continue;
			}

			cisSaveRecord(outfile, r.type, r.data, r.length, 1, r.length);
			free (r.data);
		}

		/* Get next time record */
		err = cisReadRecord (fi->fp, &r);
		if (err == -1) {
			fi->available = 0;

			for (i = 0; i < nfiles; i++)
				if (filetab[i].available)
					break;

			if (i == nfiles)
				continue;

			for (i = 0; i < fi->hdr.nhosts; i++) {
				h = fi->hosts[i];

				h->hostinfo_old = h->hostinfo;
				h->proctab_old  = h->proctab;
				h->nproc_old    = h->nproc;
				h->socktab_old  = h->socktab;
				h->nsock_old    = h->nsock;
				h->devtab_old   = h->devtab;
				h->ndev_old     = h->ndev;

				if (h->finfo == fi) {

					if (h->hostinfo_old) {
						h->hostinfo     = malloc(sizeof(struct cis_hostinfo));
						*h->hostinfo    = *h->hostinfo_old;
						h->hostinfo->status = HOST_NOT_AVAILABLE;
						h->hostinfo->CPU_available = 0;
					}
					h->proctab      = NULL;
					h->nproc        = 0;
					h->socktab      = NULL;
					h->nsock        = 0;
					h->devtab       = NULL;
					h->ndev         = 0;
				}
			}
			save_changes(fi);

			continue;
		}
		record_time(fi, r, &fi->time);
		free (r.data);
	}

        printf ("Ok.\n");
        
        exit (0);
}