/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_cut: extracts records for one or more hosts and time interval.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <netdb.h>
#include <locale.h>
#include <time.h>
#include <string.h>
#include <sys/time.h>

#include "cis.h"
//#include "cis_clnt.h"
#include "cis_rec.h"

extern int h_errno;

extern int save_size[];
extern int struct_size[];

struct host_info {
	struct cis_addr addr;
	char *name;

	struct cis_hostinfo *hostinfo;

        int nproc;
        struct cis_procinfo *proctab;
        
        int nsock;
        struct cis_sockinfo *socktab;

        int ndev;
        struct cis_netdevinfo *devtab;

	struct host_info *next;
	
} *hostlist = NULL;

unsigned long start_time = 0;
unsigned long end_time = 0;
struct record_info *table;
int table_length;
char *filename;
FILE *input;
FILE *output;

struct cis_header input_header;

struct timeval current_time;
struct timeval old_time;

int nhosts;
struct cis_addr *addrtab;

int all_hosts = 1;

struct host_info *findhostbyaddr (struct cis_addr addr)
{
        struct host_info *h;

	for (h = hostlist; h; h = h->next)
		if (!memcmp(&h->addr, &addr, sizeof(struct cis_addr)))
			return h;
	return NULL;
}

struct host_info *findhostbyname (char *name)
{
        struct host_info *h;

	for (h = hostlist; h; h = h->next)
		if (!strcmp(h->name, name))
			return h;
	return NULL;
}

int tvalcmp (struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

unsigned long get_time (char *buff)
{
        struct tm *tmp, t = {0, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX};
        char *c = buff;
        time_t sec;
        
        sec = time(NULL);
        tmp = localtime (&sec);

        c = strptime (c, "%R", &t);
        if (!c) goto next;

        if (*c == ':')
                c = strptime (c, ":%S", &t);
        if (!c) goto next;

        c = strptime (c, " %d/%m", &t);
        if (!c) goto next;

        if (*c == '/')
                c = strptime (c, "/%Y", &t);

    next:
        if (t.tm_min == INT_MAX || t.tm_hour == INT_MAX) {
                printf ("cannot get time from %s.\n", buff);
                return 0;
        }
        if (t.tm_mday == INT_MAX) t.tm_mday = tmp->tm_mday;
        if (t.tm_mon  == INT_MAX) t.tm_mon  = tmp->tm_mon;
        if (t.tm_year == INT_MAX) t.tm_year = tmp->tm_year;

        return mktime (&t);
}

struct host_info *addhost(char *name, struct cis_addr *addr)
{
	struct host_info *h, *host;

	h = calloc (1, sizeof (struct host_info));
	if (!h) {
		printf ("not enough memory\n");
		return NULL;
	}

	h->name = name;
	if (addr)
		h->addr = *addr;

	nhosts++;

	if (!hostlist) {
		hostlist = h;
		return h;
	}

	for (host = hostlist; host->next; host = host->next)
		;

	host->next = h;
	
	return(h);
}

void parse_args (int argc, char **argv)
{
        struct host_info *h;
        int c, i;
	int option_index = 0;
        static struct option long_options[] = {
		{"infile", 1, 0, 'i'},
                {"outfile", 1, 0, 'o'},
                {"from", 1, 0, 'f'},
                {"to", 1, 0, 't'},
                {0, 0, 0, 0},
        };

        input = stdin;
        output = stdout;

	while ((c = getopt_long (argc, argv, "i:o:f:t:",
				 long_options, &option_index)) != -1) {

		switch (c) {
		case 'i':
			if (!(input = fopen (optarg, "r"))) {
				fprintf (stderr, "cannot open input file %s\n", optarg);
				exit (1);
			}
			break;
		case 'o':
			if (!(output = fopen (optarg, "w+"))) {
				fprintf (stderr, "cannot open output file %s\n", optarg);
				exit (1);
			}
			break;
		case 'f':
			start_time = get_time (optarg);
			break;

		case 't':
			end_time = get_time (optarg);
			break;

		case ':':
		case '?':
			goto arg_error;
		}
	}

	for (i = optind; i < argc; i++) {
		h = addhost(argv[i], NULL);
		h->hostinfo = calloc (1, sizeof(struct cis_hostinfo));
		if (!h->hostinfo) {
			fprintf (stderr, "not enough memory\n");
			exit (1);
                }
                snprintf(h->hostinfo->name, CIS_HOSTNAMELEN, "%s", h->name);

		all_hosts = 0;
	}
	return;

arg_error:
	printf ("usage: cis_cut [-i infile] [-o outfile] [--from time] [--to time] [host1] [host2] ...\n");
	printf ("                            time format is \"HH:MM[:SS] [DD/MM[/YYYY]]\"\n");
	exit(1);
}

void record_time(struct record_info r, struct timeval *t)
{
	int i = *(int *) r.data;

	old_time = *t;
	
	t->tv_sec  = input_header.time.tv_sec + i/100;
	t->tv_usec = input_header.time.tv_usec + (i%100)*10000;
	if (t->tv_usec >= 1000000L) {
		t->tv_sec   += t->tv_usec / 1000000L;
		t->tv_usec  %= 1000000L;
	}
}

void rec_copy (FILE *f1, FILE *f2, int len)
{
        static char buff[1024];

        len += CIS_RECHDRLEN;
        
        while (len) {
		if (len < 1024) {
                        fread  (buff, len, 1, f2);
                        fwrite (buff, len, 1, f1);
			break;
                }
		fread  (buff, 1024, 1, f2);
                fwrite (buff, 1024, 1, f1);
                len -= 1024;
        }
}

void skip (FILE *f, int len)
{
	static char buff[1024];

	if (f != stdin) {
		fseek (f, len, SEEK_CUR);
		return;
	}

	while (len) {
		if (len < 1024) {
			fread  (buff, len, 1, f);
			break;
                }
		fread  (buff, 1024, 1, f);
		len -= 1024;
	}
}

int main (int argc, char *argv[])
{
	struct cis_hostinfo *hi;
        struct record_info r;
	struct host_info *h;
	struct cis_header *header;
	int ret, i, idx, len;

	setlocale (LC_ALL, "");
        setlocale (LC_NUMERIC, "C");

        parse_args (argc, argv);


	if (cisReadHeader (input, &header) < 0 || header->version != CIS_RECORD_VERSION) {
                fprintf(stderr, "incorrect record file\n");
                exit (1);
        }

	if (!hostlist)
		for (i = 0; i < header->nhosts; i++)
			addhost(NULL, &header->hosts[i]);

	current_time = header->time;

	input_header = *header;
	input_header.hosts = malloc(header->nhosts*sizeof(struct cis_addr));
	memcpy(input_header.hosts, header->hosts, header->nhosts*sizeof(struct cis_addr));

	/*
	 * Get full info.
	 */
	while (cisReadRecordHeader (input, &r) > -1) {
		if (r.type == CIS_REC_BREAK)
			break;

		if (h || r.type == CIS_REC_HOST || (!all_hosts && r.type == CIS_REC_SYSINFO))
			cisReadRecordData (input, &r);
		else {
			skip (input, r.length);
			continue;
		}

		switch (r.type) {

		case CIS_REC_HOST:
			idx = (int) *(char*) r.data;
			if (all_hosts)
				h = findhostbyaddr(header->hosts[idx]);
			free (r.data);
			break;

		case CIS_REC_SYSINFO:
			hi = (struct cis_hostinfo *) r.data;
			if (!all_hosts)
				h = findhostbyname(hi->name);
			else
				h->name = strdup(hi->name);

			if (h) {
				h->hostinfo = hi;
				h->addr.server_addr = hi->caddr;
				h->addr.node_addr   = hi->addr;
			}
			break;

		case CIS_REC_PROCESSES:
			h->proctab = (struct cis_procinfo *) r.data;
                        h->nproc = r.length / sizeof (struct cis_procinfo);
			break;

		case CIS_REC_SOCKETS:
			h->socktab = (struct cis_sockinfo *) r.data;
                        h->nsock = r.length / sizeof (struct cis_sockinfo);
			break;

		case CIS_REC_IFACES:
			h->devtab = (struct cis_netdevinfo *) r.data;
                        h->ndev = r.length / sizeof (struct cis_netdevinfo);
			break;
		}
	}
	if (r.type == CIS_REC_BREAK)
		skip(input, r.length);

	cisReadRecord(input, &r);
	record_time (r, &current_time);
	free (r.data);
	
	while (current_time.tv_sec <= start_time && cisReadRecordHeader (input, &r) > -1) {
		struct cis_hostinfo *hi;
		struct cis_procinfo *ptab;
		struct cis_sockinfo *stab;
		struct cis_netdevinfo *dtab;

		if (h || r.type == CIS_REC_HOST || r.type == CIS_REC_TIME)
			cisReadRecordData (input, &r);
		else {
			skip(input, r.length);
			continue;
		}

		switch (r.type) {

		case CIS_REC_TIME:
			record_time (r, &current_time);
			goto next1;

		case CIS_REC_HOST:
			h = findhostbyaddr(input_header.hosts[(int) *(char*) r.data]);
			break;

		case CIS_REC_SYSINFO:
			cisHostinfoApplyChanges (h->hostinfo, r.data, r.length, &hi);
			*h->hostinfo = *hi;
			free(hi);
			break;

		case CIS_REC_PROCESSES:
			len = cisProclistApplyChanges (h->proctab, h->nproc, r.data, r.length, &ptab);
			free(h->proctab);
			h->proctab = ptab;
			h->nproc = len;
                        break;

		case CIS_REC_SOCKETS:
                        len = cisSocklistApplyChanges (h->socktab, h->nsock, r.data, r.length, &stab);
        		free(h->socktab);
			h->socktab = stab;
			h->nsock = len;
			break;

                case CIS_REC_IFACES:
			len = cisNdevlistApplyChanges (h->devtab, h->ndev, r.data, r.length, &dtab);
        		free(h->devtab);
			h->devtab = dtab;
			h->ndev = len;
                        break;
		}
	next1:
		free (r.data);
	}

	if (!ret) {
		fprintf(stderr, "no records in specified interval\n");
		exit (1);
	}

	if (hostlist)
		for (h = hostlist; h->next; h = h->next)
			if (!h->next->addr.server_addr.s_addr) {
				h->next = h->next->next;
				nhosts--;
			}
	if (!hostlist->addr.server_addr.s_addr) {
		hostlist = hostlist->next;
		nhosts--;
	}

	if (!hostlist) {
		fprintf(stderr, "no hosts\n");
		exit (1);
	}
	
	addrtab = malloc(nhosts*sizeof(struct cis_addr));
	for (i = 0, h = hostlist; i < nhosts; i++, h = h->next)
		addrtab[i] = h->addr;

	if (cisSaveHeader(output, &old_time, nhosts, addrtab, header->interval) == -1) {
		fprintf(stderr, "error while saving record file header\n");
		exit(1);
	}

	for (h = hostlist; h; h = h->next) {
		cisSaveHostID(output, h->hostinfo);
		cisSaveHostinfo (output, h->hostinfo);

		if (h->hostinfo->status != HOST_AVAILABLE)
			continue;

		cisSaveProclist (output, h->proctab, h->nproc);
		cisSaveSocklist (output, h->socktab, h->nsock);
		cisSaveNdevlist (output, h->devtab,  h->ndev);
	}
	cisSaveBreak(output, 0);
	fflush(output);
	cisSaveTime(output, current_time);

	while (cisReadRecordHeader (input, &r) > -1) {
		if (h || r.type == CIS_REC_HOST || r.type == CIS_REC_TIME)
			cisReadRecordData (input, &r);

		if (r.type == CIS_REC_HOST) {
			h = findhostbyaddr(input_header.hosts[(int) *(char*) r.data]);
			if (h)
				cisSaveHostID(output, h->hostinfo);
			goto next2;
		}

		if (r.type == CIS_REC_TIME) {
			record_time (r, &current_time);
			if (end_time && current_time.tv_sec > end_time)
				break;
			cisSaveTime(output, current_time);
			goto next2;
		}

		if (!h) {
			skip(input, r.length);
			continue;
		}
		
		cisSaveRecord(output, r.type, r.data, r.length, 1, r.length);
	next2:
		free (r.data);
	        fflush(output);
	}

	exit (0);
}