/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * System calls common for all CIS daemons.
 */

#include <sys/types.h>
#include <sys/resource.h>

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>

#include <netdb.h>
#include <netinet/in.h>

#include "cis.h"

void daemonize(void)
{
        int fd;
        pid_t pid;
        struct rlimit rl;
        
        /*
         * Clear umask
         */
        umask(0);
        
        /*
         * Get the maximum number of files we can open.
         */
        if (getrlimit(RLIMIT_NOFILE, &rl) < 0) {
                syslog(LOG_ERR, "cannot get file limit");
                exit(1);
        }
        
        /*
         * Flush all file buffers.
         * Choose a reasonable limit if the limit is currently infinite.
         */
        if (rl.rlim_max == RLIM_INFINITY)
                rl.rlim_max = 1024;
        for (fd = 0; fd < rl.rlim_max; fd++)
                fdatasync(fd);

        errno = 0;  /* set from syncing of invalid fd */


        /*
         * Become a session leader to lose our controlling terminal.
         */
        if ((pid = fork()) < 0) {
                syslog(LOG_ERR, "cannot fork");
                exit(1);
        } else if (pid != 0)   /* parent */
                exit(0);

        /*
         * Change the current directory to the root of the file system tree
         * so other file systems can be unmounted while we're running.
         */
        if (chdir("/") < 0) {
                syslog(LOG_ERR, "cannot chdir to /: %s", strerror(errno));
                exit (1);
        }
        
        /*
         * Attach file descriptorc 0,1 and 2 to /dev/null.
         */
        close(0);
        close(1);
        close(2);
        if ((fd = open("/dev/null",O_RDWR)) != 0) {
                if (fd < 0)
                        syslog(LOG_ERR, "error opening /dev/null: %s", strerror(errno));
                else
                        syslog(LOG_ERR, "open /dev/null: expected fd 0, got %d", fd);
                exit (1);
        }
        if ((fd = dup (0)) != 1) {
                if (fd < 0)
                        syslog(LOG_ERR, "dup failed: %s", strerror(errno));
                else
                        syslog(LOG_ERR, "dup(0) expected fd 1, got %d", fd);
                exit (1);
        }
        if ((fd = dup (0)) != 2) {
                if (fd < 0)
                        syslog(LOG_ERR, "dup failed: %s", strerror(errno));
                else
                        syslog(LOG_ERR, "dup(0) expected fd 2, got %d", fd);
                exit (1);
        }
}

int setfdnoblock(int fdesc)
{
        int flags;
        
        /*
         * Enable nonblocking mode on fd.
         */
        flags = fcntl(fdesc, F_GETFL, 0);
        if (flags != -1)
                return (fcntl(fdesc, F_SETFL, (flags|O_NONBLOCK)));
        return (-1);
}

int setfdblock(int fdesc)
{
        int flags;
        
        /*
         * Disable nonblocking mode on fd.
         */
        flags = fcntl(fdesc, F_GETFL, 0);
        if (flags != -1)
                return (fcntl(fdesc, F_SETFL, (flags&(~O_NONBLOCK))));
        return (-1);
}

