/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * System monitoring functions.
 */

#include "cis.h"
#include "mon_common.h"

static FILE *sysinfofile;
static FILE *loadavg;
static FILE *meminfo;
static FILE *uptime;
static FILE *statfile;

static int mode = 0;

static void sys_init (void)
{
	struct stat st;

	/*
	 * Initialize input.
	 */
	if (stat("/proc/cis_sysinfo", &st) == -1)
		mode |= PROC_MODE;
	
	if (mode&PROC_MODE) {
		loadavg     = proc_open("loadavg");
		uptime      = proc_open("uptime");
		meminfo     = proc_open("meminfo");
		statfile    = proc_open("stat");
	} else {
		sysinfofile = proc_open("cis_sysinfo");
	}
}

static void sys_send_init (void)
{
	return (send_init(SYSMON_INIT));
}

static void sys_send_exit (void)
{
	return (send_exit(SYSMON_EXIT));
}

#define add_change(el) \
	if (hi->el != oldhi->el) {                 \
	*(chptr++) = OFFSET(cis_hostinfo, el);     \
	*(int *) chptr = hi->el - oldhi->el;       \
	chptr += sizeof(int);                      \
	oldhi->el = hi->el;                        \
	}

static inline void put_changes(struct cis_hostinfo *hi, struct cis_hostinfo *oldhi)
{
	unsigned char chbuff[sizeof(struct cis_hostinfo)];
	unsigned char *chptr = chbuff;

	if (hi->procnum   != oldhi->procnum)     add_change(procnum);
	if (hi->loads[0]  != oldhi->loads[0])    add_change(loads[0]);
	if (hi->loads[1]  != oldhi->loads[1])    add_change(loads[1]);
	if (hi->loads[2]  != oldhi->loads[2])    add_change(loads[2]);
	if (hi->totalswap != oldhi->totalswap)   add_change(totalswap);
	if (hi->freeswap  != oldhi->freeswap)    add_change(freeswap);
	if (hi->freeram   != oldhi->freeram)     add_change(freeram);
	if (hi->sharedram != oldhi->sharedram)   add_change(sharedram);
	if (hi->bufferram != oldhi->bufferram)   add_change(bufferram);
	if (hi->cachedram != oldhi->cachedram)   add_change(cachedram);

	if (hi->idle      != oldhi->idle)        add_change(idle);
        if (hi->ctx_swtch != oldhi->ctx_swtch)   add_change(ctx_swtch);
	if (chptr == chbuff)
		return;
	
	*(chptr++) = 0xff;
	
	put(chbuff, chptr - chbuff);
}

static void sys_check(void)
{
	static char *line;
	static int linelen;

	static struct cis_hostinfo hi, oldhi;
	float loads[3];

	/*
	 * Get the host system information.
	 */
	if (!(mode&PROC_MODE)) {
		sysinfofile = freopen("/proc/cis_sysinfo","r", sysinfofile);
		reset_time();
		rewind(sysinfofile);
		fread(&hi, HOSTINFO_LEN, 1, sysinfofile);
	} else {
		float idle;
		
		loadavg = freopen("/proc/loadavg", "r", loadavg);
		fscanf(loadavg, "%f %f %f %*d/%hd",
		       &loads[0], &loads[1], &loads[2], &hi.procnum);
		hi.loads[0] = loads[0]*100;
		hi.loads[1] = loads[1]*100;
		hi.loads[2] = loads[2]*100;
		
		uptime = freopen("/proc/uptime", "r", uptime);
		fscanf(uptime, "%*f %f", &idle);
		hi.idle = idle * ct_per_sec;

		statfile = freopen("/proc/stat", "r", uptime);
		do {
			getline(&line, &linelen, statfile);
		} while (strncmp (line, "ctxt", 4));
		sscanf(line, "%*s %ld", &hi.ctx_swtch);

		meminfo = freopen("/proc/meminfo", "r", meminfo);
		getline(&line, &linelen, meminfo); /* Field names */
		getline(&line, &linelen, meminfo); /* Memory */
		sscanf(line, "%*s %*d %*d %ld %ld %ld %ld",
		       &hi.freeram, &hi.sharedram, &hi.bufferram, &hi.cachedram);
		
		getline(&line, &linelen, meminfo); /* Swap */
		sscanf(line, "%*s %ld %*d %ld",
		       &hi.totalswap, &hi.freeswap);
		
		hi.freeram   >>= 10;
		hi.sharedram >>= 10;
		hi.bufferram >>= 10;
		hi.cachedram >>= 10;
		hi.totalswap >>= 10;
		hi.freeswap  >>= 10;
	}

	if (full_msg) {
		init_msg(SYSMON_FULL);
		put(&hi, HOSTINFO_LEN);
	} else {
		init_msg(SYSMON_CHANGES);
		put_changes(&hi, &oldhi);
	}

	if (full_msg)
		send_msg();

        oldhi = hi;
}

struct monitor_desc sysmon = {
	SYSMON,
	-1,
	sys_init,
	sys_check,
	0,
	sys_send_init,
	sys_send_exit,
};
