/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Process monitoring functions.
 */

#define DEF_PRIORITY  (20*ct_per_sec/100)

#include "cis.h"
#include "cis_mon.h"
#include "mon_common.h"

#include <linux/tasks.h>  /* for NR_TASKS */

#define HT_LENGTH  (NR_TASKS * 3 / 2)
#define hash(a)    (((a * 3) / 2) % HT_LENGTH)

#define DEF_PROCBUFF_LEN         32

static int  page_size;

struct process_info {
	struct process_info *prev, *next;
	unsigned char  flags;
	struct cis_procinfo data;
};

/* Process information */
static struct cis_procinfo *procbuff;
static int procbuff_len = DEF_PROCBUFF_LEN;
static struct process_info *proclist;
static struct process_info **hash_table;

static FILE *procfile;
static DIR *procdir;

static int mode;

static void proc_init(void)
{
	struct sockaddr_nl nladdr={AF_NETLINK,0,0,0xffffffff};
	struct stat st;

	/*
	 * Alloc memory for the hash table, the table of new processes and the
	 * table of changes.
	 */
	if (!full_mode) {
		hash_table = malloc(HT_LENGTH * sizeof (struct process_info *));
		if (hash_table == NULL) {
			syslog(LOG_ERR, "cannot allocate memory for the hash table.");
			exit(1);
		}
		
		memset(hash_table, 0, HT_LENGTH * sizeof (struct process_info *));
		proclist = NULL;
	}
	
	/*
	 * Initialize input.
	 */
	if (stat("/proc/cis_proclist", &st) != -1) {

		procbuff = malloc(procbuff_len * sizeof (struct cis_procinfo));
		if (!procbuff) {
			syslog(LOG_ERR, "not enough memory");
			exit(1);
		}

		procmon.fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_PROCMON);
		if (procmon.fd < 0) {
			syslog(LOG_ERR, "cannot create netlink socket");
		}
		
		nladdr.nl_pid = mypid;
		if (bind(procmon.fd, (struct sockaddr *) &nladdr, sizeof (nladdr)) < 0) {
			syslog(LOG_ERR, "cannot bind netlink socket");
			mode |= PROC_MODE;
		}

		procfile    = proc_open("cis_proclist");

	} else {
		mode |= PROC_MODE;
		procdir     = opendir("/proc");
		page_size   = sysconf(_SC_PAGE_SIZE);
	}
	if (procmon.fd > -1)
		setfdnoblock(procmon.fd);
}

static void proc_send_init (void)
{
	return (send_init(PROCMON_INIT));
}

static void proc_send_exit (void)
{
	return (send_exit(PROCMON_EXIT));
}


/*
 * Searching in the hash table. If the requested process was not found, a
 * pointer to the next free cell is returned.
 */
static inline struct process_info **ht_find(pid_t pid)
{
	static struct process_info **hp;

	hp = &hash_table[hash(pid)];

	while (*hp && (*hp)->data.pid != pid)

		if (++hp == hash_table + HT_LENGTH)
			hp = hash_table;

	return (hp);
}

static inline void ht_remove(struct process_info **hole)
{
	struct process_info **hp = hole;

	*(hole) = NULL;
	
	for(;;) {
		if (++hp == hash_table + HT_LENGTH)
			hp = hash_table;
		
		if (!(*hp)) break;
		
		if (!(*ht_find((*hp)->data.pid)))
			*hole = *hp, *hp = NULL, hole = hp;
	}
}

static struct process_info *register_process(struct cis_procinfo *process, struct process_info **hp)
{
	struct process_info *p;
	
	p = calloc(1, sizeof (struct process_info));
	if (!p)
		return (NULL);
	
	memcpy(&p->data, process, PROCINFO_LEN);

	*hp = p;

	list_insert(proclist, p);

	p->flags |= CONSISTENT;
	
	return (p);
}

#define add_change(el) \
	if (p->el != msg->el) {                           \
	*(chptr++) = OFFSET(cis_procinfo, el);     \
	*(int *) chptr = msg->el - p->el;          \
	chptr += sizeof(int);                      \
	p->el = msg->el;                           \
	}

#define add_element(el) \
	{                                                 \
	*(chptr++) = OFFSET(cis_procinfo, el);     \
	memcpy(chptr,  &msg->el, sizeof(msg->el)); \
	memcpy(&p->el, &msg->el, sizeof(msg->el)); \
	chptr += sizeof(msg->el);                  \
	}

static inline void put_changes(struct cis_procinfo *p, struct cis_procinfo *msg)
{
	char chbuff[sizeof(struct cis_procinfo)];
	char *chptr = chbuff;

	*(typeof(p->pid) *) chptr = p->pid;
	chptr += sizeof(p->pid);
	
	if (p->utime    != msg->utime)     add_change(utime);
	if (p->stime    != msg->stime)     add_change(stime);
	if (p->minflt   != msg->minflt)    add_change(minflt);
	if (p->majflt   != msg->majflt)    add_change(majflt);
	if (p->rd_bytes != msg->rd_bytes)  add_change(rd_bytes);
	if (p->wr_bytes != msg->wr_bytes)  add_change(wr_bytes);
	if (p->priority != msg->priority)  add_change(priority);
	if (p->rss      != msg->rss)       add_change(rss);
	if (p->vm       != msg->vm)        add_change(vm);
	if (p->ppid     != msg->ppid)      add_element(ppid);
	if (p->uid      != msg->uid)       add_element(uid);
	
	if (strcmp(p->cmd, msg->cmd))      add_element(cmd);
	
	if (chptr == chbuff+sizeof(p->pid))
		return;

	*(chptr++) = 0;

	put(chbuff, chptr - chbuff);
}

static inline void put_process(char tag, struct cis_procinfo *p)
{
	if ((bufflen + sizeof(p->pid) + 1 + PROCINFO_LEN) > MON_BUFFLEN)
		buffer_send();

	*(typeof(p->pid) *) buffptr = p->pid;
	buffptr += sizeof(p->pid);
	*(buffptr++) = tag;

	put(p, PROCINFO_LEN);
}

#define put_new(p)        put_process(1, p)
#define put_terminated(p) put_process(2, p)

/*
 * Get processes and prepare message to server.
 */
static void proc_read_kernel(void)
{
	struct cis_procinfo *proc;
	struct process_info *p, **hp;
	int nproc;
	
	if (!full_mode)
		for (p = proclist; p; p = p->next)
			p->flags &= ~CONSISTENT;
	
	procfile = freopen("/proc/cis_proclist","r", procfile);
	if (!procfile)
		return;
	
	rewind(procfile);
	
again:
	nproc = fread(procbuff, sizeof(struct cis_procinfo), procbuff_len, procfile);
	if (!nproc)
		return;

	for (proc = procbuff; proc < procbuff + nproc; proc++) {

		if (full_mode) {
			put(proc, PROCINFO_LEN);
			continue;
		}
		
		hp = ht_find(proc->pid);
		
		if (!(*hp)) {  /* New process */

			p = register_process(proc, hp);
			if (!p)
				continue;

			if (!full_msg)
				put_new(proc);

		} else {
			/*
			 * Process was found. Log the changes (if any).
			 */
			p = *hp;
			if (memcmp(&p->data, proc, PROCINFO_LEN)) {
				if (!full_msg)
					put_changes(&p->data, proc);
				else
					memcpy(&p->data, proc, PROCINFO_LEN);
			}
		}
		p->flags |= CONSISTENT;
		if (full_msg)
			put(&p->data, PROCINFO_LEN);
	}
	if (nproc == procbuff_len)
		goto again;

}

#define skip_one() {for (; *c != ' '; c++); c++;}

static inline int get_next (struct cis_procinfo *proc)
{
	static char name[128];
	static char *line = NULL;
	static int linelen = 0;
	struct dirent *ent;
	struct stat fst;
	char *cmd_start;
	register char *c;
	register int i;
	int cmd_len;
	FILE *f;
	
again:
	while ((ent = readdir(procdir)) && !(proc->pid = atoi(ent->d_name)))
		;
	
	if (!ent)
		return 0;
	
	sprintf(name, "/proc/%d/stat", proc->pid);
	
	f = fopen(name, "r");
	if (!f)
		goto again;
	
	getline(&line, &linelen, f);
	
	cmd_start = index(line, '(')+1;
	for (c = cmd_start+CMD_LENGTH+1; *c != ')'; c--)
		;
	cmd_len = c - cmd_start;
	
	strncpy(proc->cmd, cmd_start, cmd_len);
	proc->cmd[cmd_len] = 0;
	
	c += 2;
	skip_one();
	proc->ppid = atoi(c); skip_one();
	for (i = 0; i < 5; i++) skip_one();
	proc->minflt = atol(c); skip_one();
	skip_one();
	proc->majflt = atol(c); skip_one();
	skip_one();
	proc->utime = atol(c); skip_one();
	proc->stime = atol(c); skip_one();
	skip_one(); skip_one(); skip_one();
	proc->priority = atol(c); skip_one();
	skip_one(); skip_one();
	proc->start_time = atol(c); skip_one();
	proc->vm = atol(c); skip_one();
	proc->rss = atol(c);

	/* Per process IO statistics are currently not in proc */
	proc->rd_bytes = proc->wr_bytes = 0;
	
	proc->priority = ((20 - proc->priority) * DEF_PRIORITY - DEF_PRIORITY / 2) / 20 + 1;
	proc->rss *= page_size;
	
	fstat (fileno(f), &fst);
	proc->uid = fst.st_uid;
	
	fclose(f);
	
	return 1;
}

/*
 * Get processes and prepare message to server.
 */
static void proc_read_proc(void)
{
	struct cis_procinfo proc;
	struct process_info *p, **hp;

	if (!full_mode)
		for (p = proclist; p; p = p->next)
			p->flags &= ~CONSISTENT;

	rewinddir(procdir);

	while (get_next(&proc)) {

		if (full_mode) {
			put(&proc, PROCINFO_LEN);
			continue;
		}

		hp = ht_find(proc.pid);

		if (!(*hp)) {  /* New process */
			
			p = register_process(&proc, hp);
			if (!p)
				continue;
			
			if (!full_msg)
				put_new(&proc);
			
		} else {
			/*
			 * Process was found. Log the changes (if any).
			 */
			p = *hp;
			if (memcmp(&p->data, &proc, PROCINFO_LEN)) {
				if (!full_msg)
					put_changes(&p->data, &proc);
				else
					memcpy(&p->data, &proc, PROCINFO_LEN);
			}
		}
		p->flags |= CONSISTENT;
		if (full_msg)
			put(&p->data, PROCINFO_LEN);
	}
}

static void clear_terminated(void)
{
	struct process_info *p, *termproc, **hp;
	
	p = proclist;
	
	while (p) {
		if (!p->flags&CONSISTENT) {
			termproc = p;
			p = p->next;
			
			hp = ht_find(termproc->data.pid);
			if (!full_msg)
				put_terminated(&termproc->data);

			ht_remove(hp);

			list_remove(proclist, termproc);
			free(termproc);

		} else
			p = p->next;
	}
}

static void proc_check(void)
{
	init_msg(full_msg ? PROCMON_FULL : PROCMON_CHANGES);

	if (mode&PROC_MODE)
		proc_read_proc();
	else
		proc_read_kernel();

	if (!full_mode)
		clear_terminated();

	if (full_msg)
		send_msg();
}

static void proc_getmsg(void)
{
	struct cis_procinfo proc, *msg = &proc;
	struct process_info *p, **hp = NULL;
	int err;

next:
	err = read(procmon.fd, msg, sizeof(struct cis_procinfo));

	if (err != sizeof(struct cis_procinfo))
		return;

	p = NULL;
	if (!full_mode) {
		hp = ht_find(msg->pid);
		p = *hp;
	}
	switch ((int) msg->priv) {
		
	case OBJ_CREATE:  /* Started process */
		
		if (!full_mode && (*hp || !(p = register_process(msg, hp))))
			goto next;

		init_msg(PROCMON_NEW);
		put(msg, PROCINFO_LEN);
		break;
		
	case OBJ_CHANGE: /* New command executed - identity change */

		if (full_mode || !p)
			goto next;

		init_msg(PROCMON_CHANGES_ASYNC);
		put_changes(&p->data, msg);
		break;
		
	case OBJ_DESTROY: /* Terminated process */

		if (!full_mode && !p)
			goto next;

		init_msg(PROCMON_CHANGES_ASYNC);
		put_terminated(msg);
		
		if (!full_mode) {
			ht_remove(hp);
			list_remove(proclist, p);
			free(p);
		}
		break;
	}
	send_msg();
	goto next;
}

struct monitor_desc procmon = {
	PROCMON,
	-1,
	proc_init,
	proc_check,
	proc_getmsg,
	proc_send_init,
	proc_send_exit,
};

