/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Network interface monitor. Provides CIS server with the information about
 * status and activity of network interfaces.
 */

#include "cis.h"
#include "cis_mon.h"
#include "mon_common.h"

#ifndef DEF_ND_LENGTH
#define DEF_ND_LENGTH             16
#endif

struct netdev_info {
	struct netdev_info *prev, *next;
	unsigned char flags;
	struct cis_netdevinfo data;
};

/* Network device information */
static FILE *devfile;
static struct netdev_info *devlist;

static int mode;

static char *line = NULL;
static int  linelen = 0;

static void net_init(void)
{
	struct sockaddr_nl nladdr={AF_NETLINK,0,0,0xffffffff};
	struct stat st;

	devlist = NULL;
        
        /*
         * Initialize input.
         */
        if (stat("/proc/cis_netdevlist", &st) == -1)
                mode |= PROC_MODE;

        if (!(mode&PROC_MODE)) {
                
                netmon.fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_NETMON);
                if (netmon.fd < 0) {
                        syslog(LOG_ERR, "cannot create netlink socket (kernel not patched)");
                        exit(1);
                }
                
		nladdr.nl_pid = mypid;
                if (bind (netmon.fd, (struct sockaddr *) &nladdr, sizeof (nladdr)) < 0) {
                        syslog(LOG_ERR, "cannot bind netlink socket");
                        exit(1);
                }
                
                devfile = fopen("/proc/cis_netdevlist","r");
                if (!devfile) {
                        syslog(LOG_ERR, "cannot open /proc/cis_netdevlist (kernel not patched)");
                        exit(1);
                }
        } else {
                devfile = fopen("/proc/net/dev","r");
                if (!devfile) {
                        syslog(LOG_ERR, "cannot open /proc/net/dev");
                        exit(1);
                }
	}
}

static void net_send_init (void)
{
	return (send_init(NETMON_INIT));
}

static void net_send_exit (void)
{
	return (send_exit(NETMON_EXIT));
}


static struct netdev_info *register_device(struct cis_netdevinfo *msg)
{
        static char id = 1;
	struct netdev_info *dev;
        
	dev = calloc(1, sizeof (struct netdev_info));
	if (!dev)
		return (NULL);

	memcpy(&dev->data, msg, NETDEVINFO_LEN);

	list_insert(devlist, dev);

	dev->flags |= CONSISTENT;
	dev->data.id = id++;
	return (dev);
}

#define add_change(el) \
        if (d->el != msg->el) {                           \
               *(chptr++) = OFFSET(cis_netdevinfo, el);   \
               *(int *) chptr = msg->el - d->el;          \
               chptr += sizeof(int);                      \
               d->el = msg->el;                           \
        }

#define add_element(el) \
        {                                                 \
               *(chptr++) = OFFSET(cis_netdevinfo, el);   \
               memcpy(chptr,  &msg->el, sizeof(msg->el)); \
               memcpy(&d->el, &msg->el, sizeof(msg->el)); \
               chptr += sizeof(msg->el);                  \
        }

static void put_device(char tag, struct cis_netdevinfo *d)
{
        if ((bufflen + sizeof(d->id) + 1 + NETDEVINFO_LEN) > MON_BUFFLEN)
                buffer_send();

        *(typeof(d->id) *) buffptr = d->id;
        buffptr += sizeof(d->id);
        *(buffptr++) = tag;

        put(d, NETDEVINFO_LEN);
}

#define put_started(d)    put_device(1, d)
#define put_closed(d)     put_device(2, d)

static void put_changes(struct cis_netdevinfo *d, struct cis_netdevinfo *msg)
{
        char chbuff[sizeof(struct cis_netdevinfo)];
        char *chptr = chbuff;

        *(typeof(d->id) *) chptr = d->id;
        chptr += sizeof(d->id);

        if (d->tx_bytes   != msg->tx_bytes)    add_change(tx_bytes);
        if (d->rx_bytes   != msg->rx_bytes)    add_change(rx_bytes);
        if (d->collisions != msg->collisions)  add_change(collisions);
        if (d->status     != msg->status)      add_element(status);

        if (chptr == chbuff+sizeof(d->id))
                return;

        *(chptr++) = 0;
            
        if ((bufflen + (chptr - chbuff)) > MON_BUFFLEN)
                buffer_send();
        put(chbuff, chptr - chbuff);
}

static int get_next (struct cis_netdevinfo *dev)
{
        char *c;

        if (!(mode&PROC_MODE))
		return (fread(dev, sizeof(struct cis_netdevinfo), 1, devfile));

    again:
        if (getline(&line, &linelen, devfile) == -1)
                return 0;

        if (!strncmp(line, "    lo:", 7))
                goto again;

        c = index (line, ':');
        if (c)
                *c = ' ';

        sscanf(line, "%s "
               "%lu %*u %*u %*u %*u %*u %*u %*u "
               "%lu %*u %*u %*u %*u %lu %*u %*u ",
               dev->name, &dev->rx_bytes, &dev->tx_bytes, &dev->collisions);

	dev->status = 1;
        return 1;
}
/*
 * Read device information and prepare message.
 */
static void dev_read(void)
{
	static struct cis_netdevinfo dev;
	struct netdev_info *d;

	if (!full_mode)
		for (d = devlist; d; d = d->next)
			d->flags &= ~CONSISTENT;

	rewind(devfile);

        if (mode&PROC_MODE) {         /* Skip header */
		getline(&line, &linelen, devfile);
		getline(&line, &linelen, devfile);
	}
        
	while (get_next(&dev)) {

		for (d = devlist; d; d = d->next)
			if (!strncmp (d->data.name, dev.name, CIS_DEVNAMELEN))
				break;

		if (!d) {  /* New device */
                        if (!(d = register_device(&dev)))
				continue;

			if (!full_msg)
				put_started(&d->data);
                } else {
                        /*
                         * Device was found. Log the changes (if any).
                         */
			dev.id = d->data.id;
			if (memcmp(&d->data, &dev, NETDEVINFO_LEN)) {
                                if (!full_msg)
                                        put_changes(&d->data, &dev);
                                else
					memcpy(&d->data, &dev, NETDEVINFO_LEN);
			}
		}
		d->flags |= CONSISTENT;
		if (full_msg)
			put(&d->data, NETDEVINFO_LEN);
        }
}

static void net_check(void)
{
	struct netdev_info *d;

	init_msg(full_msg ? NETMON_FULL : NETMON_CHANGES);
        dev_read();

	if (!full_mode) {
                /*
                 * Test for existence of closed devices.
                 */
                for (d = devlist; d; d = d->next)
			if (!(d->flags&CONSISTENT) && d->data.status) {
				d->data.status = 0;
				put_closed(&d->data);
                        }
	}
	
	if (full_msg)
		send_msg();
}

static void net_getmsg(void)
{
	struct cis_netdevinfo msg;
	struct netdev_info *dev;
	int err;

	err = read(netmon.fd, &msg, sizeof(struct cis_netdevinfo));

	if (err != sizeof(struct cis_netdevinfo))
		return;

	for (dev = devlist; dev; dev = dev->next)
		if (!strncmp (dev->data.name, msg.name, CIS_DEVNAMELEN))
			break;

	if (!dev) {
		if (!(dev = register_device(&msg)))
			return;

		init_msg(NETMON_NEW);
		put(&dev->data, NETDEVINFO_LEN);
		send_msg();

		if (msg.status)
			return;
	}

	init_msg(NETMON_CHANGES);
	put_changes(&dev->data, &msg);
	send_msg();
}

struct monitor_desc netmon = {
	NETMON,
	-1,
	net_init,
	net_check,
	net_getmsg,
	net_send_init,
	net_send_exit,
};

