/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Integrated monitor.
 */

#include <sys/time.h>
#include <unistd.h>
#include <syslog.h>

#include "cis.h"
#include "mon_common.h"

struct monitor_desc *monitors[] = {
	&sysmon,
	&procmon,
	&sockmon,
	&netmon,
	NULL,
};

#define for_each_monitor(mon) \
	for (mon = monitors; *mon; mon++)

void cleanup(int sig)
{
	struct monitor_desc **monitor;

	for_each_monitor(monitor)
		(*monitor)->send_exit();
	send_exit(MONITOR_EXIT);
#ifndef DEBUG
	closelog();
#endif
	exit(0);
}


int main(int argc, char *argv[])
{
	struct timeval timeout = {1,0};
	struct monitor_desc **monitor;
	fd_set fdset;

	monitor_type = MONITOR;

        prepare_init(argc, argv);

	for_each_monitor(monitor)
		(*monitor)->init();

	connect_server();
	
#ifndef DEBUG
	daemonize();
#endif

	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);

	for_each_monitor(monitor)
		(*monitor)->send_init();
	send_init(MONITOR_INIT);

	for_each_monitor(monitor)
		(*monitor)->check();
		
	full_msg = FALSE;

	FD_ZERO(&fdset);
	FD_SET(socketfd, &fdset);
	for_each_monitor(monitor)
		if ((*monitor)->fd > -1)
			FD_SET((*monitor)->fd, &fdset);

	timeout = interval_tv;

	for (;;) {
		switch (select (FD_SETSIZE, &fdset, NULL, NULL, &timeout)) {
		case 1 ... FD_SETSIZE:
			for_each_monitor(monitor)
			      if ((*monitor)->fd > -1 &&
				    FD_ISSET((*monitor)->fd, &fdset))
					(*monitor)->getmsg();

			if (!FD_ISSET(socketfd, &fdset))
				break;

		case 0:
			if (recv_msg()) {
				for_each_monitor(monitor)
					(*monitor)->send_init();
				send_init(MONITOR_INIT);
			}
			for_each_monitor(monitor)
				(*monitor)->check();

			if (!full_msg)
				send_msg();
			if (!full_mode)
				full_msg = FALSE;

			timeout = interval_tv;
			break;
		case -1:
			timeout = interval_tv;
			break;
		default:
			break;
		}
		if (server_ok)
			FD_SET(socketfd, &fdset);
		else
			FD_CLR(socketfd, &fdset);

		for_each_monitor(monitor)
			if ((*monitor)->fd > -1)
				FD_SET((*monitor)->fd, &fdset);
	}
}
