/*
 * Cluster Information Service client library - common XDR routines.
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>

#include <rpc/rpc.h>
#include "cis.h"

bool_t xdr_inaddr(XDR *handle, struct in_addr *a)
{
        if (!xdr_u_int(handle, &a->s_addr))
                return (FALSE);
        
        return (TRUE);
}

bool_t xdr_tval(XDR *handle, struct timeval *t)
{
        if (!xdr_long(handle, &t->tv_sec))
                return (FALSE);
        if (!xdr_long(handle, &t->tv_usec))
                return (FALSE);
        
        return (TRUE);
}

bool_t xdr_cisinfo(XDR *handle, struct cis_info *ci)
{
	if (!xdr_wrapstring(handle, &ci->name))
		return (FALSE);

	return (TRUE);
}

bool_t xdr_hostinfo(XDR *handle, struct cis_hostinfo *hi)
{
	char *c;
	
	if (!xdr_inaddr(handle, &hi->addr))
                return (FALSE);
	if (!xdr_inaddr(handle, &hi->caddr))
                return (FALSE);

	c = hi->name;
	if (!xdr_string(handle, &c, CIS_HOSTNAMELEN))
		return (FALSE);

	if (!xdr_u_short(handle, &hi->procnum))
                return (FALSE);
        if (!xdr_vector(handle, (char *) hi->loads, 3, sizeof(short), (xdrproc_t) xdr_short))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->totalswap))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->freeswap))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->freeram))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->sharedram))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->bufferram))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->cachedram))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->idle))
                return (FALSE);

	if (!xdr_u_short(handle, &hi->CPU_available))
                return (FALSE);

        if (!xdr_u_short(handle, &hi->ct_per_sec))
                return (FALSE);
        if (!xdr_u_long(handle, &hi->totalram))
                return (FALSE);
	if (!xdr_char(handle, &hi->reliability))
                return (FALSE);
	if (!xdr_float(handle, &hi->performance))
                return (FALSE);
        return (TRUE);
}


bool_t xdr_process_common(XDR *handle, struct cis_procinfo *p)
{
        char *tmp = p->cmd;
        
        if (!xdr_u_int(handle, &p->pid))
                return (FALSE);
        if (!xdr_u_int(handle, &p->ppid))
                return (FALSE);
        if (!xdr_u_int(handle, &p->uid))
                return (FALSE);
        if (!xdr_string(handle, &tmp, CMD_LENGTH))
                return (FALSE);
        if (!xdr_long(handle, &p->priority))
                return (FALSE);
        if (!xdr_u_long(handle, &p->rss))
                return (FALSE);
        if (!xdr_u_long(handle, &p->vm))
                return (FALSE);

        return (TRUE);
}

bool_t xdr_process(XDR *handle, struct cis_procinfo *p)
{
        if (!xdr_process_common(handle, p))
                return (FALSE);
        if (!xdr_u_long(handle, &p->start_time))
                return (FALSE);
        if (!xdr_u_long(handle, &p->stime))
                return (FALSE);
        if (!xdr_u_long(handle, &p->utime))
                return (FALSE);
        
        return (TRUE);
}


bool_t xdr_u_long_long(XDR *handle, unsigned long long *l)
{
        if (!xdr_u_long(handle, (long *) l + 1))
                return (FALSE);
        if (!xdr_u_long(handle, (long *) l))
                return (FALSE);
        
        return (TRUE);
}

bool_t xdr_socket_desc(XDR *handle, struct cis_sockinfo *s)
{
        if (!xdr_short(handle, &s->type))
                return (FALSE);
        if (!xdr_inaddr(handle, &s->saddr))
                return (FALSE);
        if (!xdr_u_short(handle, &s->sport))
                return (FALSE);
        if (!xdr_inaddr(handle, &s->daddr))
                return (FALSE);
        if (!xdr_u_short(handle, &s->dport))
                return (FALSE);

        return (TRUE);
}

bool_t xdr_socket(XDR *handle, struct cis_sockinfo *s)
{

        if (!xdr_socket_desc(handle, s))
                return (FALSE);

        if (!xdr_u_int(handle, &s->pid))
                return (FALSE);
        if (!xdr_u_short(handle, &s->uid))
                return (FALSE);
        if (!xdr_u_long(handle, &s->sent))
                return (FALSE);
        if (!xdr_u_long(handle, &s->rcvd))
                return (FALSE);

        return (TRUE);
}

bool_t xdr_netdev(XDR *handle, struct cis_netdevinfo *dev)
{
        char *tmp = dev->name;

        if (!xdr_u_char(handle, &dev->status))
                return (FALSE);

        if (!xdr_string(handle, &tmp, CIS_DEVNAMELEN))
                return (FALSE);

        if (!xdr_u_long(handle, &dev->rx_bytes))
                return (FALSE);
        if (!xdr_u_long(handle, &dev->tx_bytes))
                return (FALSE);
        if (!xdr_u_long(handle, &dev->collisions))
                return (FALSE);

        return (TRUE);
}

bool_t xdr_netdev_desc(XDR *handle, struct cis_netdevinfo *dev)
{
        char *tmp = dev->name;

        if (!xdr_u_char(handle, &dev->status))
                return (FALSE);

        if (!xdr_string(handle, &tmp, CIS_DEVNAMELEN))
                return (FALSE);

        return (TRUE);
}

