/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 */

#include "cis_mon.h"

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rpc/auth.h>

#include <net/if.h>
#include <sys/utsname.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <rpc/auth.h>

/*
 * RPC numbers.
 */
#define CIS_PROG           0x22233300
#define CIS_VER            1



/*
 * CIS server dispatch routines.
 */
/* Client calls */
enum {
	CIS_INFO = 1,
        HOST_LIST,
        HOST_INFO,
        HOST_INFO_UPDATE,
        PROCESS_LIST,
        SOCKET_LIST,
        NETDEV_LIST,
};


#ifndef PROC_HISTORY_LENGTH
#define PROC_HISTORY_LENGTH     3
#endif

#ifndef HISTORY_LENGTH
#define HISTORY_LENGTH     60
#endif

struct history_value {
        unsigned long time;
        unsigned long change;
};

struct history {

        struct history_value history[HISTORY_LENGTH];
        struct history_value *last;

        unsigned long interval;
        unsigned long sum;
};

struct host_info
{
        int naddr;
	struct in_addr addrlist[MAX_ADDR_NUM];

//	char *name;
	
        struct monitor_info *monitor[NMONITORS];

        int    defprio;
	struct utsname uname;
        struct cis_hostinfo hostinfo;

	float  *performance;
        
        unsigned long idle_at;
	struct history ctx_swtch_history;
	
	int proc_ht_length;
	int proc_ht_capacity;
	int proc_ht_use;
        struct process_info **proc_ht;
	struct process_info *proclist;
        long slsum;

	int sock_ht_length;
        int sock_ht_capacity;
	int sock_ht_use;
        struct socket_info **sock_ht;
	struct socket_info *socklist;

        struct netdev_info *netdevlist;

	struct host_info *next;

};

struct clnt_req {

        struct in_addr      addr;
        struct host_info    *host;
        struct monitor_info *monitor;
        unsigned long       time;

        unsigned short uid;
        float interval;

        int code;
        int nhosts;
	struct in_addr *addrtab;

	char *perf_type;
	unsigned char reliability;
};

struct monitor_info
{
	unsigned long       clk_interval;
        
        struct timeval      start_tv;  /* refernece time in sec */
        unsigned long       start_clk; /* reference time in clock tics */

        unsigned long       local_time;
	struct timeval      time_tv;
	
        unsigned char       seqnum;
        unsigned char       flags;

	unsigned short      port;
	unsigned short      ct_per_sec;
};

struct process_info {

	struct process_info *prev, *next;

	unsigned char  flags;

	struct cis_procinfo data;

	struct history CPU_history;

	struct history majflt_history;
	unsigned long majflt_rate;

	struct history read_history;
	unsigned long read_rate;

	struct history write_history;
	unsigned long write_rate;

        unsigned long init_interval;
        
        unsigned long GMT_start_time;
        unsigned long utime_sec, stime_sec;
};

struct socket_info {

	struct socket_info *prev, *next;

	unsigned char flags;

	struct cis_sockinfo data;

        struct history sent_history;
        struct history rcvd_history;

        unsigned long send_rate;
        unsigned long recv_rate;
};

struct netdev_info {
	
	struct netdev_info *prev, *next;

	unsigned char flags;

	struct cis_netdevinfo data;
	
        struct history rxbytes_history;
        struct history txbytes_history;
        struct history collisions_history;

        unsigned long rxbytes_rate;
        unsigned long txbytes_rate;
        unsigned long collisions_rate;
};

#define tvaldiff(a, b) (a.tv_sec - b.tv_sec + ((a.tv_usec - b.tv_usec) / 1e6))

#define monitor_time(monitor) \
        (monitor->local_time + (unsigned long) ((tvaldiff(current_tv, monitor->time_tv) * \
	monitor->ct_per_sec)))

