/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * CIS server daemon - client RPC calls dispatch routines.
 */

#include "cis.h"
#include "cis_xdr.h"
#include "cis_srv.h"
#include "cis_clnt.h"

extern bool_t xdr_process_common(XDR *handle, struct cis_procinfo *p);
extern struct host_info *host_list;

/*
 * Output filters
 */
/* For messages from clients */
extern struct timeval current_tv;
extern char **perf_types;
extern int nperf_types;
extern struct cis_info my_info;

struct host_info *find_host(struct in_addr addr);

int count_pCPU(struct host_info *host, struct process_info *p, unsigned long current_time);
int availableCPU(struct host_info *host, unsigned long clk);
int monitor_idle(struct monitor_info *monitor);
unsigned long count_rate(struct history *h, struct clnt_req *req);

bool_t xdr_list_req(XDR *handle, struct clnt_req *req)
{
        if (!xdr_float(handle, &req->interval))
                return (FALSE);
	if (!xdr_u_short(handle, &req->uid))
                return (FALSE);

        return (TRUE);
}

int perf_index (char *perf)
{
	int i;

	for (i = 1; i < nperf_types; i++)
		if (!strcmp(perf_types[i], perf))
			return i;
	return 0;
}

bool_t xdr_hostlist(XDR *handle, struct clnt_req *req)
{
	struct host_info *host;
	struct cis_hostinfo *hi;
	unsigned long rate;
	int n = 0, perf_type = 0;

	perf_type = perf_index(req->perf_type);
	
	for (host = host_list; host; host = host->next)
		if (host->hostinfo.reliability >= req->reliability)
			n++;

	if (!xdr_int(handle, &n))
                return (FALSE);

	if (!n)
                return (TRUE);

	for (host = host_list; host; host = host->next)
		if (host->hostinfo.reliability >= req->reliability) {
			req->host = host;
			hi = &host->hostinfo;
			if (host->monitor[PROCMON]) {
				req->time = monitor_time(host->monitor[PROCMON]);
				hi->CPU_available = availableCPU(host, req->time);
			} else
				hi->CPU_available = 0;
			
			hi->performance   = host->performance[perf_type];

			if (host->monitor[SYSMON]) {
				req->time = monitor_time(host->monitor[SYSMON]);
				rate = count_rate(&host->ctx_swtch_history, req);
			} else
				rate = 0;

			if (!xdr_hostinfo(handle, hi))
				return (FALSE);
			if (!xdr_u_long(handle, &rate))
				return (FALSE);
		}

	return (TRUE);
}


bool_t xdr_host_info(XDR *handle, struct clnt_req *req)
{
	struct host_info *host = req->host;
	struct cis_hostinfo *hi = &host->hostinfo;
	char state = HOST_AVAILABLE;
	unsigned long rate;

	hi->CPU_available = availableCPU(host, req->time);
	hi->performance = host->performance[perf_index(req->perf_type)];
	rate = count_rate(&host->ctx_swtch_history, req);
	
	if (!xdr_char(handle, &state))
		return (FALSE);

	if (!xdr_hostinfo(handle, hi))
                return (FALSE);

	if (!xdr_u_long(handle, &rate))
		return (FALSE);

        return (TRUE);
}

bool_t xdr_host_update(XDR *handle, struct clnt_req *req)
{
        struct in_addr *addr;
        struct cis_hostinfo *hi;
        struct host_info *host;
        char stat;
	unsigned long rate;
	
        for (addr = req->addrtab; addr < req->addrtab+req->nhosts; addr++) {

		stat = HOST_AVAILABLE;
		
                host = find_host(*addr);
                if (!host)
                        stat = HOST_UNKNOWN;
		else if (!host->monitor[SYSMON])
			stat = HOST_NOT_AVAILABLE;
		else if (monitor_idle (host->monitor[SYSMON]))
			stat = HOST_NOT_AVAILABLE;

                if (!xdr_char(handle, &stat))
                        return (FALSE);

                if (stat != HOST_AVAILABLE)
                        continue;

                hi = &host->hostinfo;
                
                if (req->code & CIS_PROCNUM)
                        if (!xdr_u_short(handle, &hi->procnum))
                                return (FALSE);

                if (req->code & CIS_LOADS)
                        if (!xdr_vector(handle, (char *) hi->loads, 3, sizeof(short), (xdrproc_t) xdr_u_short))
                                return (FALSE);

                if (req->code & CIS_RAM) {
                        if (!xdr_u_long(handle, &hi->totalram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->sharedram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->bufferram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->cachedram))
                                return (FALSE);
                }

                if (req->code & CIS_SWAP) {
                        if (!xdr_u_long(handle, &hi->totalswap))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeswap))
                                return (FALSE);
                }
                
		if (req->code & CIS_CPU) {
			hi->CPU_available = host->monitor[PROCMON] ?
				availableCPU(host, monitor_time(host->monitor[PROCMON])) : 0;
                        if (!xdr_u_short(handle, &hi->CPU_available))
                                return (FALSE);
                }

                if (req->code & CIS_IDLE)
                        if (!xdr_u_long(handle, &hi->idle))
                                return (FALSE);

		if (req->code & CIS_CTXSWTCH) {
			req->host = host;
			req->time = monitor_time(host->monitor[SYSMON]);
                        rate = count_rate(&host->ctx_swtch_history, req);
                        if (!xdr_u_long(handle, &rate))
				return (FALSE);
		}
        }

        return (TRUE);
}

#define user_match(a,b) (a == 0xffff || (a ? (a == b) : (b > MIN_USER_UID)))

bool_t xdr_proclist_rep(XDR *handle, struct clnt_req *req)
{
	struct process_info *p;
	struct host_info *host = req->host;
	unsigned short uid = req->uid;
        int n;

	for (p = host->proclist, n = 0; p; p = p->next) {

		if (!user_match(uid, p->data.uid))
			continue;

		p->data.pCPU = count_pCPU(host, p, req->time);
		n++;
	}
	if (!xdr_int(handle, &n))
		return (FALSE);

	if (!n)
                return (TRUE);

	for (p = host->proclist; p; p = p->next) {

		if (!user_match(uid, p->data.uid))
			continue;

		if (!xdr_process_common(handle, &p->data))
                        return (FALSE);

		p->majflt_rate = count_rate(&p->majflt_history, req);
		p->read_rate   = count_rate(&p->read_history,   req);
		p->write_rate  = count_rate(&p->write_history,  req);

		if (!xdr_u_long(handle, &p->GMT_start_time))
                        return (FALSE);
                if (!xdr_u_long(handle, &p->stime_sec))
                        return (FALSE);
                if (!xdr_u_long(handle, &p->utime_sec))
                        return (FALSE);
		if (!xdr_short(handle, &p->data.pCPU))
			return (FALSE);

		if (!xdr_u_long(handle, &p->majflt_rate))
			return (FALSE);
		if (!xdr_u_long(handle, &p->read_rate))
			return (FALSE);
		if (!xdr_u_long(handle, &p->write_rate))
			return (FALSE);
        }
        return (TRUE);
}

bool_t xdr_socklist_rep(XDR *handle, struct clnt_req *req)
{
	struct host_info *host = req->host;
        struct socket_info *s;
	unsigned short uid = req->uid;
        int n;

	for (s = host->socklist, n = 0; s; s = s->next) {
		if (!user_match(uid, s->data.uid))
			continue;

		s->send_rate = count_rate(&s->sent_history, req);
		s->recv_rate = count_rate(&s->rcvd_history, req);

		if (!s->send_rate && !s->recv_rate)
                        continue;
		n++;
        }
        if (!xdr_int(handle, &n))
                return (FALSE);

        if (!n)
                return (TRUE);

        for (s = host->socklist; s; s = s->next) {
		if (!user_match(uid, s->data.uid))
			continue;

		if (!s->send_rate && !s->recv_rate)
                        continue;
                
                if (!xdr_socket_desc(handle, &s->data))
                        return (FALSE);

		if (!xdr_u_int(handle, &s->data.pid))
                        return (FALSE);
		if (!xdr_u_short(handle, &s->data.uid))
                        return (FALSE);

                if (!xdr_u_long(handle, &s->send_rate))
                        return (FALSE);
		if (!xdr_u_long(handle, &s->recv_rate))
                        return (FALSE);
        }
        return (TRUE);
}

bool_t xdr_netdevlist_rep(XDR *handle, struct clnt_req *req)
{
        struct host_info *host = req->host;
	struct netdev_info *d;
	int n;

        for (d = host->netdevlist, n = 0; d; d = d->next, n++)
                ;

	if (!xdr_int(handle, &n))
                return (FALSE);

        if (!n)
                return (TRUE);

        for (d = host->netdevlist; d; d = d->next) {

		if (d->data.status) {
                        d->rxbytes_rate    = count_rate(&d->rxbytes_history, req);
                        d->txbytes_rate    = count_rate(&d->txbytes_history, req);
			d->collisions_rate = count_rate(&d->collisions_history, req);
                } else
			d->rxbytes_rate = d->txbytes_rate = d->collisions_rate = 0;

		if (!xdr_netdev_desc(handle, &d->data))
                        return (FALSE);

                if (!xdr_u_long(handle, &d->rxbytes_rate))
                        return (FALSE);
                if (!xdr_u_long(handle, &d->txbytes_rate))
                        return (FALSE);
		if (!xdr_u_long(handle, &d->collisions_rate))
                        return (FALSE);
        }
        return (TRUE);
}

void clean_req(struct clnt_req *req)
{
	if (req->addrtab)
		free(req->addrtab);
	if (req->perf_type)
		free(req->perf_type);
	memset(req, 0, sizeof (struct clnt_req));
}


void dispatcher(struct svc_req *rqstp, SVCXPRT *transp)
{
        struct monitor_info *monitor = NULL;
        struct host_info *host;
        char err = 0;
        struct in_addr *addr;
	struct clnt_req req;

	gettimeofday(&current_tv, NULL);
	memset (&req, 0, sizeof(req));
	
	if (rqstp->rq_proc == CIS_INFO) {
		if (!svc_sendreply(transp, (xdrproc_t) xdr_cisinfo, (caddr_t) &my_info))
			goto system_error;
	}

	if (rqstp->rq_proc == HOST_LIST) {
		req.perf_type = NULL;
		if (!svc_getargs(transp, (xdrproc_t) xdr_inaddr,     (caddr_t) &req.addr)      ||
		    !svc_getargs(transp, (xdrproc_t) xdr_float,      (caddr_t) &req.interval)  ||
		    !svc_getargs(transp, (xdrproc_t) xdr_wrapstring, (caddr_t) &req.perf_type) ||
		    !svc_getargs(transp, (xdrproc_t) xdr_char,       (caddr_t) &req.reliability))
			goto decode_error;
		if (!svc_sendreply(transp, (xdrproc_t) xdr_hostlist, (caddr_t) &req))
			goto system_error;

		clean_req(&req);
		return;
	}
	
        if (rqstp->rq_proc == HOST_INFO_UPDATE) {
                if (!svc_getargs(transp, (xdrproc_t) xdr_int,   (caddr_t) &req.code) ||
                    !svc_getargs(transp, (xdrproc_t) xdr_float, (caddr_t) &req.interval) ||
                    !svc_getargs(transp, (xdrproc_t) xdr_int,   (caddr_t) &req.nhosts))
                        goto decode_error;

                req.addrtab = calloc(req.nhosts+1, sizeof (struct in_addr));
                if (!req.addrtab)
                        goto system_error;
                
                for (addr = req.addrtab; addr < req.addrtab+req.nhosts; addr++)
			if (!svc_getargs(transp, (xdrproc_t) xdr_inaddr, (caddr_t) addr))
				goto decode_error;

		if (!svc_sendreply(transp, (xdrproc_t) xdr_host_update, (caddr_t) &req))
			goto system_error;

		clean_req(&req);
		return;
	}

	if (!svc_getargs(transp, (xdrproc_t) xdr_inaddr, (caddr_t) &req.addr))
                goto decode_error;

	/* Host address must be specified */
	if (!req.addr.s_addr)
                goto decode_error;
        
        host = find_host(req.addr);
        if (!host) {
                err = HOST_UNKNOWN;
                goto host_error;
        }
        
        switch (rqstp->rq_proc) {
	case HOST_INFO:
                monitor = host->monitor[SYSMON];
                break;
        case PROCESS_LIST:
                monitor = host->monitor[PROCMON];
                break;
        case SOCKET_LIST:
                monitor = host->monitor[SOCKMON];
                break;
        case NETDEV_LIST:
                monitor = host->monitor[NETMON];
                break;
        }

        if (!monitor) {
                err = HOST_NOT_AVAILABLE;
                goto host_error;
        }

	if (monitor_idle(monitor)) {
		err = HOST_NOT_AVAILABLE;
		goto host_error;
	}

	req.host    = host;
        req.monitor = monitor;
        req.time    = monitor_time(monitor);

	if (rqstp->rq_proc == HOST_INFO) {
		req.perf_type = NULL;
		if (!svc_getargs(transp, (xdrproc_t) xdr_float,      (caddr_t) &req.interval) ||
		    !svc_getargs(transp, (xdrproc_t) xdr_wrapstring, (caddr_t) &req.perf_type))
			goto decode_error;
		if (!svc_sendreply(transp, (xdrproc_t) xdr_host_info, (caddr_t) &req))
			goto system_error;

		clean_req(&req);
		return;
	}

	if (!svc_getargs(transp, (xdrproc_t) xdr_list_req, (caddr_t) &req))
		goto decode_error;

        switch (rqstp->rq_proc) {
                
        case PROCESS_LIST:
                if (!svc_sendreply(transp, (xdrproc_t) xdr_proclist_rep, (caddr_t) &req))
                        goto system_error;
                break;
                
        case SOCKET_LIST:
                if (!svc_sendreply(transp, (xdrproc_t) xdr_socklist_rep, (caddr_t) &req))
                        goto system_error;
                break;
                
        case NETDEV_LIST:
                if (!svc_sendreply(transp, (xdrproc_t) xdr_netdevlist_rep, (caddr_t) &req))
                        goto system_error;
                break;
                
        default:
                svcerr_noproc(transp);
                return;
        }
        return;

host_error:
        if (!svc_sendreply(transp, (xdrproc_t) xdr_char, (caddr_t) &err))
                goto system_error;
        return;

system_error:
	svcerr_systemerr(transp);
	clean_req(&req);
        return;

decode_error:
	svcerr_decode(transp);
	clean_req(&req);
        return;
}
