/*
 * Cluster Information Service client library - client RPC calls to CIS server.
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>

#include "cis.h"
#include "cis_xdr.h"
#include "cis_srv.h"
#include "cis_clnt.h"

#define OFFSET(str, el)  (int) &(((struct str *) 0)->el)
#define SIZE(str, el)    sizeof (((struct str *) NULL)->el)
#define ELEMENT(str, el) OFFSET(str,el), SIZE(str,el)

char *rel_types[] = {"minimal", "low", "average", "high", "maximal"};

#define LAST_1 (1 << (sizeof (int)*8 - 1))

static struct timeval cis_timeout={0,500000};
int cis_errno;
static int cis_result;

static int list_length;
static int update_code;
static float req_interval;

struct host_req {
        struct in_addr addr;
	float interval;
	char *perf_type;
	unsigned char reliability;
};

struct list_req {
	struct in_addr addr;
	float interval;
	unsigned short uid;
};

int parse_cmd (char *cmds[], int n, char **c)
{
	int i;

	*c += strspn (*c, " \t");

	for (i = 0; i < n; i++)
		if (!strncasecmp (*c, cmds[i], strlen (cmds[i]))) {
			*c += strlen(cmds[i]);
			return (i);
		}
	return -1;
}

static int proc_cmp(const void *l, const void *r)
{
        struct cis_procinfo *left  = (struct cis_procinfo *) l;
        struct cis_procinfo *right = (struct cis_procinfo *) r;

        return (left->pid - right->pid);
}

static int sock_cmp(const void *l, const void *r)
{
        struct cis_sockinfo *left  = (struct cis_sockinfo *) l;
        struct cis_sockinfo *right = (struct cis_sockinfo *) r;
        int res;
        
        if ((res = left->saddr.s_addr - right->saddr.s_addr))
                return (res);
        if ((res = left->type - right->type))
                return (res);
        if ((res = left->sport - right->sport))
                return (res);
        if ((res = left->daddr.s_addr - right->daddr.s_addr))
                return (res);
        if ((res = left->dport - right->dport))
                return (res);

        return (0);
}

static int netdev_cmp(const void *l, const void *r)
{
        struct cis_netdevinfo *left  = (struct cis_netdevinfo *) l;
        struct cis_netdevinfo *right = (struct cis_netdevinfo *) r;

	return (strncmp(left->name, right->name, CIS_DEVNAMELEN));
}

static bool_t xdr_hostlist(XDR *handle, struct cis_hostinfo **listp)
{
	int length = 0;
	struct cis_hostinfo *new_list, *hi;

        list_length = 0;
	cis_result = 0;

	if (!xdr_int(handle, &length))
                return (FALSE);

        if (!length)
                return (TRUE);

	if (length < 0) {
		cis_result = length;
		return (TRUE);
	}

	free(*listp);
	*listp = NULL;

	if (!(new_list = calloc(length, sizeof (struct cis_hostinfo))))
		return (FALSE);
	*listp = new_list;

        for (hi = new_list; hi < new_list + length; hi++) {

		if (!xdr_hostinfo(handle, hi))
			return (FALSE);
		if (!xdr_u_long(handle, &hi->ctx_swtch))
			return (FALSE);
	}

        list_length = length;

        return (TRUE);
}

static bool_t xdr_host_req(XDR *handle, struct host_req *req)
{
        if (!xdr_inaddr(handle, &req->addr))
		return (FALSE);
	if (!xdr_float(handle, &req->interval))
                return (FALSE);
	if (!xdr_wrapstring(handle, &req->perf_type))
		return (FALSE);
	if (!xdr_u_char(handle, &req->reliability))
		return (FALSE);
	return (TRUE);
}

static bool_t xdr_list_req(XDR *handle, struct list_req *req)
{
        if (!xdr_inaddr(handle, &req->addr))
                return (FALSE);
	if (!xdr_float(handle, &req->interval))
                return (FALSE);
	if (!xdr_u_short(handle, &req->uid))
		return (FALSE);
	return (TRUE);
}

static bool_t xdr_host_rep(XDR *handle, struct cis_hostinfo *hi)
{
        if (!xdr_char(handle, &hi->status))
                return (FALSE);

        if (hi->status != HOST_AVAILABLE)
                return (TRUE);

        if (!xdr_hostinfo(handle, hi))
                return (FALSE);

	if (!xdr_u_long(handle, &hi->ctx_swtch))
		return (FALSE);

        return (TRUE);
}

static bool_t xdr_hosts_update_req(XDR *handle, struct cis_hostinfo **hi)
{
        int i;
        
        if (!xdr_int(handle, &update_code))
                return (FALSE);

        if (!xdr_float(handle, &req_interval))
                return (FALSE);

        if (!xdr_int(handle, &list_length))
                return (FALSE);

        for (i = 0; i < list_length; i++, hi++)
                if ((*hi)->addr.s_addr)
                        if (!xdr_inaddr(handle, &(*hi)->addr))
                                return (FALSE);

        return (TRUE);
}

static bool_t xdr_hosts_update_reply(XDR *handle, struct cis_hostinfo **hinfo)
{
        int i;
        struct cis_hostinfo *hi;

        for (i = 0; i < list_length; i++, hinfo++) {
                hi = *hinfo;
                if (!hi->addr.s_addr)
                        continue;

                if (!xdr_char(handle, &hi->status))
                        return (FALSE);

		if (hi->status != HOST_AVAILABLE) {
			hi->CPU_available = 0;
			continue;
		}

                if (update_code & CIS_PROCNUM)
                        if (!xdr_u_short(handle, &hi->procnum))
                                return (FALSE);

                if (update_code & CIS_LOADS)
                        if (!xdr_vector(handle, (char *) hi->loads, 3, sizeof(short), (xdrproc_t) xdr_float))
                                return (FALSE);

                if (update_code & CIS_RAM) {
                        if (!xdr_u_long(handle, &hi->totalram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->sharedram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->bufferram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->cachedram))
                                return (FALSE);
                }

                if (update_code & CIS_SWAP) {
                        if (!xdr_u_long(handle, &hi->totalswap))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeswap))
                                return (FALSE);
                }
                
                if (update_code & CIS_CPU)
                        if (!xdr_u_short(handle, &hi->CPU_available))
                                return (FALSE);

                if (update_code & CIS_IDLE)
                        if (!xdr_u_long(handle, &hi->idle))
                                return (FALSE);

		if (update_code & CIS_CTXSWTCH)
                        if (!xdr_u_long(handle, &hi->ctx_swtch))
                                return (FALSE);
        }

        return (TRUE);
}

static bool_t xdr_proc_reply(XDR *handle, struct cis_procinfo **listp)
{
        int length = 0;
        struct cis_procinfo *new_list, *p;

        list_length = 0;
        cis_result  = HOST_AVAILABLE;
        
        if (!xdr_int(handle, &length))
                return (FALSE);

        if (!length)
                return (TRUE);

        if (length < 0) {
                cis_result = length;
                return (TRUE);
        }

        free(*listp);
        *listp = NULL;

        if (!(new_list = calloc(length, sizeof (struct cis_procinfo))))
                return (FALSE);
        *listp = new_list;

        for (p = new_list; p < new_list + length; p++) {

                if (!xdr_process(handle, p))
                        return (FALSE);

                if (!xdr_short(handle, &p->pCPU))
                        return (FALSE);

		if (!xdr_u_long(handle, &p->majflt))
                        return (FALSE);

		if (!xdr_u_long(handle, &p->rd_bytes))
                        return (FALSE);

		if (!xdr_u_long(handle, &p->wr_bytes))
                        return (FALSE);

//                p->flags = CONSISTENT;
        }
        qsort(new_list, length, sizeof (struct cis_procinfo), proc_cmp);

        list_length = length;

        return (TRUE);
}

static bool_t xdr_sock_reply(XDR *handle, struct cis_sockinfo **listp)
{
        int length = 0;
        struct cis_sockinfo *new_list, *s;

        free(*listp);
        *listp = NULL;
        list_length = 0;
        cis_result  = HOST_AVAILABLE;

        if (!xdr_int(handle, &length))
                return (FALSE);

        if (!length)
                return (TRUE);

        if (length < 0) {
                cis_result = length;
                return (TRUE);
        }

        new_list = calloc(length, sizeof (struct cis_sockinfo));
        if (!new_list)
                return (FALSE);
        *listp = new_list;

        for (s = *listp; s < *listp + length; s++) {
                if (!xdr_socket_desc(handle, s))
                        return (FALSE);

                if (!xdr_u_int(handle, &s->pid))
			return (FALSE);

		if (!xdr_u_short(handle, &s->uid))
                        return (FALSE);

                if (!xdr_u_long(handle, &s->sent))
                        return (FALSE);
                if (!xdr_u_long(handle, &s->rcvd))
                        return (FALSE);

//                s->flags = TRUE;
        }

        qsort(*listp, length, sizeof (struct cis_sockinfo), sock_cmp);
        list_length = length;
        
        return (TRUE);
}

static bool_t xdr_netdev_reply(XDR *handle, struct cis_netdevinfo **listp)
{
        int length = 0;
        struct cis_netdevinfo *new_list, *dev;

        free(*listp);
        *listp = NULL;
        list_length = 0;
        cis_result  = HOST_AVAILABLE;

        if (!xdr_int(handle, &length))
                return (FALSE);

	if (!length) {
		cis_result = length;
                return (TRUE);
	}
	
        if (length < 0) {
                cis_result = length;
                return (TRUE);
        }

        new_list = calloc(length, sizeof (struct cis_netdevinfo));
        if (!new_list)
                return (FALSE);
        *listp = new_list;

        for (dev = *listp; dev < *listp + length; dev++) {
                if (!xdr_netdev(handle, dev))
                        return (FALSE);

//                dev->flags = TRUE;
        }

        qsort(*listp, length, sizeof (struct cis_netdevinfo), netdev_cmp);
        list_length = length;

        return (TRUE);
}

/*
 * Interface calls
 */

void cisTimeout(CLIENT *handle, struct timeval tout)
{
        clnt_control(handle, CLSET_TIMEOUT,(char *) &tout);

        cis_timeout = tout;
}

CLIENT *cisConnect(char *hostname)
{
        return (clnt_create(hostname, CIS_PROG, CIS_VER, "tcp"));
}

void cisDisconnect(CLIENT *handle)
{
        clnt_destroy(handle);
}

struct cis_info *cisInfo(CLIENT *handle)
{
	int err;
	struct cis_info *info;
	
	if (!handle) {
                cis_errno = CIS_NOCONN;
		return NULL;
	}

	info = calloc (1, sizeof (struct cis_info));
	
	err = clnt_call(handle, CIS_INFO,
			(xdrproc_t) xdr_void, (caddr_t) NULL,
			(xdrproc_t) xdr_cisinfo, (caddr_t) info,
			cis_timeout);

	if (err != RPC_SUCCESS) {
		cis_errno = err;
		return NULL;
	}

	return info;
}


int cisHostlist(CLIENT *handle, struct cis_hostinfo **list, unsigned char reliability, char *perf_type, float interval)
{
	struct host_req req = {{0}, interval, perf_type, reliability};
	int err;

	*list = NULL;

	cis_errno = 0;

	if (!handle) {
                cis_errno = CIS_NOCONN;
                return -1;
	}

	err = clnt_call(handle, HOST_LIST,
			(xdrproc_t) xdr_host_req, (caddr_t) &req,
			(xdrproc_t) xdr_hostlist, (caddr_t) list,
			cis_timeout);

	if (err != RPC_SUCCESS) {
		cis_errno = err;
		return -1;
	}

	return list_length;
}

struct cis_hostinfo *cisHostinfo(CLIENT *handle, struct in_addr addr, char *perf_type, float interval)
{
	struct host_req req = {addr, interval, perf_type, 0};
	struct cis_hostinfo *hinfo;
        int err;

	if (!handle) {
                cis_errno = CIS_NOCONN;
                return NULL;
	}
	
        hinfo = calloc(1, sizeof (struct cis_hostinfo));

        if (!hinfo) {
                cis_errno = CIS_NOMEM;
                return NULL;
        }

	err = clnt_call(handle, HOST_INFO,
			(xdrproc_t) xdr_host_req, (caddr_t) &req,
                        (xdrproc_t) xdr_host_rep, (caddr_t) hinfo,
                        cis_timeout);

        if (err != RPC_SUCCESS) {
                cis_errno = err;
                free(hinfo);
                return NULL;
        }

        return hinfo;
}

int cisUpdateHostsinfo(CLIENT *handle, int code, struct cis_hostinfo **hosttab, int nhosts, float interval)
{
        int err;

	if (!handle) {
                cis_errno = CIS_NOCONN;
                return -1;
	}

        update_code  = code;
        list_length  = nhosts;
        req_interval = interval;
            
        err = clnt_call(handle, HOST_INFO_UPDATE,
                        (xdrproc_t) xdr_hosts_update_req, (caddr_t) hosttab,
                        (xdrproc_t) xdr_hosts_update_reply, (caddr_t) hosttab,
                        cis_timeout);

        if (err != RPC_SUCCESS) {
                cis_errno = err;
                return -1;
        }
        return 0;
}


int cisUpdateHostinfo (CLIENT *handle, int code, struct cis_hostinfo *hinfo, float interval)
{
	return cisUpdateHostsinfo (handle, code, &hinfo, 1, interval);
}


int cisProclist(CLIENT *handle, struct cis_procinfo **list, struct in_addr addr, unsigned short uid, float interval)
{
	struct list_req req = {addr, interval, uid};
	int err;
        *list = NULL;

	cis_errno = 0;

	if (!handle) {
                cis_errno = CIS_NOCONN;
                return -1;
	}

        err = clnt_call(handle, PROCESS_LIST,
			(xdrproc_t) xdr_list_req,   (caddr_t) &req,
                        (xdrproc_t) xdr_proc_reply, (caddr_t) list,
                        cis_timeout);
	if (err) {
		cis_errno = err;
		return -1;
	}
	
	if (cis_result != HOST_AVAILABLE) {
                cis_errno = cis_result;
		return -1;
	}

        return list_length;
}

int cisSocklist(CLIENT *handle, struct cis_sockinfo **list, struct in_addr addr, unsigned short uid, float interval)
{
	struct list_req req = {addr, interval, uid};
        int err;
        *list = NULL;

	cis_errno = 0;

	if (!handle) {
                cis_errno = CIS_NOCONN;
                return -1;
	}

        err = clnt_call(handle, SOCKET_LIST,
			(xdrproc_t) xdr_list_req,   (caddr_t) &req,
                        (xdrproc_t) xdr_sock_reply, (caddr_t) list,
                        cis_timeout);
	if (err) {
		cis_errno = err;
		return -1;
	}
	
	if (cis_result != HOST_AVAILABLE) {
                cis_errno = cis_result;
		return -1;
	}

        return list_length;
}

int cisNdevlist(CLIENT *handle, struct cis_netdevinfo **list, struct in_addr addr, float interval)
{
	struct list_req req = {addr, interval, 0};
        int err;
        *list = NULL;

        cis_errno = 0;
	
	if (!handle) {
                cis_errno = CIS_NOCONN;
                return -1;
	}

	err = clnt_call(handle, NETDEV_LIST,
			(xdrproc_t) xdr_list_req,   (caddr_t) &req,
                        (xdrproc_t) xdr_netdev_reply, (caddr_t) list,
                        cis_timeout);
	if (err) {
		cis_errno = err;
		return -1;
	}
	
	if (cis_result != HOST_AVAILABLE) {
                cis_errno = cis_result;
		return -1;
	}
	
	return list_length;
}

