/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Kernel module for monitoring of network devices with start/stop notification
 * via netlink device.
 */

#include <linux/config.h>
#include <linux/module.h>

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <asm/uaccess.h>

#include <net/inet_common.h>
#include <net/tcp.h>
#include <linux/netdevice.h>
#include <linux/proc_fs.h>

#include "cis.h"
#include "cis_mon.h"

static struct sock *nmsk;

int read_netdevlist(char *buffer, char **start, off_t offset, int length, int dummy)
{
        int len = 0;
        off_t pos = 0;
        off_t begin;
        struct cis_netdevinfo msg;
        struct device *dev;
        struct net_device_stats *stats;

        for (dev = dev_base; dev != NULL; dev = dev->next) {

                if (dev == &loopback_dev)
                        continue;

                stats = (dev->get_stats ? dev->get_stats(dev): NULL);

                if (!stats)
                        continue;

                pos += sizeof(struct cis_netdevinfo);
                if (pos < offset)
                        continue;

                msg.status = dev->start;
		strncpy(msg.name, dev->name, IFNAMSIZ);
                msg.rx_bytes = stats->rx_bytes;
                msg.tx_bytes = stats->tx_bytes;
                msg.collisions = stats->collisions;
                

                memcpy(buffer + len, &msg, sizeof(struct cis_netdevinfo));
                len += sizeof(struct cis_netdevinfo);
                if(len >= length)
                        goto out;

        }

    out:

        begin = len - (pos - offset);
        *start = buffer + begin;
        len -= begin;
        if(len > length)
                len = length;
        if (len < 0)
                len = 0;
        return len;
}

static struct proc_dir_entry proc_root_netdevlist = {
        0,                               /* Inode number      */
        14, "cis_netdevlist",            /* The name of file with length */
        S_IFREG | S_IRUGO,               /* Acess permissions */
        1, 0, 0,                         /* Number of links, owner, group */
        0,                               /* The size of the file reported by ls. */
        NULL,                            /* Operations - use default */
        read_netdevlist                  /* The read function */
        /* nothing more */
};

static int netmon_msg(struct device *dev, unsigned char action)
{
        struct cis_netdevinfo *msg;
        size_t len =  sizeof(struct cis_netdevinfo);
        struct net_device_stats *stats = (dev->get_stats ? dev->get_stats(dev): NULL);

        struct sk_buff *outskb = alloc_skb(len, GFP_ATOMIC);

        if (outskb) {
                skb_put(outskb, len);
                msg = (struct cis_netdevinfo  *) outskb->data;

                msg->status = action;
                strncpy (msg->name, dev->name, IFNAMSIZ);

                msg->rx_bytes   = (stats) ? stats->rx_bytes : 0;
                msg->tx_bytes   = (stats) ? stats->tx_bytes : 0;
                msg->collisions = (stats) ? stats->collisions : 0;

                netlink_broadcast(nmsk, outskb, 0, ~0, GFP_KERNEL);
        }
        else
                return (-1);

	return (0);
}

static int netmon_notifier(struct notifier_block *this, unsigned long msg, void *data)
{
        struct device *dev = (struct device*)data;

        switch (msg) {
        case NETDEV_DOWN:
        case NETDEV_UNREGISTER:
		netmon_msg(dev, OBJ_DESTROY);
                break;
        case NETDEV_UP:
                netmon_msg(dev, OBJ_CREATE);
                break;
        }
        return NOTIFY_DONE;
}

struct notifier_block netmon_netdev_notifier={
        netmon_notifier,
	NULL,
	0
};

int netmon_init(void)
{
        int err;

        nmsk = netlink_kernel_create(NETLINK_NETMON, NULL);
        if (nmsk == NULL) {
                printk("netmon_init: cannot initialize netlink\n");
                return -ENODEV;
        }
        
        err = proc_register(&proc_root, &proc_root_netdevlist);
        if (err) {
                printk("netmon_init: cannot register netdevlist\n");
                return err;
        }
        register_netdevice_notifier(&netmon_netdev_notifier);

        return 0;
}

int init_module(void)
{
        return (netmon_init ());
}

void cleanup_module(void)
{
        proc_unregister(&proc_root, proc_root_netdevlist.low_ino);
        unregister_netdevice_notifier(&netmon_netdev_notifier);
        sock_release (nmsk->socket);
}
