/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_rec: background (daemon) process for saving information from CIS server
 *          into record file. The file can be analyzed using xcis or cis_print.
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <syslog.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <locale.h>

#include "cis.h"
#include "cis_clnt.h"

#define DEF_INTERVAL                1.
#define DEF_RECORD_DENSITY          50000
#define SOCK_INTERVAL               5

#define log(tag, fmt, args...) \
	if (!myuid) \
	       syslog(tag, fmt , ## args); \
	else \
	       printf(fmt"\n" , ## args);

char myhostname[MAXHOSTNAMELEN];
char *logfilename = NULL;
int  logfile = 0;

float interval = DEF_INTERVAL;
struct timeval interval_tval = {(int) DEF_INTERVAL, (DEF_INTERVAL - (int) DEF_INTERVAL) * 1000000};

unsigned short myuid;
unsigned short uid;

void allarm_handler (int sig);

int set_configfile=FALSE; char *configfilename;

struct {
        char               *filename;
        FILE               *file;
        char               *path;
        int                density;
} record = {NULL, NULL, "./", DEF_RECORD_DENSITY};

struct timeval current_time;

struct cis_info {
        char *name;
        CLIENT *handle;
        struct cis_hostinfo **htab;
        int nhosts;

} *cis = NULL;

struct host_info {
        char *name;
        struct in_addr addrlist[MAX_ADDR_NUM];
#define hostaddr addrlist[0]

        float interval;
        
        struct cis_hostinfo *hostinfo, *old_hostinfo;
        struct timeval time;
        
        int nproc;
        struct cis_procinfo *proctab;
        
        int nsock;
        struct cis_sockinfo *socktab;

        int ndev;
        struct cis_netdevinfo *devtab;
        
        int hostinfo_saved_bytes;
        int proc_saved_bytes;
        int sock_saved_bytes;
        int dev_saved_bytes;

	struct host_info *next;
} *hostlist = NULL;

int nhosts = 0;

static struct host_info *add_host (char *name)
{
	struct host_info *host;
        struct hostent *hent;
        int i;

	srand(200);
	
	if ((hent = gethostbyname (name)) == NULL) {
		log(LOG_ERR, "cannot get the address of host %s", name);
		exit(1);
	}
	if (!(host = calloc (1, sizeof (struct host_info))))
		goto nomem;

	for (i = 0; hent->h_addr_list[i]; i++)
		memcpy (&host->addrlist[i], hent->h_addr_list[i], hent->h_length);
	host->addrlist[i].s_addr = 0;

	host->hostinfo = calloc(1, sizeof(struct cis_hostinfo));
	if (!host->hostinfo)
		goto nomem;
	host->old_hostinfo = calloc (1, sizeof (struct cis_hostinfo));
	if (!host->old_hostinfo)
		goto nomem;

	host->next = hostlist;
	hostlist = host;
	nhosts++;
	return host;

nomem:
	log(LOG_ERR, "cannot allocate memory for host structures");
	exit(1);
}

void getparams(int argc, char **argv)
{
	int c;
	char **hlist, **h;
	struct host_info *host;
	struct cis_hostinfo *hi;
	int i;

	if (!myuid)
		uid = myuid;

	cis = calloc(1, sizeof (struct cis_info));
	if (!cis) {
		log(LOG_ERR, "not enough memory");
		exit(1);
	}
	while ((c = getopt(argc, argv, "i:s:p:d:u:")) != EOF) {

		switch (c) {
		case 'i':
			interval = atof(optarg);
			if (interval == 0.) {
				log(LOG_ERR, "interval cannot be set to zero. reset to default %5.2f [s]", DEF_INTERVAL);
				interval = DEF_INTERVAL;
			}
			break;
		case 's':
			cis->handle = cisConnect(optarg);
			if (!cis->handle) {
				log(LOG_ERR, "cannot connect cis server %s", optarg);
				exit(1);
			}
			cis->name = strdup(optarg);
			break;
                case 'p':
			record.path = strdup(optarg);
			break;
                case 'd':
			record.density = atoi(optarg);
			break;
                case 'u':
			sscanf(optarg, "%hd", &uid);
			break;
                case ':':
                case '?':
                        goto arg_error;
                }
        }
	if (!cis->handle) {
		cis->handle = cisConnect(myhostname);
		if (!cis->handle) {
			log(LOG_ERR, "cannot connect cis server %s", myhostname);
			exit(1);
		}
		cis->name = strdup(myhostname);
	}

	hlist = cisHostList(cis->handle, 0);
	if (!hlist)
		return;

	for (h = hlist; strlen(*h); h++) {
		host = add_host(*h);

		host->name = *h;
		host->interval = SOCK_INTERVAL;

		if (!cis->handle || !(hi = cisHostInfo(cis->handle, *h, ""))) {
			host->hostinfo->status = HOST_UNKNOWN;
			host->hostinfo->addr = host->hostaddr;
			continue;
			}
		*host->hostinfo = *hi;
		free (hi);
	}
	nhosts = h - hlist;

	cis->htab = calloc(nhosts,sizeof(struct cis_hostinfo *));
	if (!cis->htab) {
		log(LOG_ERR, "not enough memory");
		exit(1);
	}
	cis->nhosts = nhosts;

	for (host = hostlist, i = 0; i < nhosts; i++, host = host->next)
		cis->htab[i] = host->hostinfo;

	return;

arg_error:
	log(LOG_ERR, "usage: cis_rec [-s cis_server] [-i interval] [-p path] [-d density] [-u uid]");
	exit(1);
}

/*
 * Function for comparing two timeval structures.
 * It returns an integer less than, equal to, or greater than zero if t1 is
 * found, respectively, to be less than, to match, or be greater than t2.
 */
int tvalcmp(struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

void tvaladd(struct timeval *t, struct timeval inc)
{
        t->tv_sec  += inc.tv_sec;
        t->tv_usec += inc.tv_usec;
        if (t->tv_usec > 1000000L) {
                t->tv_usec -= 1000000L;
                t->tv_sec++;
        }
}

void tvalsub(struct timeval *t, struct timeval dec)
{
        t->tv_sec -= dec.tv_sec;

        if (t->tv_usec < dec.tv_usec) {
                t->tv_sec--;
                t->tv_usec -= 1000000L - dec.tv_usec;
        } else
                t->tv_usec -= dec.tv_usec;
}

struct host_info *find_host(struct in_addr addr)
{
        struct host_info *h;
        struct in_addr *a;

	for (h = hostlist; h; h = h->next)
		for (a = h->addrlist; a->s_addr ; a++)
                        if (a->s_addr == addr.s_addr)
                                return h;
        return NULL;
}

inline void host_error (struct host_info *host, int err)
{
        if (err > HOST_NOT_AVAILABLE) {
		cisDisconnect(cis->handle);
		cis->handle = NULL;
        }
}

void allarm_handler (int sig)
{
        struct host_info *h;
        static struct cis_procinfo *ptab = NULL;
        static struct cis_sockinfo *stab = NULL;
        static struct cis_netdevinfo *dtab = NULL;
        struct cis_hostinfo **new_htab;
        int len;
        struct timeval current_time;
        char *changes = NULL;
        int chlen = 0;
        
        gettimeofday (&current_time, NULL);

	if (!cis->handle)
		cis->handle = cisConnect (cis->name);

        /*
         * Archive previous host information.
         */
        for (h = hostlist; h; h = h->next) {
                if (h->hostinfo) {
                        *h->old_hostinfo = *h->hostinfo;
                        continue;
                }
		h->hostinfo = cisHostInfo (cis->handle, h->name, "");
                if (h->hostinfo) {
                        new_htab = realloc (cis->htab, (cis->nhosts+1)*sizeof(struct cis_hostinfo *));
                        if (!new_htab) {
                                free (h->hostinfo);
                                h->hostinfo = NULL;
                                continue;
                        }

                        cis->htab = new_htab;
                        cis->htab[cis->nhosts] = h->hostinfo;
                        cis->nhosts++;
                }
        }

        /*
         * Update information about hosts.
         */
	if (cis->handle)
		cisUpdateHostsInfo (cis->handle, CIS_ALL, cis->htab, cis->nhosts);

        /*
         * Host hostinfo
         */
        for (h = hostlist; h; h = h->next) {
		if (h->hostinfo_saved_bytes >= record.density) {
                        cisSaveHostinfo (record.file, h->hostaddr, h->hostinfo);
                        h->hostinfo_saved_bytes = 0;
                } else {
                        chlen = cisHostinfoPrepareChanges (h->old_hostinfo, h->hostinfo, &changes);
                        cisSaveHostinfoChanges (record.file, h->hostaddr, changes, chlen);
                        h->hostinfo_saved_bytes += chlen;
                        free (changes);
                }

                h->time = current_time;
        }

        /*
         * Processes
         */
        for (h = hostlist; h; h = h->next) {
		ptab = NULL;
                
		if (cis->handle) {
			len = cisProcessList (cis->handle, &ptab, h->hostaddr, uid);
			if (len < 0) {
				host_error (h, len);
				len = 0;
			}
		} else
			len = 0;

                if (h->proc_saved_bytes >= record.density) {
			cisSaveProcinfo (record.file, h->hostaddr, ptab, len);
                        h->proc_saved_bytes = 0;
                } else {
                        chlen = cisProcPrepareChanges (h->proctab, h->nproc, ptab, len, &changes);
                        if (chlen) {
                                cisSaveProcChanges (record.file, h->hostaddr, changes, chlen);
                                h->proc_saved_bytes += chlen;
                                free (changes);
                        }
                }

                free (h->proctab);
                h->proctab = ptab;
                h->nproc = len;
        }

        /*
         * Sockets
         */
        for (h = hostlist; h; h = h->next) {
                stab = NULL;
                len = 0;

                if (cis->handle &&
                    (len = cisSocketList (cis->handle, &stab,
                                          h->hostaddr, uid, h->interval)) < 0) {
                        host_error (h, len);
                        len = 0;
                }

                if (h->sock_saved_bytes >= record.density) {
                        cisSaveSockinfo (record.file, h->hostaddr, stab, len);
                        h->sock_saved_bytes = 0;
                } else {
                        chlen = cisSockPrepareChanges (h->socktab, h->nsock, stab, len, &changes);
                        cisSaveSockChanges (record.file, h->hostaddr, changes, chlen);
                        h->sock_saved_bytes += chlen;
                        free (changes);
                }

                free (h->socktab);
                h->socktab = stab;
                h->nsock = len;
        }
        
        /*
         * Network devices
         */
        for (h = hostlist; h; h = h->next) {
                dtab = NULL;
                len = 0;
                
                if (cis->handle &&
                    (len = cisNetdevList (cis->handle, &dtab,
                                          h->hostaddr, h->interval)) < 0) {
                        host_error (h, len);
                        len = 0;
                }

                if (h->dev_saved_bytes >= record.density) {
                        cisSaveNetdevinfo (record.file, h->hostaddr, dtab, len);
                        h->dev_saved_bytes = 0;
                } else {
                        chlen = cisNetdevPrepareChanges (h->devtab, h->ndev, dtab, len, &changes);
                        cisSaveNetdevChanges (record.file, h->hostaddr, changes, chlen);
                        h->dev_saved_bytes += chlen;
                }
                
                free (h->devtab);
                h->devtab = dtab;
                h->ndev = len;
        }
}

void cleanup (int sig)
{
        fclose (record.file);
#ifndef DEBUG
        closelog();
#endif
        exit (0);
}

int main (int argc,char *argv[])
{
        struct sigaction allarm_action;
        struct itimerval it;
        time_t ct;
        struct tm *tmp;
        char buff[MAXHOSTNAMELEN+32], *newname;
        struct host_info *h;
        int len;

        myuid = getuid();

        setlocale(LC_ALL, "");
        setlocale(LC_NUMERIC, "C");

	if (!myuid)
		openlog("cis_rec", LOG_CONS, LOG_DAEMON);

        /*
         * Initialization of signal action structures.
         */
        allarm_action.sa_handler = allarm_handler;
        sigemptyset (&allarm_action.sa_mask);
        allarm_action.sa_flags   = 0;

        if (gethostname((char *)& myhostname, MAXHOSTNAMELEN) < 0) {
                log(LOG_ERR, "cannot get my hostname");
                exit(1);
	}

        getparams(argc, argv);

        ct = time(NULL);
        tmp = localtime(&ct);

        len = strftime(buff, MAXHOSTNAMELEN+32, "%Y.%m.%d_%H:%M:%S", tmp);
        sprintf(buff+len, "_%s.rec", myhostname);
        newname = realloc(record.filename, strlen(buff)+strlen(record.path)+1);
	if (!newname) {
		log(LOG_ERR, "not enough memory");
		exit(1);
	}
        record.filename = newname;

        strcpy(newname, record.path);
        strcat(newname, buff);
                
        record.file = fopen(record.filename, "w+");
	if (!record.file) {
		log(LOG_ERR, "cannot create the file '%s'", record.filename);
		exit(1);
	}
	for (h = hostlist; h; h = h->next) {
                h->hostinfo_saved_bytes = record.density;
                h->proc_saved_bytes     = record.density;
                h->sock_saved_bytes     = record.density;
                h->dev_saved_bytes      = record.density;
        }

        /*
         * Let's install handler for SIGALRM.
         */
        sigaction(SIGALRM, &allarm_action, NULL);

        /*
         * We need to unregister rpc service before terminating ...
         */
        signal(SIGINT, cleanup);
        signal(SIGTERM, cleanup);

        
        /*
         * Set interval and reset timer.
         */
        sectotval(it.it_interval, interval);
        sectotval(it.it_value   , interval);

#ifndef DEBUG
        daemonize();
#endif

        cisSaveHeader(record.file);

        setitimer(ITIMER_REAL, &it, NULL);

        allarm_handler(0);

        while (1)
                select(0, NULL, NULL, NULL, NULL);

        exit(0);
}

