/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_print: prints binary record file in text form to stdout.
 */

#include <stdio.h>
#include <netdb.h>
#include <locale.h>

#include "cis.h"
#include "cis_clnt.h"

char *names[] = {
        "",
        "hostinfo full    ",
        "hostinfo changes ",
        "procinfo full    ",
        "procinfo changes ",
        "sockinfo full    ",
        "sockinfo changes ",
        "ndevinfo full    ",
        "ndevinfo changes ",
};

char *sock_types[] = {
        "",
        "stream  ",
        "datagram",
};

extern int h_errno;


struct host_info {
        struct in_addr addr;
        char name[MAXHOSTNAMELEN];

        struct cis_hostinfo *hostinfo;

        int nproc;
        struct cis_procinfo *proctab;
        
        int nsock;
        struct cis_sockinfo *socktab;

        int ndev;
        struct cis_netdevinfo *devtab;
        
} *hosttab = NULL;
int hosttab_length = 0;

struct host_info *findhost (struct in_addr addr)
{
        struct host_info *h;

        for (h = hosttab; h < hosttab+hosttab_length; h++)
                if (h->addr.s_addr == addr.s_addr)
                        return h;
        return NULL;
}

#define SI_PRINT(el, fmt) printf ("        "fmt" \n", n->el)
#define SI_CHPRINT(el, fmt) if (!o || o->el != n->el) SI_PRINT (el, fmt)

void print_hostinfo (struct cis_hostinfo *o, struct cis_hostinfo *n)
{
        if (!n) {
                printf ( "        not available\n");
                return;
        }
        if (o && !memcmp (o, n, sizeof (struct cis_hostinfo))) {
//                printf ( "        no change\n");
                return;
        }
        
        SI_CHPRINT (idle,     "Idle time              %ld");
        SI_CHPRINT (procnum,  "Number of processes    %d");
        SI_CHPRINT (loads[0], "Average load in 1  min %5.3f");
        SI_CHPRINT (loads[1], "Average load in 5  min %5.3f");
        SI_CHPRINT (loads[2], "Average load in 15 min %5.3f");
        SI_CHPRINT (totalram, "Total memory           %ld");
        SI_CHPRINT (freeram,  "Free memory            %ld");
        SI_CHPRINT (sharedram,"Shared memory          %ld");
        SI_CHPRINT (bufferram,"Memory buffers         %ld");
        SI_CHPRINT (cachedram,"Cached memory          %ld");
        SI_CHPRINT (freeram,  "Free memory            %ld");
        SI_CHPRINT (totalswap,"Total swap             %ld");
        SI_CHPRINT (freeswap, "Free swap              %ld");

        if (!o || o->CPU_available != n->CPU_available)
                printf ( "        CPU available [%%]      %4.2f \n", n->CPU_available / 1e2);
}

void print_procinfo_full (struct cis_procinfo *tab, int len)
{
        struct cis_procinfo *p;

        for (p = tab; p < tab + len; p++) {
                printf ("        "
                        "%-16s %5d %5d %5d %3ld %9ld %9ld %4.2f",
                        p->cmd, p->pid, p->ppid, p->uid, p->priority, p->rss, p->vm,
                        p->pCPU / 1e4);

                
                printf ("\n");
        }
}

#define PI_PRINT(el, fmt) printf ("        Changed    %5d %-16s "fmt" \n", q->pid, q->cmd, q->el)
#define PI_CHPRINT(el, fmt) if (p->el != q->el) PI_PRINT (el, fmt)

void print_procinfo_changes (struct cis_procinfo *oldtab, int nold, struct cis_procinfo *newtab, int nnew)
{
        struct cis_procinfo *p;
        struct cis_procinfo *q;

        p = oldtab;
        q = newtab;
        
        while (p < oldtab+nold || q < newtab+nnew) {

                if (q < newtab+nnew && (p == oldtab+nold || p->pid > q->pid)) {
                        printf ("        "
                                "New        %5d %-16s %5d %5d %3ld %9ld %9ld %4.2f\n",
                                q->pid, q->cmd, q->ppid, q->uid, q->priority, q->rss, q->vm,
                                q->pCPU / 1e4);
                        q++;
                        continue;
                }
                if (p < oldtab+nold && (q == newtab+nnew || p->pid < q->pid)) {
                        printf ("        "
                                "Terminated %5d %-16s %5d %5d\n",
                                p->pid, p->cmd, p->ppid, p->uid);
                        p++;
                        continue;
                }

                PI_CHPRINT(ppid,      "parent pid   %-5d");
                if (strncmp (p->cmd, q->cmd, CMD_LENGTH))
                        PI_PRINT(cmd, "command      %s");

                PI_CHPRINT(priority,  "priority     %-5ld");
                PI_CHPRINT(utime,     "user time    %-5ld");
                PI_CHPRINT(stime,     "system time  %-5ld");
                PI_CHPRINT(rss,       "resid.memory %-9ld");
                PI_CHPRINT(vm,        "virt.memory  %-9ld");

                if (p->pCPU != q->pCPU)
                        printf ("        "
                                "Changed    %5d %-16s CPU usage[%%] %4.2f\n",
                                q->pid, q->cmd, q->pCPU / 1e2);
                        
                p++;
                q++;
        }
}

void print_sockinfo_full (struct cis_sockinfo *tab, int len)
{
        struct cis_sockinfo *s;

        for (s = tab; s < tab + len; s++) {
                printf ("        "
                        "%-8s %04X %08X:%04X %5d %5d %12ld %12ld",
                        sock_types[s->type], s->sport, s->daddr.s_addr, s->dport,
                        s->pid, s->uid, s->sent, s->rcvd);
                
                printf ("\n");
        }
}

int sock_cmp (const void *l, const void *r)
{
        struct cis_sockinfo *left  = (struct cis_sockinfo *) l;
        struct cis_sockinfo *right = (struct cis_sockinfo *) r;
        int res;
        
        if ((res = left->saddr.s_addr - right->saddr.s_addr))
                return (res);
        if ((res = left->type - right->type))
                return (res);
        if ((res = left->sport - right->sport))
                return (res);
        if ((res = left->daddr.s_addr - right->daddr.s_addr))
                return (res);
        if ((res = left->dport - right->dport))
                return (res);

        return (0);
}

void print_sockinfo_changes (struct cis_sockinfo *oldtab, int nold, struct cis_sockinfo *newtab, int nnew)
{
        struct cis_sockinfo *s;
        struct cis_sockinfo *t;

        s = oldtab;
        t = newtab;
        
        while (s < oldtab+nold || t < newtab+nnew) {

                if (t < newtab+nnew &&
                    (s == oldtab+nold || sock_cmp (s, t) > 0)) {
                        printf ("        "
                                "New     %-8s %04X %08X:%04X %5d %5d %12ld %12ld\n",
                                sock_types[t->type], t->sport, t->daddr.s_addr, t->dport,
                                t->pid, t->uid, t->sent, t->rcvd);

                        t++;
                        continue;
                }
                if (s < oldtab+nold &&
                    (t == newtab+nnew || sock_cmp (s, t) < 0)) {
                        printf ("        "
                                "Closed  %-8s %04X %08X:%04X %5d %5d\n",
                                sock_types[s->type], s->sport, s->daddr.s_addr, s->dport,
                                s->pid, s->uid);
                        s++;
                        continue;
                }

                if (s->sent != t->sent)
                        printf ("        "
                                "Changed %-8s %04X %08X:%04X send rate      %-12ld\n",
                                sock_types[t->type], t->sport, t->daddr.s_addr, t->dport, t->sent);
                if (s->rcvd != t->rcvd)
                        printf ("        "
                                "Changed %-8s %04X %08X:%04X receive rate   %-12ld\n",
                                sock_types[t->type], t->sport, t->daddr.s_addr, t->dport, t->rcvd);

                s++;
                t++;
        }
}

#define DI_PRINT(el, fmt) printf ("        Changed    %-15s "fmt" \n", n->name, n->el)
#define DI_CHPRINT(el, fmt) if (o->el != n->el) DI_PRINT (el, fmt)

void print_devinfo_full (struct cis_netdevinfo *tab, int len)
{
        struct cis_netdevinfo *d;

        for (d = tab; d < tab + len; d++) {
                printf ("        "
                        "%-15s %-4s rx:%10ld tx:%10ld coll:%10ld",
                        d->name,
                        d->status ? "up" : "down",
                        d->rx_bytes, d->tx_bytes, d->collisions);
                printf ("\n");
        }
}

void print_devinfo_changes (struct cis_netdevinfo *oldtab, int nold, struct cis_netdevinfo *newtab, int nnew)
{
        struct cis_netdevinfo *o;
        struct cis_netdevinfo *n;

        o = oldtab;
        n = newtab;

        while (o < oldtab+nold || n < newtab+nnew) {

                if (n < newtab+nnew &&
                    (o == oldtab+nold || strcmp (o->name, n->name) > 0)) {
                        printf ("        "
                                "New     %-15s %-4s rx:%10ld tx:%10ld coll:%10ld\n",
                                n->name,
                                n->status ? "up" : "down",
                                n->rx_bytes, n->tx_bytes, n->collisions);

                        n++;
                        continue;
                }
                if (o < oldtab+nold &&
                    (n == newtab+nnew || strcmp (o->name, n->name) < 0)) {
                        printf ("        "
                                "Closed  %-15s rx:%10ld tx:%10ld coll:%10ld\n",
                                o->name,
                                o->rx_bytes, o->tx_bytes, o->collisions);
                        o++;
                        continue;
                }

                if (o->status != n->status)
                        printf ("        "
                                "Changed     %-15s status       %-4s\n",
                                n->name, n->status ? "up" : "down");

                DI_CHPRINT(rx_bytes,  "Receive rate %-5ld");
                DI_CHPRINT(tx_bytes,  "Send rate    %-5ld");
                DI_CHPRINT(collisions,"collisions   %-5ld");

                o++;
                n++;
        }
}

int main (int argc, char *argv[])
{
        struct cis_header header;
        struct record_info r;
        FILE *f;
        time_t t;
        struct tm *tmp;
        struct hostent *ent;
        struct host_info *h;
        struct cis_hostinfo *hi;
        struct cis_procinfo *ptab;
        struct cis_sockinfo *stab;
        struct cis_netdevinfo *dtab;
        char buff[32];
        int len;
        char *c;

        setlocale (LC_ALL, "");
        setlocale (LC_NUMERIC, "C");
        
        if (argc != 2) {
                printf ("usage: cis_print record_file\n");
                exit (1);
        }

        if (!(f = fopen (argv[1], "r"))) {
                printf ("cannot open record file\n");
                exit (1);
        }

        if (cisReadHeader (f, &header) < 0 || header.version != CIS_RECORD_VERSION) {
                printf ("incorrect record file\n");
                exit (1);
        }

        while (cisReadRecord (f, &r) > -1) {

                t = r.time.tv_sec;
                tmp = localtime (&t);

                strftime (buff, 32, "%Y.%m.%d %H:%M:%S", tmp);

                h = findhost (r.host);
                if (!h) {
                        hosttab = realloc (hosttab, (hosttab_length+1) * sizeof (struct host_info));
                        if (!hosttab) {
                                printf ("not enough memory\n");
                                exit (1);
                        }
                        h = hosttab+hosttab_length;
                        memset (h, 0, sizeof (struct host_info));
                        
                        ent = gethostbyaddr ((char*)&r.host, sizeof(r.host), AF_INET);
                        if (!ent)
                                sprintf (h->name, "unknown");
                        else  {
                                c = strchr(ent->h_name, '.');
                                strncpy (h->name, ent->h_name, c-ent->h_name);
                        }
                        h->addr = r.host;
                        h->hostinfo = calloc (1, sizeof (struct cis_hostinfo));
                        hosttab_length++;
                }

                switch (r.type) {

                case CIS_HOSTINFO:
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        print_hostinfo (NULL, (struct cis_hostinfo *) r.data);

                        free (h->hostinfo);
                        h->hostinfo = (struct cis_hostinfo *) r.data;
                        break;

                case CIS_HOSTINFO_CHANGES:
                        if (!r.length) break;
                        
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        cisHostinfoApplyChanges (h->hostinfo, r.data, r.length, &hi);
                        print_hostinfo (h->hostinfo, hi);
                        free (h->hostinfo);
                        h->hostinfo = hi;
                        free (r.data);
                        break;

                case CIS_PROCINFO:
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);

                        free (h->proctab);
                        
                        h->proctab = (struct cis_procinfo *) r.data;
                        h->nproc = r.length / sizeof (struct cis_procinfo);
                        print_procinfo_full (h->proctab, h->nproc);
                        break;
                        
                case CIS_PROC_CHANGES:
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        len = cisProcApplyChanges (h->proctab, h->nproc, r.data, r.length, &ptab);
                        print_procinfo_changes (h->proctab, h->nproc, ptab, len);
                        free (h->proctab);
                        h->proctab = ptab;
                        h->nproc   = len;
                        if (r.length)
                                free (r.data);
                        break;

                case CIS_SOCKINFO:
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        free (h->socktab);
                        h->socktab = (struct cis_sockinfo *) r.data;
                        h->nsock = r.length / sizeof (struct cis_sockinfo);
                        print_sockinfo_full (h->socktab, h->nsock);
                        break;

                case CIS_SOCK_CHANGES:
                        if (!r.length) break;
                        
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        len = cisSockApplyChanges (h->socktab, h->nsock, r.data, r.length, &stab);
                        print_sockinfo_changes (h->socktab, h->nsock, stab, len);
                        free (h->socktab);
                        h->socktab = stab;
                        h->nsock   = len;
                        if (r.length)
                                free (r.data);
                        break;

                case CIS_NETDEVINFO:
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        free (h->devtab);
                        h->devtab = (struct cis_netdevinfo *) r.data;
                        h->ndev = r.length / sizeof (struct cis_netdevinfo);
                        print_devinfo_full (h->devtab, h->ndev);
                        break;

                case CIS_NETDEV_CHANGES:
                        if (!r.length) break;
                        
                        printf ("%s %s %s\n", buff, h->name, names[(int) r.type]);
                        len = cisNetdevApplyChanges (h->devtab, h->ndev, r.data, r.length, &dtab);
                        print_devinfo_changes (h->devtab, h->ndev, dtab, len);
                        free (h->devtab);
                        h->devtab = dtab;
                        h->ndev   = len;
                        if (r.length)
                                free (r.data);
                        break;
                }
                fflush (stdout);
        }

        exit (0);
}