/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_merge: merges multiple record files into one.
 */

#include <stdio.h>
#include <netdb.h>

#include "cis.h"
#include "cis_clnt.h"

struct host_info {
        int flag;
        struct in_addr addr;
};

int nfiles;

struct file_info {
        int available;
        char *name;
        FILE *fp;
        struct record_info rec;
        int nhosts;
        struct host_info *hosts;
} *filetab;

int tvalcmp (struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

struct host_info *find_host (struct file_info *fi, struct in_addr addr)
{
        struct host_info *h;

        for (h = fi->hosts; h < fi->hosts+fi->nhosts; h++)
                if (h->addr.s_addr == addr.s_addr)
                        return h;
        return NULL;
}

void disable_host (struct in_addr addr)
{
        struct file_info *fi;
        struct host_info *h;

        for (fi = filetab; fi < filetab+nfiles; fi++)
                if ((h = find_host (fi, addr)))
                        h->flag = FALSE;
}


struct host_info *add_host (struct file_info *fi)
{
        struct host_info *h;
        
        fi->hosts = realloc (fi->hosts, (fi->nhosts+1) * sizeof (struct host_info));
        if (!fi->hosts) {
                printf ("not enough memory\n");
                exit (1);
        }
        h = fi->hosts+fi->nhosts;
        h->addr = fi->rec.host;
        h->flag = TRUE;
        fi->nhosts++;

        return h;
}

struct file_info *next_record (void)
{
        struct file_info *fi, *oldest = NULL;

        for (fi = filetab; fi < filetab+nfiles; fi++) {
                if (!fi->available) continue;

                if (!oldest || tvalcmp (oldest->rec.time, fi->rec.time) > 0)
                        oldest = fi;
        }

        return oldest;
}

void rec_copy (FILE *f1, FILE *f2, int len)
{
        static char buff[1024];

        while (len) {
                if (len < 1024) {
                        fread  (buff, len, 1, f2);
                        fwrite (buff, len, 1, f1);
                        break;;
                }
                fread  (buff, 1024, 1, f2);
                fwrite (buff, 1024, 1, f1);
                len -= 1024;
        }
}

void rec_skip (FILE *f, int len)
{
        fseek (f, len, SEEK_CUR);
}

FILE *outfile;
int gap = FALSE;
struct cis_hostinfo hinfo;

void save_empty_info (struct file_info *fi)
{
	struct host_info *h;

	for (h = fi->hosts; h < fi->hosts+fi->nhosts; h++) {
		if (!h->flag) continue;

		cisSaveRecord (outfile, fi->rec.time,
			       CIS_HOSTINFO, h->addr, (char *) &hinfo,
			       sizeof (struct cis_hostinfo), 1, sizeof (struct cis_hostinfo));
	}
	gap = !gap;
}

int main (int argc, char *argv[])
{
        struct file_info *fi;
        char **arg;
        struct cis_header header;
	struct host_info *h;

        if (argc < 4) {
                printf ("usage: cis_merge outputfile file1 file2 [file3] ...\n");
                exit (1);
        }

        if (!(outfile = fopen (argv[1], "w+"))) {
                printf ("cannot open file %s for writing.\n", argv[1]);
                exit (1);
        }
        cisSaveHeader (outfile);
        
        nfiles  = argc - 2;
        filetab = calloc (nfiles, sizeof (struct file_info));

	memset (&hinfo, 0, sizeof (struct cis_hostinfo));
	hinfo.status = HOST_NOT_AVAILABLE;

        for (fi = filetab, arg = argv+2; fi < filetab + nfiles; fi++, arg++) {
                if (!(fi->fp = fopen (*arg, "r"))) {
                        printf ("cannot open file %s for reading.\n", *arg);
                        exit (1);
                }
                fi->name = *arg;
                
                if (cisReadHeader (fi->fp, &header) < 0 || header.version != CIS_RECORD_VERSION) {
                        printf ("incorrect record file %s \n", *arg);
                        exit (1);
                }
                fi->available = cisRecordInfo (fi->fp, &fi->rec) >= 0;
        }

        while ((fi = next_record ())) {

                if (!(h = find_host (fi, fi->rec.host))) {
                        disable_host (fi->rec.host);
                        h = add_host (fi);
                }
                        
		if (gap)
			save_empty_info(fi);

		if (!h->flag)
                        rec_skip (fi->fp, fi->rec.length+CIS_RECHDRLEN);
                else
                        rec_copy (outfile, fi->fp, fi->rec.length+CIS_RECHDRLEN);

                fi->available = cisRecordInfo (fi->fp, &fi->rec) >= 0;

		if (!fi->available)
			save_empty_info(fi);
	}

        printf ("Ok.\n");
        
        exit (0);
}