/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * cis_cut: extracts records for one or more hosts and time interval.
 */

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <netdb.h>
#include <locale.h>
#include <time.h>

#include "cis.h"
#include "cis_clnt.h"

extern int h_errno;

extern int save_size[];
extern int struct_size[];

struct host_info {
        struct in_addr addrlist[MAX_ADDR_NUM];
#define hostaddr addrlist[0]
        char *name;

        void *sysinfo;

        int nproc;
        void *proctab;
        
        int nsock;
        void *socktab;

        int ndev;
        void *devtab;
        struct host_info *next;
	
} *hostlist = NULL;

unsigned long start_time = 0;
unsigned long end_time = 0;
struct record_info *table;
int table_length;
char *filename;
FILE *input;
char *outfilename;
FILE *output;

int all_hosts = TRUE;

struct host_info *findhost (struct in_addr addr)
{
        struct host_info *h;
        struct in_addr *a;

	for (h = hostlist; h; h = h->next)
		for (a = h->addrlist; a->s_addr ; a++)
                        if (a->s_addr == addr.s_addr)
                                return h;
	return NULL;
}

int tvalcmp (struct timeval t1, struct timeval t2)
{
        if (t1.tv_sec < t2.tv_sec)
                return -1;
        if (t1.tv_sec > t2.tv_sec)
                return 1;

        if (t1.tv_usec < t2.tv_usec)
                return -1;
        if (t1.tv_usec > t2.tv_usec)
                return 1;

        return 0;
}

unsigned long get_time (char *buff)
{
        struct tm *tmp, t = {0, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX, INT_MAX};
        char *c = buff;
        time_t sec;
        
        sec = time(NULL);
        tmp = localtime (&sec);

        c = strptime (c, "%R", &t);
        if (!c) goto next;

        if (*c == ':')
                c = strptime (c, ":%S", &t);
        if (!c) goto next;

        c = strptime (c, " %d/%m", &t);
        if (!c) goto next;

        if (*c == '/')
                c = strptime (c, "/%Y", &t);

    next:
        if (t.tm_min == INT_MAX || t.tm_hour == INT_MAX) {
                printf ("cannot get time from %s.\n", buff);
                return 0;
        }
        if (t.tm_mday == INT_MAX) t.tm_mday = tmp->tm_mday;
        if (t.tm_mon  == INT_MAX) t.tm_mon  = tmp->tm_mon;
        if (t.tm_year == INT_MAX) t.tm_year = tmp->tm_year;

        return mktime (&t);
}

struct host_info *addhost(struct in_addr *addr, char *name)
{
	struct host_info *h, *host;
	struct hostent *ent;
	int i;

	if (name)
		ent = gethostbyname (name);
	else
		ent = gethostbyaddr ((char *) addr, sizeof (*addr), AF_INET);

	if (!ent)
		return NULL;

	h = calloc (1, sizeof (struct host_info));
	if (!h) {
		printf ("not enough memory\n");
		return NULL;
	}

	h->name = strdup(ent->h_name);

	for (i = 0; ent->h_addr_list[i]; i++)
		memcpy (&h->addrlist[i], ent->h_addr_list[i], ent->h_length);
	h->addrlist[i].s_addr = 0;

	h->sysinfo = calloc (1, sizeof (struct cis_hostinfo));

	if (!hostlist) {
		hostlist = h;
		return h;
	}
	
	for (host = hostlist; host->next; host = host->next)
		;

	host->next = h;

	return(h);
}

void parse_args (int argc, char **argv)
{
        int c, i;
        int option_index = 0;
        static struct option long_options[] = {
                {"from", 1, 0, 'f'},
                {"to", 1, 0, 't'},
                {0, 0, 0, 0},
        };
        if (argc < 4)
                goto arg_error;
        outfilename = argv[1];

        while ((c = getopt_long (argc-1, argv+1, "f:t:", long_options, &option_index)) != EOF) {

                switch (c) {
                case 'f':
                        start_time = get_time (optarg);
                        break;

                case 't':
                        end_time = get_time (optarg);
                        break;

                case ':':
                case '?':
                        goto arg_error;
                }
        }

        optind++;
        
        if (!argv[optind])
                goto arg_error;
        filename = argv[optind];

	for (i = optind+1; i < argc; i++) {
		if (!addhost(NULL, argv[i])) {
			printf ("cannot add host %s\n", argv[i]);
			exit(1);
		}
		all_hosts = FALSE;
	}
        return;

    arg_error:
        printf ("usage: cis_cut outputfile [--from time] [--to time] record_file host1 [host2] ...\n");
        exit(1);
}

void rec_copy (FILE *f1, FILE *f2, int len)
{
        static char buff[1024];

        len += CIS_RECHDRLEN;
        
        while (len) {
		if (len < 1024) {
                        fread  (buff, len, 1, f2);
                        fwrite (buff, len, 1, f1);
			break;
                }
		fread  (buff, 1024, 1, f2);
                fwrite (buff, 1024, 1, f1);
                len -= 1024;
        }
}

int rec_skip (FILE *f, int len)
{
        return (fseek (f, len + CIS_RECHDRLEN, SEEK_CUR));
}

int parse_file (FILE *f)
{
        struct record_info tmp;
        long offset;
        int ret;

        while (!feof (f)) {

                offset = ftell (f);

                ret = cisRecordInfo (f, &tmp);
                if (ret < 0)
                        break;

                if (rec_skip (input, tmp.length) == -1)
                        break;

		tmp.host_ptr = findhost (tmp.host);

		if (all_hosts && !tmp.host_ptr)
			tmp.host_ptr = addhost (&tmp.host, NULL);

		if (!tmp.host_ptr)
			continue;

                table = realloc (table, (table_length+1) * sizeof (struct record_info));
                if (!table)
                        goto error;

                memcpy (&table[table_length], &tmp, sizeof (struct record_info));
                table_length++;
        }

        return TRUE;
    error:
        return FALSE;
}

void get_state (int type, struct host_info *host, void **objptr, int *lenptr, struct record_info *bound)
{
        struct record_info *r;
	void *tab, *newtab;
        int len;

        *objptr = NULL;
        if (lenptr)
                *lenptr = 0;

        for (r = bound-1; r >= table; r--)
                if (r->host_ptr == host && r->type == type)
                        break;

        if (r < table)
                return;

        fseek (input, r->offset, SEEK_SET);
        if (cisReadRecord (input, r) < 0)
                return;

        tab = r->data;
        len = r->length / struct_size[type];

        for (; r < bound; r++) {
                if (r->host_ptr != host || r->type != type+1)
                        continue;

                fseek (input, r->offset, SEEK_SET);
                if (cisReadRecord (input, r) < 0)
                        continue;

                len = cisApplyChanges (r->type, tab, len, r->data, r->length, &newtab);
                free (tab);
                if (r->length)
                        free (r->data);
                tab = newtab;
        }

        *objptr = tab;
        if (lenptr)
                *lenptr = len;

        return;
}


int main (int argc, char *argv[])
{
        struct cis_header header;
        struct record_info *r;
        struct host_info *h;
        
        setlocale (LC_ALL, "");
        setlocale (LC_NUMERIC, "C");

        parse_args (argc, argv);

        if (!(input = fopen (filename, "r"))) {
                printf ("cannot open record file\n");
                exit (1);
        }

        if (cisReadHeader (input, &header) < 0 || header.version != CIS_RECORD_VERSION) {
                printf ("incorrect record file\n");
                exit (1);
        }

        printf ("parsing record file...");
        if (!parse_file (input)) {
                printf ("incorrect record file\n");
                exit (1);
        }
        printf ("ok.\n");

        if (!table ||
            (start_time && start_time > table[table_length-1].time.tv_sec) ||
            (end_time && end_time < table->time.tv_sec)) {
                printf ("no records available.\n");
                exit (0);
        }

        if (start_time < table->time.tv_sec)
                start_time = table->time.tv_sec;

        if (!end_time || end_time > table[table_length-1].time.tv_sec)
                end_time = table[table_length-1].time.tv_sec+1;

        if (!(output = fopen (outfilename, "w+"))) {
                printf ("cannot open output file %s\n", outfilename);
                exit (1);
        }
        printf ("Saving records to %s.\n", outfilename);
        cisSaveHeader (output);

        r = table;

        if (start_time == r->time.tv_sec)
                goto copying;

        for (; r < table+table_length && r->time.tv_sec < start_time; r++)
                ;

        for (h = hostlist; h; h = h->next) {
                get_state (CIS_HOSTINFO, h, &h->sysinfo, NULL, r);
                cisSaveRecord (output, r->time, CIS_HOSTINFO, h->hostaddr,
			       h->sysinfo, struct_size[CIS_HOSTINFO], 1,
                               save_size[CIS_HOSTINFO]);
	}
        for (h = hostlist; h; h = h->next) {
                get_state (CIS_PROCINFO, h, &h->proctab, &h->nproc, r);
                cisSaveRecord (output, r->time, CIS_PROCINFO, h->hostaddr,
                               h->proctab, struct_size[CIS_PROCINFO], h->nproc,
                               save_size[CIS_PROCINFO]);
	}
        for (h = hostlist; h; h = h->next) {
                get_state (CIS_SOCKINFO, h, &h->socktab, &h->nsock, r);
                cisSaveRecord (output, r->time, CIS_SOCKINFO, h->hostaddr,
                               h->socktab, struct_size[CIS_SOCKINFO], h->nsock,
                               save_size[CIS_SOCKINFO]);
	}
        for (h = hostlist; h; h = h->next) {
                get_state (CIS_NETDEVINFO, h, &h->devtab, &h->ndev, r);
                cisSaveRecord (output, r->time, CIS_NETDEVINFO, h->hostaddr,
                               h->devtab, struct_size[CIS_NETDEVINFO], h->ndev,
                               save_size[CIS_NETDEVINFO]);
        }

    copying:
        for (;r->time.tv_sec < end_time && r < table+table_length; r++) {
                fseek (input, r->offset, SEEK_SET);
                rec_copy (output, input, r->length);
        }
        
        exit (0);
}