/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Socket monitor. Provides CIS server with the information about connections
 * between processes (sockets).
 */

#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/times.h>
#include <unistd.h>
#include <signal.h>
#include <locale.h>
#include <syslog.h>
#include <errno.h>

#include <linux/types.h>
#include <linux/netlink.h>

#include "sockmon.h"

#define FULL_MODE 1

#define HT_LENGTH(a)  (a * 3 / 2)
#define hash(a)    (((a * 3) / 2) % ht_length)
#define isroot()   (!myuid)

#ifndef MY_PRIORITY
#define MY_PRIORITY             -15
#endif

#ifndef DEF_INTERVAL
#define DEF_INTERVAL              1.
#endif

#ifndef DEF_SOCKTAB_LENGTH
#define DEF_SOCKTAB_LENGTH        10
#endif

#ifdef DEBUG
#define syslog(tag, fmt, args...)  printf(fmt"\n" , ## args)
#endif

extern void daemonize (void);
extern int setfdnoblock (int fd);

/* Internal sockmon information */
int mypid;
int myuid;

clock_t        start_ct;

/* Parameters of connection with cis daemon */
char           *srv_name;

/* Connection variables */
int            mode = 0;
int            socketfd;
struct monitor_msg buff = {0, 0, 1};
char           *buffptr       = buff.data;
int            full_msg;
#define bufflen (buffptr - (char *) &buff)

struct monitor_reply reply;

/* Host information */
char           my_hostname[MAXHOSTNAMELEN];
struct in_addr addrlist[MAX_ADDR_NUM];
clock_t        time_ct;
int            ct_per_sec;

/* Socket information */
FILE               *sockfile;
struct cis_sockinfo *socklist;
int                nsock;
int                ht_capacity = DEF_SOCKTAB_LENGTH;
int                ht_length   = HT_LENGTH (DEF_SOCKTAB_LENGTH);
struct cis_sockinfo **hash_table;

void parse_args(int argc, char **argv)
{
        int c;

        while ((c = getopt(argc, argv, "f")) != EOF) {

                switch (c) {
                case 'f':
                        mode |= FULL_MODE;
                        break;
                }
        }
        if (argv[optind])
                srv_name = argv[optind];
}

/*
 * Searching in the hash table. If the requested socket was not found, a
 * pointer to the next free cell is returned.
 */
static inline struct cis_sockinfo **ht_find(struct cis_sockinfo *socket)
{
        static struct cis_sockinfo **hs;
        
        hs = &hash_table[hash(socket->sport)];

        while (*hs && memcmp(*hs, socket, SOCKINFO_HDRLEN))
                if (++hs == hash_table + ht_length)
                        hs = hash_table;

        return (hs);
}

static inline void ht_remove(struct cis_sockinfo **hole)
{
        struct cis_sockinfo **hs = hole;

        *(hole) = NULL;

        for(;;) {
                if (++hs == hash_table + ht_length)
                        hs = hash_table;

                if (!(*hs)) break;

                if (!(*ht_find(*hs)))
                        *hole = *hs, *hs = NULL, hole = hs;
        }
}

struct cis_sockinfo *register_socket(struct cis_sockinfo *socket, struct cis_sockinfo **hs)
{
        struct cis_sockinfo *s, *tmp;

        s = malloc(sizeof (struct cis_sockinfo));
        if (!s)
                return (NULL);

        memcpy(s, socket, SOCKINFO_LEN);

        if (++nsock > ht_capacity) {
                struct cis_sockinfo **new_ht;

                /*
                 * Expand the hash_table.
                 */
                ht_capacity = ht_capacity * 3 / 2;
                ht_length   = HT_LENGTH(ht_capacity);

                new_ht = calloc(ht_length, sizeof (struct cis_sockinfo *));
                if (!new_ht) {
                        free(s);
                        syslog(LOG_WARNING, "not enough memory for hash table");
                        return (NULL);
                }

                free(hash_table);
                hash_table = new_ht;

                for (tmp = socklist; tmp; tmp = tmp->next)
                        *(ht_find(tmp)) = tmp;

                *(ht_find(s)) = s;

        } else
                *hs = s;

        s->flags = CONSISTENT;
        
        list_insert(socklist, s);
        return (s);
}

#define SOCKFLAGS     (MSG_DONTWAIT | MSG_NOSIGNAL)

void buffer_send(void)
{
        buff.time = times(NULL);
        buff.seqnum++;

        send(socketfd, &buff, bufflen, SOCKFLAGS);
        buffptr = buff.data;
        if (buff.flags & MSG_FIRST)
                buff.flags &= ~MSG_FIRST;
}

void buffer_flush(void)
{
        buff.flags |= MSG_LAST;
        buffer_send();
}

void buffer_init(char tag)
{
        if (buffptr > buff.data)
                buffer_flush();

        buff.flags = MSG_FIRST;
        buff.tag = tag;
        buffptr  = buff.data;
}

#define put(ptr, len) \
        {                                           \
               if ((bufflen + len) > MON_BUFFLEN)   \
                      buffer_send();                \
               memcpy(buffptr, ptr, len);           \
                      buffptr += len;               \
        }

#define put_object(obj) put(&obj, sizeof(obj))

void pack_init()
{
        int i, n;

        put_object(start_ct);
        put_object(ct_per_sec);
        for (i = 0; addrlist[i].s_addr; i++)
                ;
        n = i;
        put_object(n);
        for (i = 0; addrlist[i].s_addr; i++)
                put_object(addrlist[i]);

}

void send_init(void)
{
        start_ct = times(NULL);
        buffer_init(SOCKMON_INIT);
        pack_init();
        buffer_flush();
}

#define add_change(el) \
        if (s->el != msg->el) {                           \
               *(chptr++) = OFFSET(cis_sockinfo, el);     \
               *(int *) chptr = msg->el - s->el;          \
               chptr += sizeof(int);                      \
               s->el = msg->el;                           \
        }

#define add_element(el) \
        {                                                 \
               *(chptr++) = OFFSET(cis_sockinfo, el);     \
               memcpy(chptr,  &msg->el, sizeof(msg->el)); \
               memcpy(&s->el, &msg->el, sizeof(msg->el)); \
               chptr += sizeof(msg->el);                  \
        }

void put_changes(struct cis_sockinfo *s, struct cis_sockinfo *msg)
{
        char chbuff[sizeof(struct cis_sockinfo)];
        char *chptr = chbuff;

        memcpy (chbuff, msg, SOCKINFO_HDRLEN);
        chptr += SOCKINFO_HDRLEN;

        if (s->pid      != msg->pid)       add_element(pid);
        if (s->uid      != msg->uid)       add_element(uid);
        if (s->sent     != msg->sent)      add_change(sent);
        if (s->rcvd     != msg->rcvd)      add_change(rcvd);
        
        if (chptr == chbuff+SOCKINFO_HDRLEN)
                return;

        *(chptr++) = 0;
            
        if ((bufflen + (chptr - chbuff)) > MON_BUFFLEN)
                buffer_send();
        put(chbuff, chptr - chbuff);
}

void put_socket(char tag, struct cis_sockinfo *s)
{
        if ((bufflen + SOCKINFO_LEN + 1) > MON_BUFFLEN)
                buffer_send();

        put (s, SOCKINFO_HDRLEN);
        *(buffptr++) = tag;

        put(s+SOCKINFO_HDRLEN, SOCKINFO_LEN-SOCKINFO_HDRLEN);
}

#define put_new(s)        put_socket(1, s)
#define put_closed(s)     put_socket(2, s)

/*
 * Read information about sockets and prepare message for server.
 */
void sock_read(void)
{
        static struct cis_sockinfo sock, *s, **hs;

        if (!(mode&FULL_MODE))
                for (s = socklist; s; s = s->next)
                        s->flags &= ~CONSISTENT;

        rewind(sockfile);
        while (fread((char *) &sock, SOCKINFO_LEN, 1, sockfile)) {

                /*
                 * Test addresses and change 127.0.0.1 and 0.0.0.0
                 * to local address.
                 */
//                if (!sock.saddr.s_addr || sock.saddr.s_addr == 0x0100007f)
//                        sock.saddr = addrlist[0];

//                if (!sock.daddr.s_addr || sock.daddr.s_addr == 0x0100007f)
//                        sock.daddr = addrlist[0];
                
                hs = ht_find(&sock);

                if (!(*hs)) {  /* New socket */
                        if (!(s = register_socket(&sock, hs)))
                                continue;

                        if (!full_msg)
                                put_new(s);
                } else {
                        /*
                         * Socket was found. Log the changes (if any).
                         */
                        s = *hs;

                        if (memcmp(s, &sock, SOCKINFO_LEN)) {
                                if (!full_msg)
                                        put_changes(s, &sock);
                                else
                                        memcpy(s, &sock, SOCKINFO_LEN);
                        }
                }
                s->flags |= CONSISTENT;
                if (full_msg)
                        put(s, SOCKINFO_LEN);
        }
}
        
void clear_closed(void)
{
        struct cis_sockinfo *s, *clsock, **hs;

        s = socklist;

        while(s) {
                if (!s->flags&CONSISTENT) {
                        clsock = s;
                        s = s->next;

                        hs = ht_find(clsock);
                        if (!full_msg)
                                put_closed(clsock);

                        ht_remove(hs);

                        list_remove(socklist, clsock);
                        free(clsock);
                        nsock--;

                } else
                        s = s->next;
        }
}

void allarm_handler(int sig)
{
        struct itimerval it;
        int to_send_init = FALSE;

        while (recv(socketfd, &reply, sizeof(reply), MSG_NOSIGNAL) == sizeof(reply)) {

                if (reply.code == -EAGAIN)
                        to_send_init = TRUE;

                full_msg = TRUE;

                sectotval(it.it_value,    reply.interval);
                sectotval(it.it_interval, reply.interval);
                setitimer(ITIMER_REAL, &it, NULL);
        }
        if (to_send_init)
                send_init();


        buffer_init(full_msg ? SOCKMON_FULL : SOCKMON_CHANGES);
        sock_read();
        
        if (!(mode&FULL_MODE))
                clear_closed();
        buffer_flush();

        if (!(mode&FULL_MODE))
                full_msg = FALSE;

}

void cleanup(int sig)
{
        buffer_init(SOCKMON_EXIT);
        buffer_flush();
//        syslog(LOG_INFO, "terminated");
#ifndef DEBUG
        closelog();
#endif
        exit(0);
}

int main(int argc,char *argv[])
{
        struct sigaction allarm_action;
        struct timeval t = {0, 50000};
        struct itimerval it;
        int i;
        struct hostent *hent;
        struct cis_sockinfo sock, *msg = &sock, *s, **hs;

        struct in_addr addr;
        struct servent *ent;
        struct sockaddr_in inaddr = {AF_INET, 0, {htonl (INADDR_ANY)}};
        unsigned short srv_port, mon_port = 0;
        int err;

        sigset_t sigblockset;
        struct sockaddr_nl nladdr={AF_NETLINK,0,0,0xffffffff};
        int nl_fd;

        setlocale (LC_ALL,"");
        setlocale (LC_NUMERIC, "C");

#ifndef DEBUG
        if (getuid()) {
                fprintf(stderr, "You don't have permission to run sockmon.\n");
                exit (1);
        }
#endif

        openlog("sockmon", LOG_CONS, LOG_DAEMON);

        /*
         * Initialization of signal action structures.
         */
        allarm_action.sa_handler = allarm_handler;
        sigemptyset(&allarm_action.sa_mask);
        allarm_action.sa_flags   = 0;
        
        mypid  = getpid();
        myuid  = getuid();

        if (gethostname(my_hostname, MAXHOSTNAMELEN) == -1) {
                syslog(LOG_ERR, "cannot get my hostname");
                exit(1);
        }

        if (!(hent = gethostbyname (my_hostname))) {
                syslog(LOG_ERR, "cannot get my address");
                exit(1);
        }

        for (i = 0; hent->h_addr_list[i]; i++)
                memcpy(&addrlist[i], hent->h_addr_list[i], hent->h_length);
        
        ct_per_sec = sysconf(_SC_CLK_TCK);

        /*
         * Parse the command line for parameters.
         */
        parse_args(argc, argv);
        if (!srv_name)
                srv_name = my_hostname;
        if (!gethostbyname(srv_name)) {
                syslog(LOG_ERR, "Host %s was not found", srv_name);
                exit(1);
        }
        addr.s_addr = *(long *) hent->h_addr;

        /*
         * Allarm signal for periodical checking the socket list.
         */
        sigaction(SIGALRM, &allarm_action, NULL);
        
        /*
         * We need to block allarm, when dispatching an asynchronous
         * notification from netlink device.
         */
        if (sigemptyset(&sigblockset) == -1) {
                syslog(LOG_ERR, "cannot clear set of blocked signals");
                exit(1);
        }
        if (sigaddset(&sigblockset, SIGALRM) == -1) {
                syslog(LOG_ERR, "cannot initialize set of blocked signals");
                exit(1);
        }
        /*
         * We need to unregister rpc service before terminating ...
         */
        signal(SIGINT, cleanup);
        signal(SIGTERM, cleanup);
        
        if (isroot())
                setpriority(PRIO_PROCESS, mypid, MY_PRIORITY);
        
        /*
         * Alloc memory for the hash table.
         */
        if (!(mode&FULL_MODE)) {
                hash_table = malloc(ht_length * sizeof (struct cis_sockinfo *));
                if (hash_table == NULL) {
                        syslog(LOG_ERR, "cannot allocate memory for the hash table.");
                        exit(1);
                }
        
                memset(hash_table, 0, ht_length * sizeof (struct cis_sockinfo *));
        }

        socklist = NULL; nsock = 0;
        
        /*
         * Initialize input.
         */
        nl_fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_SOCKMON);
        if (nl_fd < 0) {
                syslog(LOG_ERR, "cannot create netlink socket (kernel not patched)");
                exit(1);
        }

        nladdr.nl_pid = mypid;
        if (bind (nl_fd, (struct sockaddr *) &nladdr, sizeof (nladdr)) < 0) {
                syslog(LOG_ERR, "cannot bind netlink socket");
                exit(1);
        }

        sockfile = fopen("/proc/socklist","r");
        if (!sockfile) {
                syslog(LOG_ERR, "cannot open /proc/socklist (kernel not patched)");
                exit(1);
        }

        /*
         * Contact the CIS server.
         */
        if (!(ent = getservbyname("cis", "udp"))) {
                syslog(LOG_ERR, "cannot get CIS server port");
                exit(1);
        }
        srv_port = ent->s_port;

#ifndef DEBUG
        if (!(ent = getservbyname("sockmon", "udp"))) {
                syslog(LOG_ERR, "cannot get monitor port");
                exit(1);
        }
        mon_port = ent->s_port;
#endif

        socketfd = socket(PF_INET, SOCK_DGRAM, 0);
        if (socketfd == -1) {
                syslog(LOG_ERR, "cannot create UDP client socket");
                exit(1);
        }

        inaddr.sin_port = mon_port;
        if (bind (socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
                syslog(LOG_ERR, "cannot bind UDP client socket");
                exit(1);
        }

        inaddr.sin_port = srv_port;
        inaddr.sin_addr = addr;
        if (connect(socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
                syslog(LOG_ERR, "cannot connect to CIS server");
                exit(1);
        }

        /*
         * If we got here, everything seems to be OK.
         * It's time to do the magic to become a daemon }8-{Q.
         */

#ifndef DEBUG
        daemonize();
#endif

        /*
         * Leave the note in the log.
         */
        syslog(LOG_INFO, "started");

        send_init();

        setfdnoblock(socketfd);
        select(0, NULL, NULL, NULL, &t);

        err = recv(socketfd, &reply, sizeof(reply), 0);

        if (err != sizeof(reply) || reply.code != 1)
                reply.interval = DEF_INTERVAL;

        full_msg = TRUE;

        /*
         *  Set the timer for the interval.
         */
        sectotval(it.it_value,    reply.interval);
        sectotval(it.it_interval, reply.interval);
        
        allarm_handler(0);
        setitimer(ITIMER_REAL, &it, NULL);

        for (;;) {
                err = read(nl_fd, msg, SOCKINFO_LEN+1);
                if (err != SOCKINFO_LEN+1)
                        continue;

                /*
                 * Disable allarm signal while preparing the message.
                 */
                if (sigprocmask(SIG_BLOCK, &sigblockset, NULL) == -1) {
                        syslog(LOG_WARNING, "problems with signal blocking");
                        continue;
                }
                
                hs = ht_find(msg);
		if (!(*hs)) {
			if (!(s = register_socket(msg, hs)))
				goto out;
			hs = ht_find(msg);

			buffer_init(SOCKMON_NEW);
                        put(s, SOCKINFO_LEN);
			buffer_flush();
		}
		s = *hs;

		if (msg->flags == SM_CONNECT)
			goto out;

		buffer_init(SOCKMON_CHANGES);
		put_closed(s);
		buffer_flush();

		ht_remove(hs);
		list_remove(socklist, s);
		free(s);
		nsock--;

            out:
                if (sigprocmask(SIG_UNBLOCK, &sigblockset, NULL) == -1)
                        syslog(LOG_WARNING, "problems with signal unblocking");
            
        }
}