/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Process monitor. Provides CIS server with information about processes.
 */

#include <sys/resource.h>
#include <sys/sysinfo.h>
#include <sys/socket.h>
#include <sys/times.h>
#include <unistd.h>
#include <signal.h>
#include <locale.h>
#include <syslog.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <linux/types.h>
#include <linux/netlink.h>

#include <dirent.h>
#include <sys/stat.h>

#define DEF_PRIORITY  (20*ct_per_sec/100)

#include "procmon.h"

#define FULL_MODE 1
#define PROC_MODE 2

#include <linux/tasks.h>  /* for NR_TASKS */

#define HT_LENGTH  (NR_TASKS * 3 / 2)
#define hash(a)    (((a * 3) / 2) % HT_LENGTH)
#define isroot()   (!myuid)

#ifndef MY_PRIORITY
#define MY_PRIORITY             -15
#endif

#define DEF_PROCTAB_LEN         32

#define DEF_INTERVAL  1.

#ifdef DEBUG
#define syslog(tag, fmt, args...)  printf(fmt"\n" , ## args)

struct timeval _start_time, _end_time;

#define reset_time()   gettimeofday(&_start_time, 0)
#define show_time(fmt)    { \
	gettimeofday(&_end_time, 0); \
	printf(fmt" %ld\n", (_end_time.tv_sec - _start_time.tv_sec) * 1000000L + _end_time.tv_usec - _start_time.tv_usec); \
	gettimeofday(&_start_time, 0); \
	}

#else

#define reset_time()  {}
#define show_time(fmt)  {}

#endif

extern void daemonize (void);
extern int setfdnoblock (int fd);

/* Internal procmon information */
int mypid;
int myuid;

clock_t        start_ct;

/* Parameters of connection with cis daemon */
char           *srv_name;

/* Connection variables */
int            mode = 0;
int            socketfd;
struct monitor_msg buff = {0, 0, 1};
char           *buffptr       = buff.data;
int            full_msg;
#define bufflen (buffptr - (char *) &buff)

struct monitor_reply reply;

/* Host information */
char                my_hostname[MAXHOSTNAMELEN];
struct in_addr      addrlist[MAX_ADDR_NUM];
struct utsname      my_uname;
unsigned long       idle_time;
int                 ct_per_sec;
int                 defprio;
int                 page_size;
clock_t             time_ct;
struct sysinfo      sinfo;

/* Process information */
struct cis_procinfo *proctab;
int proctab_len = DEF_PROCTAB_LEN;
struct cis_procinfo *proclist;
struct cis_procinfo **hash_table;

FILE *sysinfofile;
FILE *procfile;
FILE *loadavg;
FILE *meminfo;
FILE *uptime;
DIR *procdir;

char *line;
int linelen;

void parse_args(int argc, char **argv)
{
	int c;
	
	while ((c = getopt(argc, argv, "fb")) != EOF) {
		
		switch (c) {
		case 'f':
			mode |= FULL_MODE;
			break;
		case 'b':
			proctab_len = atoi(optarg);
			if (proctab_len > NR_TASKS)
				proctab_len = NR_TASKS;
			if (proctab_len < 0)
				proctab_len = 1;
			break;
		}
	}
	if (argv[optind])
		srv_name = argv[optind];
}

FILE *proc_open(char *name)
{
	char buff[128];
	FILE *f;
	
	sprintf(buff, "/proc/%s", name);
	f = fopen(buff,"r");
	if (!f) {
		syslog(LOG_ERR, "cannot open /proc/%s", name);
		exit(1);
	}
	return f;
}

/*
 * Searching in the hash table. If the requested process was not found, a
 * pointer to the next free cell is returned.
 */
static inline struct cis_procinfo **ht_find(pid_t pid)
{
	static struct cis_procinfo **hp;
	
	hp = &hash_table[hash(pid)];
	
	while (*hp && (*hp)->pid != pid)
		
		if (++hp == hash_table + HT_LENGTH)
			hp = hash_table;
	
	return (hp);
}

static inline void ht_remove(struct cis_procinfo **hole)
{
	struct cis_procinfo **hp = hole;
	
	*(hole) = NULL;
	
	for(;;) {
		if (++hp == hash_table + HT_LENGTH)
			hp = hash_table;
		
		if (!(*hp)) break;
		
		if (!(*ht_find((*hp)->pid)))
			*hole = *hp, *hp = NULL, hole = hp;
	}
}

struct cis_procinfo *register_process(struct cis_procinfo *process, struct cis_procinfo **hp)
{
	struct cis_procinfo *p;
	
	p = calloc(1, sizeof (struct cis_procinfo));
	if (!p)
		return (NULL);
	
	memcpy(p, process, PROCINFO_LEN);
	
	*hp = p;
	
	list_insert(proclist, p);
	
	p->flags |= CONSISTENT;
	
	return (p);
}

#define SOCKFLAGS     (MSG_DONTWAIT | MSG_NOSIGNAL)

void buffer_send(void)
{
	buff.time = times(NULL);
	buff.seqnum++;
	
	send(socketfd, &buff, bufflen, SOCKFLAGS);
	buffptr = buff.data;
	if (buff.flags & MSG_FIRST)
		buff.flags &= ~MSG_FIRST;
}

void buffer_flush(void)
{
	buff.flags |= MSG_LAST;
	buffer_send();
}

void buffer_init(char tag)
{
	if (buffptr > buff.data)
		buffer_flush();
	
	buff.flags = MSG_FIRST;
	buff.tag = tag;
	buffptr  = buff.data;
}

#define put(ptr, len) \
	{                                           \
	if ((bufflen + len) > MON_BUFFLEN)   \
	buffer_send();                \
	memcpy(buffptr, ptr, len);           \
	buffptr += len;               \
	}

#define put_object(obj) put(&obj, sizeof(obj))

void pack_init()
{
	int i, n;
	
	put_object(start_ct);
	put_object(ct_per_sec);
	for (i = 0; addrlist[i].s_addr; i++)
		;
	n = i;
	put_object(n);
	for (i = 0; addrlist[i].s_addr; i++)
		put_object(addrlist[i]);
	
	put_object(my_uname);
	put_object(defprio);
	put_object(sinfo.totalram);
	put_object(sinfo.totalswap);
}

void send_init(void)
{
	sysinfo(&sinfo);
	sinfo.totalram  >>= 10;
	sinfo.totalswap >>= 10;
	start_ct = times(NULL);
	buffer_init(PROCMON_INIT);
	pack_init();
	buffer_flush();
}

#define add_change(el) \
	if (p->el != msg->el) {                           \
	*(chptr++) = OFFSET(cis_procinfo, el);     \
	*(int *) chptr = msg->el - p->el;          \
	chptr += sizeof(int);                      \
	p->el = msg->el;                           \
	}

#define add_element(el) \
	{                                                 \
	*(chptr++) = OFFSET(cis_procinfo, el);     \
	memcpy(chptr,  &msg->el, sizeof(msg->el)); \
	memcpy(&p->el, &msg->el, sizeof(msg->el)); \
	chptr += sizeof(msg->el);                  \
	}

inline void put_changes(struct cis_procinfo *p, struct cis_procinfo *msg)
{
	char chbuff[sizeof(struct cis_procinfo)];
	char *chptr = chbuff;
	
	*(typeof(p->pid) *) chptr = p->pid;
	chptr += sizeof(p->pid);
	
	if (p->utime    != msg->utime)     add_change(utime);
	if (p->stime    != msg->stime)     add_change(stime);
	if (p->priority != msg->priority)  add_change(priority);
	if (p->rss      != msg->rss)       add_change(rss);
	if (p->vm       != msg->vm)        add_change(vm);
	if (p->ppid     != msg->ppid)      add_element(ppid);
	if (p->uid      != msg->uid)       add_element(uid);
	
	if (strcmp(p->cmd, msg->cmd))      add_element(cmd);
	
	if (chptr == chbuff+sizeof(p->pid))
		return;
	
	*(chptr++) = 0;
	
	if ((bufflen + (chptr - chbuff)) > MON_BUFFLEN)
		buffer_send();
	put(chbuff, chptr - chbuff);
}

inline void put_process(char tag, struct cis_procinfo *p)
{
	if ((bufflen + sizeof(p->pid) + 1 + PROCINFO_LEN) > MON_BUFFLEN)
		buffer_send();
	
	*(typeof(p->pid) *) buffptr = p->pid;
	buffptr += sizeof(p->pid);
	*(buffptr++) = tag;
	
	put(p, PROCINFO_LEN);
}

#define put_new(p)        put_process(1, p)
#define put_terminated(p) put_process(2, p)

/*
 * Get processes and prepare message to server.
 */
void proc_read_kernel(void)
{
	struct cis_procinfo *proc, *p, **hp;
	char *ptr;
	int nproc;
	
	if (!(mode&FULL_MODE))
		for (p = proclist; p; p = p->next)
			p->flags &= ~CONSISTENT;
	
	procfile = freopen("/proc/proclist","r", procfile);
	if (!procfile)
		return;
	
	rewind(procfile);
	
again:
	nproc = fread(proctab, PROCINFO_LEN, proctab_len, procfile);
	if (!nproc)
		return;
	
	for (ptr = (char *) proctab; ptr < ((char *) proctab)+(nproc*PROCINFO_LEN); ptr += PROCINFO_LEN) {
		proc = (struct cis_procinfo *) ptr;
		
		if (mode&FULL_MODE) {
			put(proc, PROCINFO_LEN);
			continue;
		}
		
		hp = ht_find(proc->pid);
		
		if (!(*hp)) {  /* New process */
			
			p = register_process(proc, hp);
			if (!p)
				continue;
			
			if (!full_msg)
				put_new(p);
			
		} else {
			/*
			 * Process was found. Log the changes (if any).
			 */
			p = *hp;
			if (memcmp(p, proc, PROCINFO_LEN)) {
				if (!full_msg)
					put_changes(p, proc);
				else
					memcpy(p, proc, PROCINFO_LEN);
			}
		}
		p->flags |= CONSISTENT;
		if (full_msg)
			put(p, PROCINFO_LEN);
	}
	if (nproc == proctab_len)
		goto again;
}

#define skip_one() {for (; *c != ' '; c++); c++;}

inline int get_next (struct cis_procinfo *proc)
{
	static char name[128];
	struct dirent *ent;
	struct stat fst;
	char *cmd_start;
	register char *c;
	register int i;
	int cmd_len;
	//        int err;
	FILE *f;
	
again:
	while ((ent = readdir(procdir)) && !(proc->pid = atoi(ent->d_name)))
		;
	
	if (!ent)
		return 0;
	
	sprintf(name, "/proc/%d/stat", proc->pid);
	
	f = fopen(name, "r");
	if (!f)
		goto again;
	
	getline(&line, &linelen, f);
	
	cmd_start = index(line, '(')+1;
	for (c = cmd_start+CMD_LENGTH+1; *c != ')'; c--)
		;
	cmd_len = c - cmd_start;
	
	strncpy(proc->cmd, cmd_start, cmd_len);
	proc->cmd[cmd_len] = 0;
	
	c += 2;
	skip_one();
	proc->ppid = atoi(c); skip_one();
	for (i = 0; i < 9; i++) skip_one();
	proc->utime = atol(c); skip_one();
	proc->stime = atol(c); skip_one();
	skip_one(); skip_one(); skip_one();
	proc->priority = atol(c); skip_one();
	skip_one(); skip_one();
	proc->start_time = atol(c); skip_one();
	proc->vm = atol(c); skip_one();
	proc->rss = atol(c);
	
	
	//        err = sscanf(c+2,
	//                     "%*c "
	//                     "%d "           /* ppid */
	//                     "%*d %*d %*d %*d %*u %*u %*u %*u %*u "
	//                     "%ld %ld "      /* utime, stime */
	//                     "%*d %*d "
	//                     "%*d "          /* nice */
	//                     "%ld %*u %*u "
	//                     "%ld %lu %lu ", /* start_time, vmsize, rss */
	//                     &proc->ppid, &proc->utime, &proc->stime, &proc->priority,
	//                     &proc->start_time, &proc->vm, &proc->rss);
	
	proc->priority = ((20 - proc->priority) * DEF_PRIORITY - DEF_PRIORITY / 2) / 20 + 1;
	proc->rss *= page_size;
	
	fstat (fileno(f), &fst);
	proc->uid = fst.st_uid;
	
	fclose(f);
	
	return 1;
}

/*
 * Get processes and prepare message to server.
 */
void proc_read_proc(void)
{
	struct cis_procinfo proc, *p, **hp;
	
	if (!(mode&FULL_MODE))
		for (p = proclist; p; p = p->next)
			p->flags &= ~CONSISTENT;
	
	rewinddir(procdir);
	
	while (get_next(&proc)) {
		
		if (mode&FULL_MODE) {
			put(&proc, PROCINFO_LEN);
			continue;
		}
		
		hp = ht_find(proc.pid);
		
		if (!(*hp)) {  /* New process */
			
			p = register_process(&proc, hp);
			if (!p)
				continue;
			
			if (!full_msg)
				put_new(p);
			
		} else {
			/*
			 * Process was found. Log the changes (if any).
			 */
			p = *hp;
			if (memcmp(p, &proc, PROCINFO_LEN)) {
				if (!full_msg)
					put_changes(p, &proc);
				else
					memcpy(p, &proc, PROCINFO_LEN);
			}
		}
		p->flags |= CONSISTENT;
		if (full_msg)
			put(p, PROCINFO_LEN);
	}
}

void clear_terminated(void)
{
	struct cis_procinfo *p, *termproc, **hp;
	
	p = proclist;
	
	while (p) {
		if (!p->flags&CONSISTENT) {
			termproc = p;
			p = p->next;
			
			hp = ht_find(termproc->pid);
			if (!full_msg)
				put_terminated(termproc);
			
			ht_remove(hp);
			
			list_remove(proclist, termproc);
			free(termproc);
			
		} else
			p = p->next;
	}
}

void allarm_handler(int sig)
{
	struct cis_hostinfo di;
	struct itimerval it;
	int to_send_init = FALSE;

	while (recv(socketfd, &reply, sizeof(reply), MSG_NOSIGNAL) == sizeof(reply)) {
		
		if (reply.code == -EAGAIN)
			to_send_init = TRUE;
		
		full_msg = TRUE;
		
		sectotval(it.it_value,    reply.interval);
		sectotval(it.it_interval, reply.interval);
		setitimer(ITIMER_REAL, &it, NULL);
	}
	if (to_send_init)
		send_init();
	/*
	 * Get the host system information.
	 */
	if (!(mode&PROC_MODE)) {
		sysinfofile = freopen("/proc/sysinfo","r", sysinfofile);
		reset_time();
		rewind(sysinfofile);
		fread(&di, HOSTINFO_LEN, 1, sysinfofile);
	} else {
		float idle;
		
		loadavg = freopen("/proc/loadavg", "r", loadavg);
		fscanf(loadavg, "%f %f %f %*d/%hd",
		       di.loads, di.loads+1, di.loads+2, &di.procnum);
		
		uptime = freopen("/proc/uptime", "r", uptime);
		fscanf(uptime, "%*f %f", &idle);
		di.idle = idle * ct_per_sec;
		
		meminfo = freopen("/proc/meminfo", "r", meminfo);
		getline(&line, &linelen, meminfo); /* Field names */
		getline(&line, &linelen, meminfo); /* Memory */
		sscanf(line, "%*s %*d %*d %ld %ld %ld %ld",
		       &di.freeram, &di.sharedram, &di.bufferram, &di.cachedram);
		
		getline(&line, &linelen, meminfo); /* Swap */
		sscanf(line, "%*s %ld %*d %ld",
		       &di.totalswap, &di.freeswap);
		
		di.freeram   >>= 10;
		di.sharedram >>= 10;
		di.bufferram >>= 10;
		di.cachedram >>= 10;
		di.totalswap >>= 10;
		di.freeswap  >>= 10;
	}
	
	buffer_init(PROCMON_SYSINFO);
	put(&di, HOSTINFO_LEN);
	buffer_flush();
	
	/*
	 * Prepare the information about processes.
	 */
	buffer_init(full_msg ? PROCMON_FULL : PROCMON_CHANGES);
	if (mode&PROC_MODE)
		proc_read_proc();
	else
		proc_read_kernel();
	
	if (!(mode&FULL_MODE))
		clear_terminated();
	buffer_flush();
	
	if (!(mode&FULL_MODE))
		full_msg = FALSE;
}

void cleanup(int sig)
{
	buffer_init(PROCMON_EXIT);
	buffer_flush();
	//        syslog(LOG_INFO, "terminated");
#ifndef DEBUG
	closelog();
#endif
	exit(0);
}

int main(int argc, char *argv[])
{
	struct sigaction allarm_action;
	struct timeval t = {0, 50000};
	struct itimerval it;
	struct hostent *hent;
	sigset_t sigblockset;
	int i;
	struct in_addr addr;
	struct servent *ent;
	struct sockaddr_in inaddr = {AF_INET, 0, {htonl (INADDR_ANY)}};
	unsigned short srv_port, mon_port = 0;
	int err;
	
	struct cis_procinfo proc, *msg = &proc, *p, **hp = NULL;
	struct sockaddr_nl nladdr={AF_NETLINK,0,0,0xffffffff};
	int nl_fd = 0;
	struct stat st;
	
	setlocale(LC_ALL,"");
	setlocale(LC_NUMERIC, "C");
	
#ifndef DEBUG
	if (getuid()) {
		fprintf(stderr, "You don't have permission to run procmon.\n");
		exit (1);
	}
	
	openlog("procmon", LOG_CONS, LOG_DAEMON);
#endif
	
	/*
	 * Initialization of signal action structures.
	 */
	allarm_action.sa_handler = allarm_handler;
	sigemptyset(&allarm_action.sa_mask);
	allarm_action.sa_flags   = 0;
	
	mypid = getpid();
	myuid = getuid();
	
	if (gethostname(my_hostname, MAXHOSTNAMELEN) == -1) {
		syslog(LOG_ERR, "cannot get my hostname");
		exit(1);
	}
	if (!(hent = gethostbyname(my_hostname))) {
		syslog(LOG_ERR, "cannot get my address");
		exit(1);
	}
	
	for (i = 0; hent->h_addr_list[i]; i++)
		memcpy(&addrlist[i], hent->h_addr_list[i], hent->h_length);
	
	if (uname(&my_uname) == -1) {
		syslog(LOG_ERR, "cannot get system information");
		exit(1);
	}
	
	ct_per_sec     = sysconf(_SC_CLK_TCK);
	defprio        = (20*ct_per_sec/100);
	page_size      = sysconf(_SC_PAGE_SIZE);
	
	/*
	 * Parse the command line for parameters.
	 */
	parse_args(argc, argv);
	
	if (!srv_name)
		srv_name = my_uname.nodename;
	if (!(hent = gethostbyname(srv_name))) {
		syslog(LOG_ERR, "Host %s was not found", srv_name);
		exit(1);
	}
	addr.s_addr = *(long *) hent->h_addr;
	
	/*
	 * Allarm signal for periodical checking the process list.
	 */
	sigaction(SIGALRM, &allarm_action, NULL);
	
	/*
	 * We need to block allarm, when dispatching an asynchronous
	 * notification from netlink device.
	 */
	if (sigemptyset(&sigblockset) == -1) {
		syslog(LOG_ERR, "cannot clear set of blocked signals");
		exit(1);
	}
	
	if (sigaddset(&sigblockset, SIGALRM) == -1) {
		syslog(LOG_ERR, "cannot initialize set of blocked signals");
		exit(1);
	}
	
	/*
	 * We need to unregister rpc service before terminating ...
	 */
	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);
	
	if (isroot())
		setpriority(PRIO_PROCESS, mypid, MY_PRIORITY);
	
	/*
	 * Alloc memory for the hash table, the table of new processes and the
	 * table of changes.
	 */
	if (!(mode&FULL_MODE)) {
		hash_table = malloc(HT_LENGTH * sizeof (struct cis_procinfo *));
		if (hash_table == NULL) {
			syslog(LOG_ERR, "cannot allocate memory for the hash table.");
			exit(1);
		}
		
		memset(hash_table, 0, HT_LENGTH * sizeof (struct cis_procinfo *));
		proclist = NULL;
	}
	
	/*
	 * Initialize input.
	 */
	if (stat("/proc/proclist", &st) == -1)
		mode |= PROC_MODE;
	
	if (!(mode&PROC_MODE)) {
		proctab = malloc(proctab_len * sizeof (struct cis_procinfo));
		if (!proctab) {
			syslog(LOG_ERR, "not enough memory");
			exit(1);
		}
		
		nl_fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_PROCMON);
		if (nl_fd < 0) {
			syslog(LOG_ERR, "cannot create netlink socket");
			mode |= PROC_MODE;
		}
		
		nladdr.nl_pid = mypid;
		if (bind(nl_fd, (struct sockaddr *) &nladdr, sizeof (nladdr)) < 0) {
			syslog(LOG_ERR, "cannot bind netlink socket");
			mode |= PROC_MODE;
		}
		
		procfile    = proc_open("proclist");
		sysinfofile = proc_open("sysinfo");
	}
	
	if (mode&PROC_MODE) {
		loadavg     = proc_open("loadavg");
		uptime      = proc_open("uptime");
		meminfo     = proc_open("meminfo");
		
		procdir     = opendir("/proc");
	}
	
	/*
	 * Connect to the CIS server.
	 */
	
	if (!(ent = getservbyname("cis", "udp"))) {
		syslog(LOG_ERR, "cannot get CIS server port");
		exit(1);
	}
	srv_port = ent->s_port;
	
#ifndef DEBUG
	if (!(ent = getservbyname("procmon", "udp"))) {
		syslog(LOG_ERR, "cannot get monitor port");
		exit(1);
	}
	mon_port = ent->s_port;
#endif
	
	socketfd = socket(PF_INET, SOCK_DGRAM, 0);
	if (socketfd == -1) {
		syslog(LOG_ERR, "cannot create UDP client socket");
		exit(1);
	}
	
	inaddr.sin_port = mon_port;
	if (bind (socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
		syslog(LOG_ERR, "cannot bind UDP client socket");
		exit(1);
	}
	
	inaddr.sin_port = srv_port;
	inaddr.sin_addr = addr;
	if (connect(socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
		syslog(LOG_ERR, "cannot connect to CIS server");
		exit(1);
	}
	
	/*
	 * If we got here, everything seems to be OK.
	 * It's time to do the abraca-dabra and become the daemon }8-{Q.
	 */
	
#ifndef DEBUG
	daemonize();
#endif
	
	/*
	 * Leave a note in the log.
	 */
	syslog(LOG_INFO, "started");
	
	send_init();
	
	setfdnoblock(socketfd);
	select(0, NULL, NULL, NULL, &t);
	
	err = recv(socketfd, &reply, sizeof(reply), 0);
	
	if (err != sizeof(reply) || reply.code != 1)
		reply.interval = DEF_INTERVAL;
	
	full_msg = TRUE;
	/*
	 *  Set the timer for the interval.
	 */
	sectotval(it.it_value,    reply.interval);
	sectotval(it.it_interval, reply.interval);
	
	allarm_handler(0);
	setitimer(ITIMER_REAL, &it, NULL);
	
	if (mode&PROC_MODE)
		for (;;)
			select(0, NULL, NULL, NULL, NULL);
	
	for (;;) {
		int err = read(nl_fd, msg, PROCINFO_LEN+1);
		if (err != PROCINFO_LEN+1)
			continue;
		
		/*
		 * Disable allarm signal while parsing the message.
		 */
		if (sigprocmask(SIG_BLOCK, &sigblockset, NULL) == -1) {
			syslog(LOG_WARNING, "problems with signal blocking");
			continue;
		}
		
		p = NULL;
		if (!(mode&FULL_MODE)) {
			hp = ht_find(msg->pid);
			p = *hp;
		}
		switch (msg->flags) {
			
		case PM_FORK:  /* Started process */
			
			if (!(mode&FULL_MODE)) {
				if (*hp) {
					/*
					 * Just for the case ...
					 */
#ifdef DEBUG
					syslog(LOG_WARNING, "started process is already in hash table ");
#endif
					
				} else if (!(p = register_process(msg, hp)))
					goto out;
			}
			
			buffer_init(PROCMON_NEW);
			put(msg, PROCINFO_LEN);
			break;
			
		case PM_EXEC: /* New command executed - identity change */
			
			if (mode&FULL_MODE || !p)
				goto out;
			
			buffer_init(PROCMON_CHANGES_ASYNC);
			put_changes(p, msg);
			break;
			
		case PM_EXIT: /* Terminated process */
			
			if (!(mode&FULL_MODE) && !p)
				goto out;
			
			buffer_init(PROCMON_CHANGES_ASYNC);
			put_terminated(msg);
			
			if (!(mode&FULL_MODE)) {
				ht_remove(hp);
				list_remove(proclist, p);
				free(p);
			}
			break;
		}
		buffer_flush();
	out:
		if (sigprocmask(SIG_UNBLOCK, &sigblockset, NULL) == -1)
			syslog(LOG_WARNING, "problems with signal unblocking");
	}
}

