/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Network interface monitor. Provides CIS server with the information about
 * status and activity of network interfaces.
 */

#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/times.h>
#include <unistd.h>
#include <signal.h>
#include <locale.h>
#include <syslog.h>
#include <errno.h>

#include <linux/types.h>
#include <linux/netlink.h>

#include <dirent.h>
#include <sys/stat.h>

#include "netmon.h"

#define FULL_MODE 1
#define PROC_MODE 2

#define isroot()   (!myuid)

#ifndef MY_PRIORITY
#define MY_PRIORITY             -15
#endif

#ifndef DEF_INTERVAL
#define DEF_INTERVAL              1.
#endif

#ifndef DEF_ND_LENGTH
#define DEF_ND_LENGTH             16
#endif

#ifdef DEBUG
#define syslog(tag, fmt, args...)  printf(fmt"\n" , ## args)
#endif

extern void daemonize (void);
extern int setfdnoblock (int fd);

/* Internal netmon information */
int mypid;
int myuid;

clock_t        start_ct;

/* Parameters of connection with cis daemon */
char           *srv_name;

/* Connection variables */
int            mode = 0;
int            socketfd;
struct monitor_msg buff = {0, 0, 1};
char           *buffptr       = buff.data;
int            full_msg;
#define bufflen (buffptr - (char *) &buff)

struct monitor_reply reply;

/* Host information */
char           my_hostname[MAXHOSTNAMELEN];
struct in_addr addrlist[MAX_ADDR_NUM];
clock_t        time_ct;
int            ct_per_sec;

/* Network device information */
FILE *devfile;
struct cis_netdevinfo *devlist;
int ndev;
int nd_length  = DEF_ND_LENGTH;
struct cis_netdevinfo **new_devices;
struct change_info *change_table;

char *line = NULL;
int  linelen = 0;

struct dev_type {
        short type;
        char name[20];
} dev_types[] = {
        {1,   "eth"},
        {-1},
};

int dev_type(char *name)
{
        struct dev_type *t;

        for (t = dev_types; t->type != -1; t++)
                if (!strncpy(t->name, name, strlen(name)))
                        return t->type;
        return -1;
}

void parse_args(int argc, char **argv)
{
        int c;

        while ((c = getopt(argc, argv, "f")) != EOF) {

                switch (c) {
                case 'f':
                        mode |= FULL_MODE;
                        break;
                }
        }
        if (argv[optind])
                srv_name = argv[optind];
}

struct cis_netdevinfo *register_device(struct cis_netdevinfo *msg)
{
        static char id = 1;
        struct cis_netdevinfo *dev;
        
        dev = calloc(1, sizeof (struct cis_netdevinfo));
        if (!dev)
                return (NULL);

        memcpy(dev, msg, NETDEVINFO_LEN);

        list_insert(devlist, dev);

        dev->flags |= CONSISTENT;
        dev->id    |= id++;
        return (dev);
}

#define SOCKFLAGS     (MSG_DONTWAIT | MSG_NOSIGNAL)

void buffer_send(void)
{
        buff.time = times(NULL);
        buff.seqnum++;

        send(socketfd, &buff, bufflen, SOCKFLAGS);
        buffptr = buff.data;
        if (buff.flags & MSG_FIRST)
                buff.flags &= ~MSG_FIRST;
}

void buffer_flush(void)
{
        buff.flags |= MSG_LAST;
        buffer_send();
}

void buffer_init(char tag)
{
        if (buffptr > buff.data)
                buffer_flush();

        buff.flags = MSG_FIRST;
        buff.tag = tag;
        buffptr  = buff.data;
}

#define put(ptr, len) \
        {                                           \
               if ((bufflen + len) > MON_BUFFLEN)   \
                      buffer_send();                \
               memcpy(buffptr, ptr, len);           \
                      buffptr += len;               \
        }

#define put_object(obj) put(&obj, sizeof(obj))

void pack_init()
{
        int i, n;

        put_object(start_ct);
        put_object(ct_per_sec);
        for (i = 0; addrlist[i].s_addr; i++)
                ;
        n = i;
        put_object(n);
        for (i = 0; addrlist[i].s_addr; i++)
                put_object(addrlist[i]);
}

void send_init(void)
{
        start_ct = times(NULL);
        buffer_init(NETMON_INIT);
        pack_init();
        buffer_flush();
}

#define add_change(el) \
        if (d->el != msg->el) {                           \
               *(chptr++) = OFFSET(cis_netdevinfo, el);   \
               *(int *) chptr = msg->el - d->el;          \
               chptr += sizeof(int);                      \
               d->el = msg->el;                           \
        }

#define add_element(el) \
        {                                                 \
               *(chptr++) = OFFSET(cis_netdevinfo, el);   \
               memcpy(chptr,  &msg->el, sizeof(msg->el)); \
               memcpy(&d->el, &msg->el, sizeof(msg->el)); \
               chptr += sizeof(msg->el);                  \
        }

void put_device(char tag, struct cis_netdevinfo *d)
{
        if ((bufflen + sizeof(d->id) + 1 + NETDEVINFO_LEN) > MON_BUFFLEN)
                buffer_send();

        *(typeof(d->id) *) buffptr = d->id;
        buffptr += sizeof(d->id);
        *(buffptr++) = tag;

        put(d, NETDEVINFO_LEN);
}

#define put_started(d)    put_device(1, d)
#define put_closed(d)     put_device(2, d)

void put_changes(struct cis_netdevinfo *d, struct cis_netdevinfo *msg)
{
        char chbuff[sizeof(struct cis_netdevinfo)];
        char *chptr = chbuff;

        *(typeof(d->id) *) chptr = d->id;
        chptr += sizeof(d->id);

        if (d->tx_bytes   != msg->tx_bytes)    add_change(tx_bytes);
        if (d->rx_bytes   != msg->rx_bytes)    add_change(rx_bytes);
        if (d->collisions != msg->collisions)  add_change(collisions);
        if (d->status     != msg->status)      add_element(status);

        if (chptr == chbuff+sizeof(d->id))
                return;

        *(chptr++) = 0;
            
        if ((bufflen + (chptr - chbuff)) > MON_BUFFLEN)
                buffer_send();
        put(chbuff, chptr - chbuff);
}

int get_next (struct cis_netdevinfo *dev)
{
        char *c;

        if (!(mode&PROC_MODE))
                return (fread(dev, NETDEVINFO_LEN, 1, devfile));

    again:
        if (getline(&line, &linelen, devfile) == -1)
                return 0;

        if (!strncmp(line, "    lo:", 7))
                goto again;

        c = index (line, ':');
        if (c)
                *c = ' ';

        sscanf(line, "%s "
               "%lu %*u %*u %*u %*u %*u %*u %*u "
               "%lu %*u %*u %*u %*u %lu %*u %*u ",
               dev->name, &dev->rx_bytes, &dev->tx_bytes, &dev->collisions);

	dev->status = 1;
        return 1;
}
/*
 * Read device information and prepare message.
 */
void dev_read(void)
{
        static struct cis_netdevinfo dev, *d;

        if (!(mode&FULL_MODE))
                for (d = devlist; d; d = d->next)
                        d->flags &= ~CONSISTENT;

        rewind(devfile);

        if (mode&PROC_MODE) {         /* Skip header */
                getline(&line, &linelen, devfile);
                getline(&line, &linelen, devfile);
        }
        
        while (get_next(&dev)) {

                for (d = devlist; d; d = d->next)
                        if (!strncmp (d->name, dev.name, IFNAMSIZ))
                                break;

                if (!d) {  /* New device */
                        if (!(d = register_device(&dev)))
                                continue;

                        if (!full_msg)
                                put_started(d);
                } else {
                        /*
                         * Device was found. Log the changes (if any).
                         */
                        dev.id = d->id;
                        if (memcmp(d, &dev, NETDEVINFO_LEN)) {
                                if (!full_msg)
                                        put_changes(d, &dev);
                                else
                                        memcpy(d, &dev, NETDEVINFO_LEN);
                        }
                }
                d->flags |= CONSISTENT;
                if (full_msg)
                        put(d, NETDEVINFO_LEN);
        }
}

void allarm_handler(int sig)
{
        struct cis_netdevinfo *d;
        struct itimerval it;
        int to_send_init = FALSE;

        while (recv(socketfd, &reply, sizeof(reply), MSG_NOSIGNAL) == sizeof(reply)) {

                if (reply.code == -EAGAIN)
                        to_send_init = TRUE;

                full_msg = TRUE;

                sectotval(it.it_value,    reply.interval);
                sectotval(it.it_interval, reply.interval);
                setitimer(ITIMER_REAL, &it, NULL);
        }
        if (to_send_init)
                send_init();

        /*
         * Prepare the information about devices.
         */
        buffer_init(full_msg ? NETMON_FULL : NETMON_CHANGES);
        dev_read();

        if (!(mode&FULL_MODE)) {
                /*
                 * Test for existence of closed devices.
                 */
                for (d = devlist; d; d = d->next)
                        if (!(d->flags&CONSISTENT) && d->status) {
                                d->status = 0;
                                put_closed(d);
                        }
        }
        buffer_flush();

        if (!(mode&FULL_MODE))
                full_msg = FALSE;
}

void cleanup(int sig)
{
        buffer_init(PROCMON_EXIT);
        buffer_flush();

        //        syslog(LOG_INFO, "terminated");
#ifndef DEBUG
        closelog();
#endif
        exit(0);
}

int main(int argc,char *argv[])
{
        struct sigaction allarm_action;
        struct timeval t = {0,50000};
        struct itimerval it;
        struct hostent *hent;
        sigset_t sigblockset;
        int i;
        int err;
        
        struct in_addr addr;
        struct servent *ent;
        struct sockaddr_in inaddr = {AF_INET, 0, {htonl (INADDR_ANY)}};
        unsigned short srv_port, mon_port = 0;

        struct cis_netdevinfo msg, *dev;
        struct sockaddr_nl nladdr={AF_NETLINK,0,0,0xffffffff};
        int nl_fd;
        struct stat st;

        setlocale (LC_ALL,"");
        setlocale (LC_NUMERIC, "C");

#ifndef DEBUG
        if (getuid()) {
                fprintf(stderr, "You don't have permission to run netmon.\n");
                exit (1);
        }

        openlog("netmon", LOG_CONS, LOG_DAEMON);
#endif

        /*
         * Initialization of signal action structures.
         */
        allarm_action.sa_handler = allarm_handler;
        sigemptyset(&allarm_action.sa_mask);
        allarm_action.sa_flags   = 0;
        
        mypid  = getpid();
        myuid  = getuid();

        if (gethostname(my_hostname, MAXHOSTNAMELEN) == -1) {
                syslog(LOG_ERR, "cannot get my hostname");
                exit(1);
        }

        if (!(hent = gethostbyname (my_hostname))) {
                syslog(LOG_ERR, "cannot get my address");
                exit(1);
        }

        for (i = 0; hent->h_addr_list[i]; i++)
                memcpy(&addrlist[i], hent->h_addr_list[i], hent->h_length);
        
        ct_per_sec = sysconf(_SC_CLK_TCK);

        /*
         * Parse the command line for parameters.
         */
        parse_args(argc, argv);
        if (!srv_name)
                srv_name = my_hostname;
        if (!gethostbyname(srv_name)) {
                syslog(LOG_ERR, "Host %s was not found", srv_name);
                exit(1);
        }
        addr.s_addr = *(long *) hent->h_addr;

        /*
         * Allarm signal for periodical checking the device list.
         */
        sigaction(SIGALRM, &allarm_action, NULL);
        
        /*
         * We need to block allarm, when dispatching an asynchronous
         * notification from netlink device.
         */
        if (sigemptyset(&sigblockset) == -1) {
                syslog(LOG_ERR, "cannot clear set of blocked signals");
                exit(1);
        }
        if (sigaddset(&sigblockset, SIGALRM) == -1) {
                syslog(LOG_ERR, "cannot initialize set of blocked signals");
                exit(1);
        }
        /*
         * We need to unregister rpc service before terminating ...
         */
        signal(SIGINT, cleanup);
        signal(SIGTERM, cleanup);
        
        if (isroot())
                setpriority(PRIO_PROCESS, mypid, MY_PRIORITY);
        
        devlist = NULL;
        
        /*
         * Initialize input.
         */
        if (stat("/proc/netdevlist", &st) == -1)
                mode |= PROC_MODE;

        if (!(mode&PROC_MODE)) {
                
                nl_fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_NETMON);
                if (nl_fd < 0) {
                        syslog(LOG_ERR, "cannot create netlink socket (kernel not patched)");
                        exit(1);
                }
                
                nladdr.nl_pid = mypid;
                if (bind (nl_fd, (struct sockaddr *) &nladdr, sizeof (nladdr)) < 0) {
                        syslog(LOG_ERR, "cannot bind netlink socket");
                        exit(1);
                }
                
                devfile = fopen("/proc/netdevlist","r");
                if (!devfile) {
                        syslog(LOG_ERR, "cannot open /proc/netdevlist (kernel not patched)");
                        exit(1);
                }
        } else {
                devfile = fopen("/proc/net/dev","r");
                if (!devfile) {
                        syslog(LOG_ERR, "cannot open /proc/net/dev");
                        exit(1);
                }
        }
        
        /*
         * Contact the CIS server.
         */
        if (!(ent = getservbyname("cis", "udp"))) {
                syslog(LOG_ERR, "cannot get CIS server port");
                exit(1);
        }
        srv_port = ent->s_port;

#ifndef DEBUG
        if (!(ent = getservbyname("netmon", "udp"))) {
                syslog(LOG_ERR, "cannot get monitor port");
                exit(1);
        }
        mon_port = ent->s_port;
#endif

        socketfd = socket(PF_INET, SOCK_DGRAM, 0);
        if (socketfd == -1) {
                syslog(LOG_ERR, "cannot create UDP client socket");
                exit(1);
        }

        inaddr.sin_port = mon_port;
        if (bind (socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
                syslog(LOG_ERR, "cannot bind UDP client socket");
                exit(1);
        }

        inaddr.sin_port = srv_port;
        inaddr.sin_addr = addr;
        if (connect(socketfd, (struct sockaddr *) &inaddr, sizeof (inaddr)) < 0) {
                syslog(LOG_ERR, "cannot connect to CIS server");
                exit(1);
        }

        /*
         * If we got here, everything seems to be OK.
         * It's time to do the magic to become a daemon }8-{Q.
         */

#ifndef DEBUG
        daemonize();
#endif

        /*
         * Leave the note in the log.
         */
        syslog(LOG_INFO, "started");

        send_init();

        setfdnoblock(socketfd);
        select(0, NULL, NULL, NULL, &t);

        err = recv(socketfd, &reply, sizeof(reply), 0);

        if (err != sizeof(reply) || reply.code != 1)
                reply.interval = DEF_INTERVAL;

        full_msg = TRUE;
        /*
         *  Set the timer for the interval.
         */
        sectotval(it.it_value,    reply.interval);
        sectotval(it.it_interval, reply.interval);
        
        allarm_handler(0);
        setitimer(ITIMER_REAL, &it, NULL);

        if (mode&PROC_MODE)
                for (;;)
                        select(0, NULL, NULL, NULL, NULL);

        for (;;) {
                err = read(nl_fd, &msg, NETDEVINFO_LEN);
                if (err != NETDEVINFO_LEN)
                        continue;

                /*
                 * Disable allarm signal while preparing the message.
                 */
                if (sigprocmask(SIG_BLOCK, &sigblockset, NULL) == -1) {
                        syslog(LOG_WARNING, "problems with signal blocking");
                        continue;
                }

                for (dev = devlist; dev; dev = dev->next)
                        if (!strncmp (dev->name, msg.name, IFNAMSIZ))
                                break;

                if (!dev) {
                        if (!(dev = register_device(&msg)))
                                goto out;

                        buffer_init(NETMON_NEW);
                        put(dev, NETDEVINFO_LEN);
                        buffer_flush();

                        if (msg.status)
                                goto out;
                }

                buffer_init(NETMON_CHANGES);
                put_changes(dev,&msg);
                buffer_flush();

            out:
                if (sigprocmask(SIG_UNBLOCK, &sigblockset, NULL) == -1)
                        syslog(LOG_WARNING, "problems with signal unblocking");
            
        }
}