/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * CIS server daemon - core code with interface to monitors.
 */

#include <unistd.h>
#include <signal.h>
#include <locale.h>
#include <syslog.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "cis.h"
#include "cis_xdr.h"
#include "cis_srv.h"

#define DEF_CONFIG                "/etc/cis.conf"

#define DEF_OPEN_MODE             O_CREAT|O_APPEND|O_NOCTTY

#ifndef DEF_INTERVAL
#define DEF_INTERVAL              1.
#endif

#ifndef DEF_MSG_COUNT
#define DEF_MSG_COUNT             60
#endif

#ifndef PROCTAB_DEF_CAPACITY
#define PROCTAB_DEF_CAPACITY     128
#endif

#ifndef SOCKTAB_DEF_CAPACITY
#define SOCKTAB_DEF_CAPACITY    64
#endif

#ifndef SOCKET_TIMEOUT
#define SOCKET_TIMEOUT   60.    /* in seconds */
#endif


#define SOCKFLAGS     (MSG_DONTWAIT | MSG_NOSIGNAL)

#define ht_length(a)  ((a * 3)/2)
#define hash(a, b)    (((a * 3) / 2) % b)


#define allocCPU(a,b,c) (((a) * ((10000 * (b)) / (c))) / 10000)

#ifdef DEBUG
#define syslog(tag, fmt, args...)  printf(fmt"\n" , ## args)
#endif

#define is_marked(a) (a->flags & TMP_FLAG)
#define mark(a)      {a->flags |= TMP_FLAG;}
#define unmark(a)    {a->flags &= ~TMP_FLAG;}

extern void dispatcher(struct svc_req *rqstp, SVCXPRT *transp);
extern void daemonize (void);
extern uint create_auth_code (void);

struct sockaddr_in srv_addr;
struct in_addr myaddr;

int prognum = CIS_PROG;

struct host_info *host_list = NULL;

struct timeval timeout={5,0};
struct timeval current_tv;

struct monitor_msg monbuff;
char *buffptr = monbuff.data;
#define bufflen (buffptr - (char *) &monbuff)
int msglen;
int monitor_socket;
float interval = DEF_INTERVAL;
struct monitor_reply reply;

char *config_file = NULL;

int accept_new_hosts = 0;

char *commands[]   = {"#", "PerfTypes", "PerfValues", "Reliability", "AcceptNewHost"};
char *rel_types[]  = {"minimal", "low", "average", "high", "maximal"};
char *yesno[]      = {"no", "yes"};

char **perf_types = NULL;
int nperf_types = 0;;
	
void parse_args(int argc, char **argv)
{
        int c;

	while ((c = getopt(argc, argv, "i:c:")) != EOF) {

                switch (c) {
                case 'i':
                        interval = atof(optarg);
                        if (interval == 0.) {
                                syslog(LOG_WARNING, "interval cannot be set to zero. reset to default %5.2f [s]", DEF_INTERVAL);
                                interval = DEF_INTERVAL;
                        }
                        break;
                case 'c':
                        config_file = optarg;
			break;
                }
        }
}

int parse_cmd (char *cmds[], int n, char **c)
{
	int i;

	*c += strspn (*c, " \t");

	for (i = 0; i < n; i++)
		if (!strncasecmp (*c, cmds[i], strlen (cmds[i]))) {
			*c += strlen(cmds[i]);
			return (i);
		}
	return -1;
}

int parse_config(void)
{
	struct host_info *host;
        struct hostent *hent;
	FILE *file = NULL;
	int line = 0;
	char name[MAXHOSTNAMELEN];
	char *c, *type;
	int i, cmd;
	int rel = 0;
	float *performance = NULL;
	
	int buflen = 0;
	char *buf  = NULL;

	/*
         * Open the config file.
         */
	if (!(file = fopen(config_file ? config_file : DEF_CONFIG, "r"))) {
		syslog(LOG_ERR, "cannot open config file %s", config_file);
		exit (1);
	}

	while (getline(&buf, &buflen, file) != -1) {

		line++;
		c = buf;
		
		cmd = parse_cmd(commands, CMD_TABLEN, &c);
		if (cmd == 0) continue;

		c += strspn(c, " \t");       /* strip white spaces */

                switch (cmd) {
                        
		case CMD_PERFTYPES:

			c = strdup(c);
			type = strtok(c, ", \t\n");

			do {
				perf_types = realloc(perf_types, (nperf_types+1)* sizeof(char *));
				if (!perf_types)
					goto nomem;

				perf_types[nperf_types++] = type;

			} while ((type = strtok(NULL, ", \t\n")));

			performance = calloc(nperf_types, sizeof(float));

			break;

		case CMD_PERFVALUES:

			for (i = 0; *c; i++, c++) {

				performance[i] = atof(c);

				c = strchr(c, ',');
				if (i == nperf_types || !c)
					break;

				c++;
			}
			break;

		case CMD_RELIABILITY:

			rel = parse_cmd(rel_types, REL_TABLEN, &c);
			if (rel == -1)
				goto error;

			break;

		case CMD_ACCEPTNEWHOST:

			accept_new_hosts = parse_cmd(yesno, 2, &c);
			break;

		default:

			if (sscanf(c, "%s", name) != 1 || *name == '\n')
				continue;

			if ((hent = gethostbyname(name)) == NULL) {
				syslog(LOG_ERR, "cannot get the address of host %s", name);
				exit(1);
			}

			host = calloc(1, sizeof (struct host_info));
			if (!host)
				goto nomem;

			host->name = strdup(name);
			if (!host->name)
				goto nomem;
			
			for (i = 0; hent->h_addr_list[i]; i++)
                                memcpy(&host->addrlist[i], hent->h_addr_list[i], hent->h_length);
			host->naddr = i;

			host->hostinfo.reliability = rel;
			host->performance = malloc(nperf_types * sizeof(float));
			if (!host->performance)
				goto nomem;

			memcpy (host->performance, performance, nperf_types*sizeof(float));

			host->next = host_list;
			host_list = host;
			
			break;
		}
	}
	fclose (file);

       	return 1;
nomem:
	syslog(LOG_ERR, "not enough memory");
       	exit(1);

error:
	syslog(LOG_ERR, "bad entry in config file at line %d", line);
	exit (1);
}


void free_host(struct host_info *host)
{
        free(host->name);
        free(host->proc_ht);
        free(host->sock_ht);
        free(host);
}

void delete_host(struct host_info *host)
{
        static struct host_info *h;
        
        if (host == host_list)
                host_list = host->next;
        else {
                for (h = host_list; h->next && h->next != host; h = h->next);
                
                if (h->next != host)
                        return; /* Not found */
                
                h->next = host->next;
        }
        free_host(host);
}

struct host_info *find_host(struct in_addr addr)
{
        struct host_info *h;
        struct in_addr *a;

	if (addr.s_addr == 0x0100007f)
                addr = myaddr;
	
        for (h = host_list; h; h = h->next) {
		if (h->hostinfo.addr.s_addr == addr.s_addr)
                        return h;
                for (a = h->addrlist; a < h->addrlist + h->naddr; a++)
                        if (a->s_addr == addr.s_addr)
                                return h;
        }
        return NULL;
}

/*
 * Hash tables manipulation functions
 */

/*
 * Searching in the proctable. If the requested process was not found, a
 * pointer to the next free cell is returned.
 */
static inline struct cis_procinfo **proc_ht_find(struct host_info *host, unsigned int pid)
{
        struct cis_procinfo **hp;

        if (!host->monitor[PROCMON])
                return NULL;
        
        hp = host->proc_ht + hash(pid, host->proc_ht_length);
        
        while (*hp && (*hp)->pid != pid)

                if (++hp == host->proc_ht + host->proc_ht_length)
                        hp = host->proc_ht;
        
        return (hp);
}

static inline struct cis_procinfo *proc_find(struct host_info *host, unsigned int pid)
{
        return *proc_ht_find (host, pid);
}

static inline void proc_ht_remove(struct host_info *host, struct cis_procinfo **hole)
{
        struct cis_procinfo **hp = hole;

        *(hole) = NULL;

        for(;;) {
                if (++hp == host->proc_ht + host->proc_ht_length)
                        hp = host->proc_ht;

                if (!*hp) break;

                if (!(*proc_ht_find(host, (*hp)->pid)))
                        *hole = *hp, *hp = NULL, hole = hp;
        }
}


/*
 * Enlarge the process table in ratio 3:2
 */
int expand_proc_ht(struct host_info *host)
{
        struct cis_procinfo **new_table, **old_table, **pp;
        int old_length;

        if (!(new_table = calloc(ht_length(host->proc_ht_length),
                                 sizeof (struct cis_procinfo *))))
                return (FALSE);

        old_table  = host->proc_ht;
        old_length = host->proc_ht_length;
        
        host->proc_ht          = new_table;
        host->proc_ht_capacity = host->proc_ht_length;
        host->proc_ht_length   = ht_length(host->proc_ht_capacity);

        /* Rehash the table */
        for (pp = old_table; pp < old_table + old_length; pp++)
                if (*pp)
                        *(proc_ht_find(host, (*pp)->pid)) = *pp;

        free (old_table);

        return (TRUE);
}

/*
 * When parent exits, ppid for all children should be set to 1 (init).
 * In order to keep the process tree consistent.
 */
void proc_release_children(struct host_info *host, int pid)
{
        struct cis_procinfo *p;
        
        for (p = host->proclist; p; p = p->next)
                if (p->ppid == pid)
                        p->ppid = 1;
}

/*
 * Searching in the socktable. If the requested socket was not found, a
 * pointer to the next free cell is returned.
 */
static inline struct cis_sockinfo **sock_ht_find(struct host_info *host, struct cis_sockinfo *socket)
{
        struct cis_sockinfo **hs;

        if (!host->monitor[SOCKMON])
                return NULL;
        
        hs = host->sock_ht + hash(socket->sport, host->sock_ht_length);
        
        while (*hs && memcmp(*hs, socket, SOCKINFO_HDRLEN))
                if (++hs == host->sock_ht + host->sock_ht_length)
                        hs = host->sock_ht;

        return (hs);
}

static inline struct cis_sockinfo *sock_find(struct host_info *host,  struct cis_sockinfo *s)
{
        return *sock_ht_find (host, s);
}

static inline void sock_ht_remove(struct host_info *host, struct cis_sockinfo **hole)
{
        struct cis_sockinfo **hs = hole;

        *(hole) = NULL;

        for(;;) {
                if (++hs == host->sock_ht + host->sock_ht_length)
                        hs = host->sock_ht;

                if (!*hs) break;

                if (!(*sock_ht_find(host, *hs)))
                        *hole = *hs, *hs = NULL, hole = hs;
        }
}

/*
 * Enlarge the socket table in ratio 3:2
 */
int expand_sock_ht(struct host_info *host)
{
        struct cis_sockinfo **new_table, **old_table, **sp;
        int old_length;

        if (!(new_table = calloc(ht_length(host->sock_ht_length),
                                 sizeof (struct cis_sockinfo *))))
                return (FALSE);

        old_table  = host->sock_ht;
        old_length = host->sock_ht_length;

        host->sock_ht          = new_table;
        host->sock_ht_capacity = host->sock_ht_length;
        host->sock_ht_length   = ht_length(host->sock_ht_capacity);

        /* Rehash the table */
        for (sp = old_table; sp < old_table + old_length; sp++)
                if (*sp)
                        *(sock_ht_find(host, *sp)) = *sp;

        free (old_table);

        return (TRUE);
}

/*
 * End of hash tables manipulation functions.
 */

/*
 * Process related functions
 */
/*int slice (struct host_info *host, int niceval)
{
        int slice = niceval;
        
        if (niceval < 0)
                slice = -niceval;
        if (slice > 20)
                slice = 20;
        slice = (slice * host->defprio + 10) / 20 + host->defprio;

        if (niceval >= 0) {
                slice = 2*host->defprio - slice;
                if (!slice)
                        slice = 1;
        }
        return (slice);
        } */

int count_pCPU(struct host_info *host, struct process_private_info *p, unsigned long current_time)
{
        int pCPU = 0;

        /*
         * No usage.
         */
        if (!p->interval)
                goto out;
        
        /*
         * No change during 2*'interval', process is sleeping.
         */
        if (current_time > p->last->time + 2*p->interval)
                goto out;

        /*
         * Hold the last usage during 'interval' after last change.
         */
        pCPU  = (10000 * p->CPU_used) / p->interval;

        if (current_time < p->last->time + p->interval)
                goto out;

        /*
         * Between, usage will fall down to zero.
         */
        pCPU *= p->last->time + 2*p->interval - current_time;
        pCPU /= p->interval;

    out:
        return pCPU;
}

int availableCPU(struct host_info *host, unsigned long clk)
{
        struct cis_procinfo *p;
        struct process_private_info *pinfo;
        struct monitor_info *monitor = host->monitor[PROCMON];
        unsigned long psum_tmp, slsum_tmp, available_tmp;
        unsigned long available = 0;
        unsigned long min_interval = 2*monitor->clk_interval;
        unsigned long psum = 0;
        unsigned long slsum = 0;
        int n = 0;

        for (p = host->proclist; p; p = p->next) {
                pinfo = (struct process_private_info *) p->priv;
                n++;
                unmark(p);
                
                /* What CPU usage process has ? */
                if (pinfo->interval) {
                        p->pCPU = count_pCPU(host, pinfo, clk);
                        slsum += p->priority;
                        psum += p->pCPU;
                        mark(p);
                        
                        /*
                         * Minimal process interval is needed for deciding whether
                         * the CPU was busy all the time or it was idle some time.
                         */
                        if (pinfo->interval < min_interval)
                                min_interval = pinfo->interval;
                        continue;
                }
                p->pCPU = 0;

                if (clk < pinfo->last->time + 2*pinfo->init_interval) {
                        p->pCPU = 10000;

                        if (clk > pinfo->last->time + pinfo->init_interval) {
                                p->pCPU *= pinfo->last->time + 2*pinfo->init_interval - clk;
                                p->pCPU /= pinfo->init_interval;
                        }
                }
        }

        host->hostinfo.procnum = n;

        if (!n)
                return (0);

        if (!psum)
                return (10000);

        if (host->idle_at < monitor->local_time - min_interval && psum < 10000) {
                /*
                 * If CPU was not idle and there was terminated process
                 * (i.e. pCPU sum is < 100 %), we must recompute
                 * needs of all 'greedy' processes.
                 */
                available = 10000 - psum;

                slsum_tmp = 0; psum_tmp = psum;
                for (p = host->proclist; p; p = p->next) {
                        if (!is_marked(p)) continue;

                        if (p->pCPU < allocCPU(psum, p->priority, slsum)) {
                                psum_tmp -= p->pCPU;
                                unmark(p);
                        }
                        else
                                slsum_tmp += p->priority;

                }
                slsum = slsum_tmp;
                psum = psum_tmp;
                
                for (p = host->proclist; p; p = p->next) {
                        if (!is_marked(p)) continue;

                        p->pCPU = allocCPU(psum + available, p->pCPU, psum);
                        unmark(p);
                }
        }

        /*
         * Estimate how much CPU time a process with default priority
         * can get (default response time).
         */
        slsum = host->defprio + host->slsum;
        available = 20000;
        available_tmp = 10000;

        while (available > available_tmp) {

                available = available_tmp;
                slsum_tmp = host->defprio;

                for (p = host->proclist; p; p = p->next) {
                        if (!p->pCPU) continue;

                        if (p->pCPU < allocCPU(available, p->priority, slsum)) {
                                available_tmp -= p->pCPU;
                                p->pCPU = 0;
                        } else
                                slsum_tmp += p->priority;
                }
                slsum = slsum_tmp;
        }

        available = allocCPU(available, host->defprio, slsum);

        return (available);
}


struct cis_procinfo *proc_add(struct host_info *host, struct cis_procinfo *p)
{
        struct cis_procinfo *process, **pp;
        struct process_private_info *pinfo;
        struct monitor_info *monitor = host->monitor[PROCMON];
        struct history *hp;
        struct cis_procinfo *q;
        int interval = 0;
        unsigned long time;
        
        if (host->proc_ht_use == host->proc_ht_length)
                return NULL;

        process = calloc(1, sizeof (struct cis_procinfo));
        if (!process)
                return NULL;

        if (p) memcpy(process, p, sizeof (struct cis_procinfo));

        pinfo = calloc(1, sizeof (struct process_private_info));
        if (!pinfo) {
                free(process);
                return NULL;
        }

        time = process->start_time;
        if (!time)
                time = monitor->local_time - monitor->clk_interval;

        /* Recompute start_time to GMT time. It's ugly but we have to cope
         * with possibly very high unsigned longs */
        pinfo->GMT_start_time = monitor->start_tv.tv_sec;
        if (time < monitor->start_clk)
                pinfo->GMT_start_time -= (monitor->start_clk - time)/monitor->ct_per_sec;
        else
                pinfo->GMT_start_time += (time - monitor->start_clk)/monitor->ct_per_sec;
         
        /* Process history initialization. */
        for (hp = pinfo->history;
             hp < pinfo->history+PROC_HISTORY_LENGTH;
             hp++)
                hp->time = time;
        pinfo->last = pinfo->history;

        process->priv = pinfo;
        process->flags = CONSISTENT;

        list_insert(host->proclist, process);

        /* How much to wait for first usage ? */
        if (host->slsum)
                for (q = host->proclist; q; q = q->next) {
                        q->pCPU = count_pCPU(host, q->priv, monitor->local_time);
                        /* Compute time to wait for CPU usage (i.e. the sum of
                         * slices of all CPU intensive processes with higher
                         * priority than the new process has */
                        if (q->priority >= process->priority &&
                            q->pCPU > ((10000 * q->priority) / host->slsum))
                                interval += q->priority;
                }

        if (interval < monitor->clk_interval)
                pinfo->init_interval = monitor->clk_interval;
        else
                pinfo->init_interval = interval;

        host->slsum += process->priority;

        host->proc_ht_use++;
        if ((host->proc_ht_use + 1) > host->proc_ht_capacity)
                expand_proc_ht(host);

        pp = proc_ht_find(host, process->pid);
        *pp = process;
        
        return process;
}

void proc_remove(struct host_info *host, struct cis_procinfo *p)
{
        struct cis_procinfo **hp = proc_ht_find(host, p->pid);
        struct cis_procinfo *process = *hp;
        struct proc_private_info *pinfo;

        if (!process) return;

        *hp = NULL;
        list_remove(host->proclist, process);
        host->slsum -= process->priority;
        proc_release_children(host, process->pid);

        pinfo = process->priv;

        free(process->priv);
        free(process);

        host->proc_ht_use--;

        proc_ht_remove(host, hp);
}

void proc_update(struct host_info *host, struct cis_procinfo *oldp, struct cis_procinfo *newp)
{
        struct history *hp;
        struct process_private_info *pinfo = oldp->priv;
        struct monitor_info *monitor = host->monitor[PROCMON];
        unsigned short ctps = host->hostinfo.ct_per_sec;
        unsigned long change;

        if (oldp->priority != newp->priority)
                host->slsum += newp->priority - oldp->priority;

        change = newp->utime - oldp->utime + newp->stime - oldp->stime;

        if (change) {
                pinfo->last++;
                if (pinfo->last == pinfo->history + PROC_HISTORY_LENGTH)
                        pinfo->last = pinfo->history;
                
                hp = pinfo->last;
                
                pinfo->CPU_used -= hp->change;
                
                pinfo->interval  = monitor->local_time - hp->time;
                hp->time         = monitor->local_time;
                hp->change       = change;
                
                pinfo->CPU_used += change;
                
                pinfo->utime_sec = newp->utime/ctps;
                pinfo->stime_sec = newp->stime/ctps;
        }
        memcpy(oldp, newp, PROCINFO_LEN);

        oldp->flags |= CONSISTENT;
}

/*
 *
 */
void init_net_history(struct net_history *h, unsigned long time)
{
        struct history *hp;
        
        for (hp = h->history;
             hp < h->history+NET_HISTORY_LENGTH;
             hp++)
                hp->time = time;
        h->last = h->history;
}

void net_history_add(struct net_history *h, unsigned long time, unsigned long change)
{
        struct history *hp;

        h->last++;
        if (h->last == h->history + NET_HISTORY_LENGTH)
                h->last = h->history;

        hp = h->last;
        
        h->bytes    += change - hp->change;
        h->interval = time - hp->time;

        hp->time     = time;
        hp->change   = change;
}

/*
 * Socket related functions
 */
struct cis_sockinfo *sock_add(struct host_info *host, struct cis_sockinfo *s)
{
        struct cis_sockinfo *socket, **hs;
        struct socket_private_info *sinfo;
        unsigned long time = host->monitor[SOCKMON]->local_time;

        if (host->sock_ht_use == host->sock_ht_length)
                return NULL;
        
        socket = calloc(1, sizeof (struct cis_sockinfo));
        if (!socket)
                return NULL;
        memcpy(socket, s, sizeof (struct cis_sockinfo));

        sinfo = calloc(1, sizeof (struct socket_private_info));
        if (!sinfo) {
                free(socket);
                return NULL;
        }

        list_insert(host->socklist, socket);

        init_net_history(&sinfo->sent_history, time);
        init_net_history(&sinfo->rcvd_history, time);

        socket->priv = sinfo;
        socket->flags = CONSISTENT;

        host->sock_ht_use++;
        if ((host->sock_ht_use + 1) > host->sock_ht_capacity)
                expand_sock_ht(host);

        hs = sock_ht_find(host, socket);
        *hs = socket;

        return socket;
}

void sock_update(struct host_info *host, struct cis_sockinfo *olds, struct cis_sockinfo *news)
{
        struct socket_private_info *sinfo = olds->priv;
        unsigned long time = host->monitor[SOCKMON]->local_time;
        int val;

        if ((val = news->sent - olds->sent))
                net_history_add(&sinfo->sent_history, time, val);
        if ((val = news->rcvd - olds->rcvd))
                net_history_add(&sinfo->rcvd_history, time, val);

        memcpy(olds, news, SOCKINFO_LEN);

        olds->flags |= CONSISTENT;
}

void sock_remove (struct host_info *host, struct cis_sockinfo *s)
{
        struct cis_sockinfo **hs = sock_ht_find(host, s);

        if (!(*hs))
                return;
        
        (*hs)->flags &= ~CONSISTENT;
        sock_ht_remove(host, hs);
        host->sock_ht_use--;
}

void clear_sockets(struct host_info *host, unsigned long time)
{
        struct cis_sockinfo *s, *tmp;
        struct socket_private_info *sinfo;

        time -= SOCKET_TIMEOUT * host->hostinfo.ct_per_sec;

        for (s = host->socklist; s;) {
                sinfo = s->priv;
                tmp = NULL;

                if (!(s->flags&CONSISTENT) &&
                    sinfo->sent_history.last->time < time &&
                    sinfo->rcvd_history.last->time < time)
                        tmp = s;

                s = s->next;

                if (tmp) {
                        list_remove(host->socklist, tmp);
                        free(tmp->priv);
                        free(tmp);
                }
        }
}

/*
 * Network device related functions
 */
static inline struct cis_netdevinfo *netdev_find(struct host_info *host, unsigned char id)
{
        struct cis_netdevinfo *d;

        for (d = host->netdevlist; d; d = d->next)
                if (d->id == id)
                        return d;
        return NULL;
}

struct cis_netdevinfo *netdev_add (struct host_info *host, struct cis_netdevinfo *dev)
{
        struct cis_netdevinfo *device = calloc(1, sizeof (struct cis_netdevinfo));
        struct netdev_private_info *ndinfo;
        unsigned long time = host->monitor[NETMON]->local_time;

        if (!device)
                return NULL;
        memcpy(device, dev, NETDEVINFO_LEN);

        ndinfo = calloc(1, sizeof (struct netdev_private_info));
        if (!ndinfo) {
                free(device);
                return NULL;
        }

        init_net_history(&ndinfo->rxbytes_history, time);
        init_net_history(&ndinfo->txbytes_history, time);
        init_net_history(&ndinfo->collisions_history, time);

        device->priv = ndinfo;
        device->flags = CONSISTENT;
        list_insert(host->netdevlist, device);

        return device;
}

void netdev_update(struct host_info *host, struct cis_netdevinfo *oldd, struct cis_netdevinfo *newd)
{
        struct netdev_private_info *ndinfo = oldd->priv;
        unsigned long time = host->monitor[NETMON]->local_time;
        int val;
        
        if ((val = newd->rx_bytes - oldd->rx_bytes))
                net_history_add(&ndinfo->rxbytes_history, time, val);
        if ((val = newd->tx_bytes - oldd->tx_bytes))
                net_history_add(&ndinfo->txbytes_history, time, val);
        if ((val = newd->collisions - oldd->collisions))
                net_history_add(&ndinfo->collisions_history, time, val);

        memcpy (oldd, newd, NETDEVINFO_LEN);
        
        oldd->flags |= CONSISTENT;
}

int monitor_idle(struct monitor_info *monitor)
{
//	return (monitor_time(monitor) > (monitor->local_time + 2*monitor->clk_interval));

	return (tvaldiff (current_tv, monitor->time_tv) > 3*interval);
}

unsigned long net_count_rate(struct net_history *h, struct clnt_req *req)
{
        unsigned long bytes = 0;
        unsigned long time  = req->time;
        unsigned long cticks = req->monitor->ct_per_sec;
        unsigned long bound = time - req->interval * cticks;
        struct history *hp;

        if (h->last->time < bound)
                return 0;

        if (h->last->time - h->interval > bound)
                return (h->bytes / (((float) (time - h->last->time + h->interval))
                        / cticks));

        for (hp = h->last; hp->time >= bound;) {

                bytes += hp->change;

                if (--hp < h->history)
                        hp = h->history + NET_HISTORY_LENGTH - 1;

                if (hp == h->last)
                        break;
        }

        return (bytes / req->interval);
}


/*
 * Functions for parsing message from monitors
 */

#define get(ptr, len) \
        {                                            \
               if ((bufflen + len) > msglen)         \
                      goto next;                     \
               memcpy(ptr, buffptr, len);            \
                      buffptr += len;                \
        }

#define get_object(obj) get(&obj, sizeof(obj))
#define get_change(obj) \
        {                                    \
               get_object(change);           \
               obj += change;                \
        }

#define buffer_get(obj) \
        {                                                                    \
               if ((bufflen + sizeof(*(obj))) > len)                         \
                      goto next;                                             \
               else {                                                        \
                      memcpy(obj, buffptr, sizeof(*(obj)));                  \
                      buffptr += sizeof(*(obj));                             \
               }                                                             \
        }

void clear_proclist(struct host_info *host)
{
        while (host->proclist)
                proc_remove(host, host->proclist);
        memset(host->proc_ht, 0, host->proc_ht_length * sizeof (struct cis_procinfo *));
}

void clear_socklist(struct host_info *host)
{
        struct cis_sockinfo *socket;
        
        while (host->socklist) {
                socket = host->socklist;
                host->socklist = host->socklist->next;
                free(socket);
        }
        memset(host->sock_ht, 0, host->sock_ht_length * sizeof (struct cis_sockinfo *));
}

void clear_netdevlist(struct host_info *host)
{
        struct cis_netdevinfo *device;
        
        while (host->netdevlist) {
                device = host->netdevlist;
                host->netdevlist = host->netdevlist->next;
                free(device);
        }
}

int add_monitor(struct host_info *host, int msglen, struct sockaddr_in *mon_addr)
{
        unsigned long  start_clk;
        int naddr, i;
        struct in_addr addrlist[MAX_ADDR_NUM];
        struct utsname uname;
        int ct_per_sec, defprio;
        unsigned long totalram, totalswap;
        struct in_addr addr = mon_addr->sin_addr;
        unsigned short port = mon_addr->sin_port;

        int new_host = FALSE;
        struct monitor_info *monitor = NULL;
        struct hostent *hent;
        int mtype = monitor_type(monbuff.tag);

        /*
         * Only init message is accepted for new monitor.
         */
        if (!init_msg(monbuff.tag)) {
//                syslog(LOG_WARNING, "wrong init message from monitor");
                return -EINVAL;
        }
        /*
         * Parse init message
         */
        buffptr = (char *) monbuff.data;
        
        get_object(start_clk);
        get_object(ct_per_sec);
        get_object(naddr);
        for (i = 0; i < naddr; i++)
                get_object(addrlist[i]);
        
        switch (mtype) {
                
        case PROCMON:
                get_object(uname);
                get_object(defprio);
                get_object(totalram);
                get_object(totalswap);
                break;
                
        case SOCKMON:
        case NETMON:
                break;
        }
        /*
         * Init message was ok.
         */
        if (host)
                monitor = host->monitor[mtype];

        if (!monitor && !(monitor = calloc(1, sizeof (struct monitor_info)))) {
                syslog(LOG_WARNING, "not enough memory for monitor");
                return -ENOMEM;
        }
        monitor->port = port;
        monitor->clk_interval = interval*ct_per_sec;

        monitor->start_clk  = start_clk;
        monitor->ct_per_sec = ct_per_sec;
        monitor->start_tv   = current_tv;
        monitor->seqnum     = monbuff.seqnum;
        monitor->flags      = CONSISTENT;
        
        if (!host) {
                if (!(host = calloc(1, sizeof (struct host_info)))) {
                        syslog(LOG_WARNING, "not enough memory for monitor");
                        return -ENOMEM;
                }
                new_host = TRUE;
        }
        
        switch (mtype) {
                
        case PROCMON:
                if (!host->proc_ht) {
                        if (!(host->proc_ht = calloc(ht_length(PROCTAB_DEF_CAPACITY),
                                                     sizeof (struct cis_procinfo *)))){
                                if (new_host)
                                        free_host(host);
                                free(monitor);
                                return -ENOMEM;
                        }
                        host->proc_ht_capacity = PROCTAB_DEF_CAPACITY;
                        host->proc_ht_length   = ht_length(PROCTAB_DEF_CAPACITY);
                        host->proc_ht_use      = 0;
                        host->slsum = 0;
                } else
                        clear_proclist(host);

                host->defprio            = defprio;
                host->hostinfo.totalram  = totalram;
                host->hostinfo.totalswap = totalswap;

                break;
                
        case SOCKMON:
                if (!host->sock_ht) {
                        if (!(host->sock_ht = calloc(ht_length(SOCKTAB_DEF_CAPACITY),
                                                     sizeof (struct cis_sockinfo *)))){
                                if (!new_host)
                                        free_host(host);
                                free(monitor);
                                return -ENOMEM;
                        }
                        host->sock_ht_capacity = SOCKTAB_DEF_CAPACITY;
                        host->sock_ht_length   = ht_length(SOCKTAB_DEF_CAPACITY);
                        host->sock_ht_use      = 0;
                } else
                        clear_socklist(host);

                break;
                
        case NETMON:

                clear_netdevlist(host);
                break;
	}

	host->hostinfo.ct_per_sec = ct_per_sec;

        if (!host->monitor[mtype])
                host->monitor[mtype] = monitor;
        
        if (new_host) {
		memcpy(host->addrlist, addrlist, sizeof (addrlist));
		host->naddr = naddr;

		host->hostinfo.addr = addr;
                
                if (!(hent = gethostbyaddr((char *) &addr.s_addr, sizeof (addr.s_addr), AF_INET)))
                        syslog(LOG_WARNING, "unable to get hostname for %s\n", inet_ntoa(addr));
                else
                        host->name = strdup(hent->h_name);
                
                memcpy(&host->uname, &uname, sizeof (struct cis_uname));

                host->next = host_list;
                host_list  = host;
        }

        return 1;
    next:
        return -E2BIG;
}

        
void proc_test(struct host_info *host, struct cis_procinfo *p)
{
        struct cis_procinfo *process = proc_find(host, p->pid);

        if (!process)
                proc_add(host, p);
        else
                proc_update(host, process, p);
}

#define proc_code(el)   OFFSET(cis_procinfo, el)

void proc_change(struct host_info *host, unsigned int pid)
{
        struct cis_procinfo proc, *process = proc_find(host, pid);
        char code;
        int change;

        get_object(code);

        if (code == 1) {/* New process */
                get(&proc, PROCINFO_LEN);
                proc_test(host, &proc);
                return;
        }
        if (code == 2) {/* Terminated process */
                get(&proc, PROCINFO_LEN);
                proc_remove(host, process);
                return;
        }
        
        if (process)
                proc = *process;
        else {
                /* Change of nonexisting process */
                memset(&proc, 0, sizeof(proc));
                proc.pid = pid;
                sprintf(proc.cmd, "unknown");
                proc.priority = host->defprio;
                process = proc_add(host, &proc);
        }

        while (code) {
                switch (code) {
                case proc_code(utime):    get_change(proc.utime);    break;
                case proc_code(stime):    get_change(proc.stime);    break;
                case proc_code(priority): get_change(proc.priority); break;
                case proc_code(rss):      get_change(proc.rss);      break;
                case proc_code(vm):       get_change(proc.vm);       break;
                case proc_code(ppid):     get_object(proc.ppid);     break;
                case proc_code(uid):      get_object(proc.uid);      break;
                case proc_code(cmd):      get_object(proc.cmd);      break;
                }
                get_object(code);
        }
        proc_update(host, process, &proc);
    next:
}

void sock_test(struct host_info *host, struct cis_sockinfo *s)
{
        struct cis_sockinfo *socket = sock_find(host, s);

        if (!socket)
                sock_add(host, s);
        else
                sock_update(host, socket, s);
}

#define sock_code(el)   OFFSET(cis_sockinfo, el)

void sock_change(struct host_info *host, struct cis_sockinfo *s)
{
        struct cis_sockinfo sock, *socket = sock_find(host, s);
        char code;
        int change;

        get_object(code);

        if (socket)
                sock = *socket;
        else {
                /* Change of nonexisting socket */
                memset(&sock, 0, sizeof(sock));
                memcpy(&sock, s, SOCKINFO_HDRLEN);
                socket = sock_add(host, &sock);
        }

        if (code == 1) {/* New socket */
                get(&sock + SOCKINFO_HDRLEN, SOCKINFO_LEN-SOCKINFO_HDRLEN);
                sock_update(host, socket, &sock);
                return;
        }
        if (code == 2) {/* Closed socket */
                get(&sock + SOCKINFO_HDRLEN, SOCKINFO_LEN-SOCKINFO_HDRLEN);
                sock_update(host, socket, &sock);
                sock_remove(host, socket);
                return;
        }

        while (code) {
                switch (code) {
                case sock_code(pid):      get_object(sock.pid);      break;
                case sock_code(uid):      get_object(sock.uid);      break;
                case sock_code(sent):     get_change(sock.sent);     break;
                case sock_code(rcvd):     get_change(sock.rcvd);     break;
                }
                get_object(code);
        }
        sock_update(host, socket, &sock);
    next:
}

void netdev_test(struct host_info *host, struct cis_netdevinfo *d)
{
        struct cis_netdevinfo *device = netdev_find(host, d->id);

        if (!device)
                netdev_add(host, d);
        else
                netdev_update(host, device, d);
}

#define netdev_code(el)   OFFSET(cis_netdevinfo, el)

void netdev_change(struct host_info *host, unsigned char id)
{
        struct cis_netdevinfo dev, *device = netdev_find(host, id);
        char code;
        int change;

        get_object(code);

        if (code == 1) {/* New device */
                get(&dev, NETDEVINFO_LEN);
                netdev_test(host, &dev);
                return;
        }
        
        if (device)
                dev = *device;
        else {
                /* Change of nonexisting device */
                memset(&dev, 0, sizeof(dev));
                dev.id = id;
                sprintf(dev.name, "unknown");
                device = netdev_add(host, &dev);
        }

        while (code) {
                switch (code) {
                case netdev_code(tx_bytes):   get_change(dev.tx_bytes);    break;
                case netdev_code(rx_bytes):   get_change(dev.rx_bytes);    break;
                case netdev_code(collisions): get_change(dev.collisions);  break;
                case netdev_code(status):     get_object(dev.status);      break;
                }
                get_object(code);
        }
        netdev_update(host, device, &dev);
    next:
}

void monitor_dispatch(void)
{
        struct cis_procinfo proc, *process, *p;
        struct cis_sockinfo sock, *s;
        struct cis_netdevinfo dev, *d;
	struct in_addr addr;
        unsigned short port;
        socklen_t length;
        int tag;
        struct host_info *host;
        int mtype;
        int return_code;
        struct monitor_info *monitor = NULL;
	unsigned long msg_time, old_idle;
	struct sockaddr_in mon_addr = {AF_INET};

    next:
        msglen = recvfrom(monitor_socket, &monbuff, sizeof(struct monitor_msg),
                       MSG_NOSIGNAL, &mon_addr, &length);
        if (msglen == -1)
                return;
        
        addr = mon_addr.sin_addr;
        port = mon_addr.sin_port;

#ifndef DEBUG
        if (ntohs(port) >= 1024)   /* Authorization check. Only root may bind */
                goto next;         /* ports below 1024 */
#endif
        
        gettimeofday(&current_tv, NULL);

        /*
         * If the originator of the message is not monitor.
         */
        if ((mtype = monitor_type(monbuff.tag)) == 0)
                goto next;

	host = find_host(addr);
	
	if (!host && !accept_new_hosts)
		goto system_error;
	
        if (!host || !host->monitor[mtype] || init_msg(monbuff.tag)) {
                reply.code = -EAGAIN;

                if (init_msg(monbuff.tag))
                        reply.code = add_monitor(host, msglen, &mon_addr);

                sendto(monitor_socket, &reply, sizeof(reply), SOCKFLAGS, &mon_addr, length);
                goto next;
        }
        
        /* Messages from monitors */
        monitor = host->monitor[mtype];
        if (monitor->port != port)
                goto system_error;

        /* Some messages are lost. Ask for full message */
        if ((!(monitor->flags & CONSISTENT) && tag == PROCMON_SYSINFO) ||
            ++monitor->seqnum != monbuff.seqnum ||
            !monbuff.seqnum) {
                
                reply.code = 1;
                sendto(monitor_socket, &reply, sizeof(reply), SOCKFLAGS, &mon_addr, length);
                monitor->seqnum = monbuff.seqnum;
                monitor->flags &= ~CONSISTENT;
        }
        
        buffptr  = (char *) monbuff.data;
        msg_time = monbuff.time;
        tag      = monbuff.tag;

        if (tag != PROCMON_NEW &&
	    tag != PROCMON_CHANGES_ASYNC) {
                monitor->local_time = msg_time;
		monitor->time_tv = current_tv;
	}
	
        switch (tag) {
                unsigned int pid;
                unsigned char devid;
                
                /*************************
                 * Messages from procmon *
                 *************************/
                
        case PROCMON_EXIT:

                clear_proclist(host);
                free(host->monitor[PROCMON]);
                host->monitor[PROCMON] = NULL;
                break;
                
        case PROCMON_FULL:         /* The information about all processes */
                
                if (monbuff.flags & MSG_FIRST) {
                        monitor->flags |= CONSISTENT;

                        for (p = host->proclist; p; p = p->next)
                                p->flags &= ~CONSISTENT;

                        host->proc_ht_use = 0;
                }

                while (buffptr - (char *) &monbuff < msglen) {
                        get(&proc, PROCINFO_LEN);
                        proc_test(host, &proc);

                        host->proc_ht_use++;
                        if ((host->proc_ht_use + 1) > host->proc_ht_capacity)
                                expand_proc_ht(host);
                }

                if (!(monbuff.flags & MSG_LAST) || !(monitor->flags & CONSISTENT))
                        break;

                /*
                 * Free all terminated processes.
                 */
                p = host->proclist;
                while (p) {
                        process = p;
                        p = p->next;
                        if (process->flags&CONSISTENT) continue;
                        
                        proc_remove(host, process);
                }
                break;
                
        case PROCMON_NEW:   /* New process */

                get(&proc, PROCINFO_LEN);
                proc_test(host, &proc);
                break;
                
        case PROCMON_CHANGES:
        case PROCMON_CHANGES_ASYNC:

                while (buffptr - (char *) &monbuff < msglen) {
                        get_object(pid);
                        proc_change(host, pid);
                }
                break;

        case PROCMON_SYSINFO:
                
                old_idle = host->hostinfo.idle;
                get(&host->hostinfo, HOSTINFO_LEN);
                if (old_idle != host->hostinfo.idle)
                        host->idle_at = msg_time;
                break;
                
                /*************************
                 * Messages from sockmon *
                 *************************/
                
        case SOCKMON_EXIT:

                clear_socklist(host);
                free(host->monitor[SOCKMON]);
                host->monitor[SOCKMON] = NULL;
                break;
                
        case SOCKMON_FULL:    /* The information about all sockets */
                
                if (monbuff.flags & MSG_FIRST) {
                        monitor->flags |= CONSISTENT;

                        for (s = host->socklist; s; s = s->next)
                                s->flags &= ~CONSISTENT;

                        host->sock_ht_use = 0;
                }

                while (buffptr - (char *) &monbuff < msglen) {
                        get(&sock, SOCKINFO_LEN);
                        sock_test(host, &sock);

                        host->sock_ht_use++;
                        if ((host->sock_ht_use + 1) > host->sock_ht_capacity)
                                expand_sock_ht(host);
                }

                if (!(monbuff.flags & MSG_LAST) || !(monitor->flags & CONSISTENT))
                        break;
                /*
                 * Free all closed sockets.
                 */
                clear_sockets(host, msg_time);

                for (s = host->socklist; s; s = s->next) {
                        if (s->flags&CONSISTENT) continue;
                        sock_remove(host, s);
                }
                break;

        case SOCKMON_NEW:     /* New sockets */
                
                get(&sock, SOCKINFO_LEN);
                sock_test(host, &sock);
                break;

        case SOCKMON_CHANGES:
                
                while (buffptr - (char *) &monbuff < msglen) {
                        get(&sock, SOCKINFO_HDRLEN);
                        sock_change(host, &sock);
                }

                clear_sockets(host, msg_time);

                break;

                /************************
                 * Messages from netmon *
                 ************************/

        case NETMON_EXIT:         /* Netmon terminated */

                clear_netdevlist(host);
                free(host->monitor[NETMON]);
                host->monitor[NETMON] = NULL;
                break;
                
        case NETMON_FULL:         /* The information about all devices */
                
                if (monbuff.flags & MSG_FIRST) {
                        monitor->flags |= CONSISTENT;

                        for (d = host->netdevlist; d; d = d->next)
                                d->flags &= ~CONSISTENT;
                }

                while (buffptr - (char *) &monbuff < msglen) {
                        get(&dev, NETDEVINFO_LEN);
                        netdev_test(host, &dev);
                }

                if (!(monbuff.flags & MSG_LAST) || !(monitor->flags & CONSISTENT))
                        break;

                /*
                 * Clear status for stopped devices.
                 */
                for (d = host->netdevlist; d; d = d->next) {
                        if (d->flags&CONSISTENT) continue;
                        d->status = 0;
                }
                break;

        case NETMON_NEW:          /* New devices */

                get(&dev, NETDEVINFO_LEN);
                netdev_test(host, &dev);
                break;

        case NETMON_CHANGES:
                
                while (buffptr - (char *) &monbuff < msglen) {
                        get_object(devid);
                        netdev_change(host, devid);
                }
                break;
                
        default:
                goto next;
        }
        
        goto next;

    system_error:
        return_code = -EPERM;
        sendto(monitor_socket, &return_code, sizeof(return_code),
               SOCKFLAGS, &mon_addr, length);
        goto next;
}

void cleanup (int sig)
{
        (void) pmap_unset(prognum, CIS_VER);
        
//        syslog(LOG_INFO, "terminated");
        closelog();
        exit(0);
}

int main (int argc,char *argv[])
{
        struct timeval t;
        
        register SVCXPRT *transp_tcp;
        struct servent *srv_ent;
        struct sockaddr_in addr;
        int flags;
        fd_set rfds;
	int err;
	char myhostname[MAXHOSTNAMELEN];
	struct hostent *hent;

	get_myaddress(&srv_addr);
	if (gethostname(myhostname, MAXHOSTNAMELEN) == -1) {
		syslog(LOG_ERR, "cannot get my hostname");
		exit(1);
	}

	if (!(hent = gethostbyname(myhostname))) {
		syslog(LOG_ERR, "cannot get my address");
		exit(1);
	}
	myaddr = *(struct in_addr *) hent->h_addr;
	
#ifndef DEBUG
        if (getuid()) {
                fprintf(stderr, "You don't have permission to run cis server.\n");
                exit (1);
        }

        openlog("cis", LOG_CONS, LOG_DAEMON);

        if (!getuid()) {
                /*
                 * Look at the portmapper whether there is a cis daemon already registered.
                 */

		if (pmap_getport(&srv_addr, CIS_PROG, CIS_VER, IPPROTO_TCP)) {
			syslog(LOG_ERR, "cis daemon is already registered");
                        exit(1);
                }
        }
#endif

        setlocale(LC_ALL,"");
        setlocale (LC_NUMERIC, "C");

        /*
         * Parse command line for parameters.
         */
	parse_args(argc, argv);
	reply.interval = interval;

	parse_config();
        /*
         * We need to unregister rpc service before terminating ...
         */
        signal(SIGINT, cleanup);
        signal(SIGTERM, cleanup);
        signal(SIGPIPE, SIG_IGN);
        
        /*
         * Initialize random number generator for authentication.
         */
        gettimeofday(&t, NULL);
        srand((unsigned int) t.tv_usec);

        srv_ent = getservbyname("cis", "udp");
        if (srv_ent ==  NULL) {
                syslog(LOG_ERR, "cannot get UDP server port");
                exit(1);
        }

        monitor_socket = socket(PF_INET, SOCK_DGRAM, 0);
        if (monitor_socket == -1) {
                syslog(LOG_ERR, "cannot create UDP server socket");
                exit(1);
        }

        addr.sin_family = AF_INET;
        addr.sin_port = srv_ent->s_port;
        addr.sin_addr.s_addr = htonl (INADDR_ANY);
        err = bind (monitor_socket, (struct sockaddr *) &addr, sizeof (addr));
        if (err < 0) {
                syslog(LOG_ERR, "cannot bind UDP server socket");
                exit(1);
        }

        /*
         * Enable nonblocking mode on socket.
         */
        flags = fcntl(monitor_socket, F_GETFL, 0);
        if (flags == -1) {
                syslog(LOG_ERR, "cannot get UDP server socket flags");
                exit(1);
        }

        if (fcntl(monitor_socket, F_SETFL, (flags|FNDELAY)) == -1) {
                syslog(LOG_ERR, "cannot set UDP server socket flags");
                exit(1);
        };

        
#ifdef DEBUG
        (void) pmap_unset(CIS_PROG, CIS_VER);
#endif

        /*
         * Make a server point and register it with rpcbinder.
         */
        transp_tcp = svctcp_create(RPC_ANYSOCK, 0, 0);
        if (transp_tcp == NULL) {
                syslog(LOG_ERR, "cannot create tcp service");
                exit(1);
        }

        if (!svc_register(transp_tcp, prognum, CIS_VER, dispatcher, IPPROTO_TCP)) {
                syslog(LOG_ERR, "cannot register tcp service with the rpcbinder");
                exit(1);
        }

#ifndef DEBUG
        /*
         * It's time to do the magic to become a daemon.
         */
        daemonize();
#endif

        syslog(LOG_INFO, "started");

        for(;;) {

                rfds = svc_fdset;
                FD_SET(monitor_socket, &rfds);
                
                switch (select (FD_SETSIZE, &rfds, NULL, NULL, NULL)) {

                case -1:
                        continue;

                default:
                        if (FD_ISSET(monitor_socket, &rfds))
                                monitor_dispatch();

                        FD_CLR(monitor_socket, &rfds);
                        svc_getreqset (&rfds);
                }
        }
}
