/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * CIS server daemon - client RPC calls dispatch routines.
 */

#include "cis.h"
#include "cis_xdr.h"
#include "cis_srv.h"

extern bool_t xdr_process_common(XDR *handle, struct cis_procinfo *p);
extern struct host_info *host_list;

/*
 * Output filters
 */
/* For messages from clients */
extern struct timeval current_tv;
extern char **perf_types;
extern int nperf_types;

struct host_info *find_host(struct in_addr addr);

int count_pCPU(struct host_info *host, struct process_private_info *p, unsigned long current_time);
int availableCPU(struct host_info *host, unsigned long clk);
int monitor_idle(struct monitor_info *monitor);
unsigned long net_count_rate(struct net_history *h, struct clnt_req *req);

bool_t xdr_list_req(XDR *handle, struct clnt_req *req)
{
        if (!xdr_u_short(handle, &req->uid))
                return (FALSE);
        if (!xdr_float(handle, &req->interval))
                return (FALSE);

        return (TRUE);
}

bool_t xdr_hostlist(XDR *handle, unsigned char reliability)
{
	struct host_info *host;
	char *end = "";
	
	for (host = host_list; host; host = host->next) {
		if (host->hostinfo.reliability >= reliability)
			xdr_wrapstring(handle, &host->name);
	}
	xdr_wrapstring(handle, &end);

	return (TRUE);
}


bool_t xdr_host_info(XDR *handle, struct clnt_req *req)
{
        struct cis_hostinfo *hi = &req->host->hostinfo;
	char state = HOST_AVAILABLE;
	int i;
        
	hi->CPU_available = availableCPU(req->host, req->time);
	hi->performance = req->host->performance[0];;
	
	for (i = 1; i < nperf_types; i++)
		if (!strcmp(perf_types[i], req->perf_type))
			hi->performance = req->host->performance[i];

        if (!xdr_char(handle, &state))
                return (FALSE);

        if (!xdr_inaddr(handle, &hi->addr))
                return (FALSE);

        if (!xdr_hostinfo(handle, hi))
                return (FALSE);

        return (TRUE);
}

bool_t xdr_host_update(XDR *handle, struct clnt_req *req)
{
        struct in_addr *addr;
        struct cis_hostinfo *hi;
        struct host_info *host;
        char stat;

        for (addr = req->addrtab; addr < req->addrtab+req->nhosts; addr++) {

		stat = HOST_AVAILABLE;
		
                host = find_host(*addr);
                if (!host)
                        stat = HOST_UNKNOWN;
		else if (!host->monitor[PROCMON])
			stat = HOST_NOT_AVAILABLE;
		else if (monitor_idle (host->monitor[PROCMON]))
			stat = HOST_NOT_AVAILABLE;

                if (!xdr_char(handle, &stat))
                        return (FALSE);

                if (stat != HOST_AVAILABLE)
                        continue;

                hi = &host->hostinfo;
                
                if (req->code & CIS_PROCNUM)
                        if (!xdr_u_short(handle, &hi->procnum))
                                return (FALSE);

                if (req->code & CIS_LOADS)
                        if (!xdr_vector(handle, (char *) hi->loads, 3, sizeof(float), (xdrproc_t) xdr_float))
                                return (FALSE);

                if (req->code & CIS_RAM) {
                        if (!xdr_u_long(handle, &hi->totalram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->sharedram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->bufferram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->cachedram))
                                return (FALSE);
                }

                if (req->code & CIS_SWAP) {
                        if (!xdr_u_long(handle, &hi->totalswap))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeswap))
                                return (FALSE);
                }
                
                if (req->code & CIS_CPU) {
                        hi->CPU_available = availableCPU(host, monitor_time(host->monitor[PROCMON]));
                        if (!xdr_u_short(handle, &hi->CPU_available))
                                return (FALSE);
                }

                if (req->code & CIS_IDLE)
                        if (!xdr_u_long(handle, &hi->idle))
                                return (FALSE);
        }

        return (TRUE);
}

#define user_match(a,b) (a == 0xffff || (a ? (a == b) : (b > MIN_USER_UID)))

bool_t xdr_proclist_rep(XDR *handle, struct clnt_req *req)
{
        struct cis_procinfo *p;
        struct process_private_info *pinfo;
	struct host_info *host = req->host;
	unsigned short uid = req->uid;
        int n;

        for (p = host->proclist, n = 0; p; p = p->next) {

		if (!user_match(uid, p->uid))
			continue;

		p->pCPU = count_pCPU(host, p->priv, req->time);
                n++;
        }
        if (!xdr_int(handle, &n))
                return (FALSE);

        if (!n)
                return (TRUE);

        for (p = host->proclist; p; p = p->next) {

		if (!user_match(uid, p->uid))
			continue;

                if (!xdr_process_common(handle, p))
                        return (FALSE);

                pinfo = p->priv;
                
                if (!xdr_u_long(handle, &pinfo->GMT_start_time))
                        return (FALSE);
                if (!xdr_u_long(handle, &pinfo->stime_sec))
                        return (FALSE);
                if (!xdr_u_long(handle, &pinfo->utime_sec))
                        return (FALSE);
                if (!xdr_short(handle, &p->pCPU))
                        return (FALSE);
        }
        return (TRUE);
}

bool_t xdr_socklist_rep(XDR *handle, struct clnt_req *req)
{
        struct host_info *host = req->host;
        struct cis_sockinfo *s;
        struct socket_private_info *sinfo;
	unsigned short uid = req->uid;
        int n;

        for (s = host->socklist, n = 0; s; s = s->next) {
		if (!user_match(uid, s->uid))
			continue;

                sinfo = s->priv;

                sinfo->send_rate = net_count_rate(&sinfo->sent_history, req);
                sinfo->recv_rate = net_count_rate(&sinfo->rcvd_history, req);

                if (!sinfo->send_rate && !sinfo->recv_rate)
                        continue;

                n++;
        }
        if (!xdr_int(handle, &n))
                return (FALSE);

        if (!n)
                return (TRUE);

        for (s = host->socklist; s; s = s->next) {
		if (!user_match(uid, s->uid))
			continue;

                sinfo = s->priv;

                if (!sinfo->send_rate && !sinfo->recv_rate)
                        continue;
                
                if (!xdr_socket_desc(handle, s))
                        return (FALSE);

                if (!xdr_u_int(handle, &s->pid))
                        return (FALSE);
                if (!xdr_u_short(handle, &s->uid))
                        return (FALSE);

                if (!xdr_u_long(handle, &sinfo->send_rate))
                        return (FALSE);
                if (!xdr_u_long(handle, &sinfo->recv_rate))
                        return (FALSE);
        }
        return (TRUE);
}

bool_t xdr_netdevlist_rep(XDR *handle, struct clnt_req *req)
{
        struct host_info *host = req->host;
        struct cis_netdevinfo *dev;
        struct netdev_private_info *ndinfo;
        int n;

        for (dev = host->netdevlist, n = 0; dev; dev = dev->next, n++)
                ;

        if (!xdr_int(handle, &n))
                return (FALSE);

        if (!n)
                return (TRUE);

        for (dev = host->netdevlist; dev; dev = dev->next) {
                ndinfo = dev->priv;

                if (dev->status) {
                        ndinfo->rxbytes_rate    = net_count_rate(&ndinfo->rxbytes_history, req);
                        ndinfo->txbytes_rate    = net_count_rate(&ndinfo->txbytes_history, req);
                        ndinfo->collisions_rate = net_count_rate(&ndinfo->collisions_history, req);
                } else
                        ndinfo->rxbytes_rate = ndinfo->txbytes_rate = ndinfo->collisions_rate = 0;

                if (!xdr_netdev_desc(handle, dev))
                        return (FALSE);

                if (!xdr_u_long(handle, &ndinfo->rxbytes_rate))
                        return (FALSE);
                if (!xdr_u_long(handle, &ndinfo->txbytes_rate))
                        return (FALSE);
                if (!xdr_u_long(handle, &ndinfo->collisions_rate))
                        return (FALSE);
        }
        return (TRUE);
}

void dispatcher(struct svc_req *rqstp, SVCXPRT *transp)
{
        struct monitor_info *monitor = NULL;
        struct host_info *host;
        char err = 0;
        struct in_addr *addr;
        struct clnt_req req;
	unsigned char reliability;

        gettimeofday(&current_tv, NULL);

	if (rqstp->rq_proc == HOST_LIST) {
		if (!svc_getargs(transp, (xdrproc_t) xdr_char, (caddr_t) &reliability))
			goto decode_error;
		if (!svc_sendreply(transp, (xdrproc_t) xdr_hostlist, (caddr_t) (int) reliability))
			goto system_error;
		return;
	}
	
        if (rqstp->rq_proc == HOST_INFO_UPDATE) {
                if (!svc_getargs(transp, (xdrproc_t) xdr_inaddr, (caddr_t) &req.code) ||
                    !svc_getargs(transp, (xdrproc_t) xdr_inaddr, (caddr_t) &req.nhosts))
                        goto decode_error;
                
                req.addrtab = calloc(req.nhosts+1, sizeof (struct in_addr));
                if (!req.addrtab)
                        goto system_error;
                
                for (addr = req.addrtab; addr < req.addrtab+req.nhosts; addr++)
                        if (!svc_getargs(transp, (xdrproc_t) xdr_inaddr, (caddr_t) addr)) {
                                free(req.addrtab);
                                goto decode_error;
                        }
                
                if (!svc_sendreply(transp, (xdrproc_t) xdr_host_update, (caddr_t) &req)) {
                        free(req.addrtab);
                        goto system_error;
                }
                
                free(req.addrtab);
                return;
        }
        
        if (!svc_getargs(transp, (xdrproc_t) xdr_inaddr, (caddr_t) &req.addr))
                goto decode_error;
        
        /* Host address must be specified */
        if (!req.addr.s_addr)
                goto decode_error;
        
        host = find_host(req.addr);
        if (!host) {
                err = HOST_UNKNOWN;
                goto host_error;
        }
        
        switch (rqstp->rq_proc) {
	case HOST_INFO:
        case PROCESS_LIST:
                monitor = host->monitor[PROCMON];
                break;
        case SOCKET_LIST:
                monitor = host->monitor[SOCKMON];
                break;
        case NETDEV_LIST:
                monitor = host->monitor[NETMON];
                break;
        }
        
        if (!monitor) {
                err = HOST_NOT_AVAILABLE;
                goto host_error;
        }

	
	if (monitor_idle(monitor)) {
		err = HOST_NOT_AVAILABLE;
		goto host_error;
	}
        
        req.host    = host;
        req.monitor = monitor;
        req.time    = monitor_time(monitor);
        
	if (rqstp->rq_proc == HOST_INFO) {
		req.perf_type = NULL;
		if (!svc_getargs(transp, (xdrproc_t) xdr_wrapstring, (caddr_t) &req.perf_type))
			goto decode_error;
		if (!svc_sendreply(transp, (xdrproc_t) xdr_host_info, (caddr_t) &req)) {
			free (req.perf_type);
			goto system_error;
		}
		free (req.perf_type);
	}
        
        if (!svc_getargs(transp, (xdrproc_t) xdr_list_req, (caddr_t) &req))
                goto decode_error;
        
        switch (rqstp->rq_proc) {
                
        case PROCESS_LIST:
                if (!svc_sendreply(transp, (xdrproc_t) xdr_proclist_rep, (caddr_t) &req))
                        goto system_error;
                break;
                
        case SOCKET_LIST:
                if (!svc_sendreply(transp, (xdrproc_t) xdr_socklist_rep, (caddr_t) &req))
                        goto system_error;
                break;
                
        case NETDEV_LIST:
                if (!svc_sendreply(transp, (xdrproc_t) xdr_netdevlist_rep, (caddr_t) &req))
                        goto system_error;
                break;
                
        default:
                svcerr_noproc(transp);
                return;
        }
        return;

    host_error:
        if (!svc_sendreply(transp, (xdrproc_t) xdr_char, (caddr_t) &err))
                goto system_error;
        return;

    system_error:
        svcerr_systemerr(transp);
        return;

    decode_error:
        svcerr_decode(transp);
        return;
}
