/*
 * Cluster Information Service client library - client RPC calls to CIS server.
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>

#include "cis.h"
#include "cis_xdr.h"
#include "cis_clnt.h"

#define OFFSET(str, el)  (int) &(((struct str *) 0)->el)
#define SIZE(str, el)    sizeof (((struct str *) NULL)->el)
#define ELEMENT(str, el) OFFSET(str,el), SIZE(str,el)

char *rel_types[] = {"minimal", "low", "average", "high", "maximal"};

struct description cis_hostinfo_desc[] = {
        {0, 0, sizeof(struct cis_hostinfo)},
        {ELEMENT(cis_hostinfo, procnum),       1},
        {ELEMENT(cis_hostinfo, loads[0]),      1},
        {ELEMENT(cis_hostinfo, loads[1]),      1},
        {ELEMENT(cis_hostinfo, loads[2]),      1},
        {ELEMENT(cis_hostinfo, totalram),      1},
        {ELEMENT(cis_hostinfo, freeram),       1},
        {ELEMENT(cis_hostinfo, sharedram),     1},
        {ELEMENT(cis_hostinfo, bufferram),     1},
        {ELEMENT(cis_hostinfo, cachedram),     1},
        {ELEMENT(cis_hostinfo, totalswap),     1},
        {ELEMENT(cis_hostinfo, freeswap),      1},
        {ELEMENT(cis_hostinfo, CPU_available), 1},
        {ELEMENT(cis_hostinfo, reliability),   1},
        {ELEMENT(cis_hostinfo, performance),   1},
};

#define cis_hostinfo_desclen (sizeof (cis_hostinfo_desc)/sizeof (struct description))

struct description cis_procinfo_desc[] = {
        {ELEMENT(cis_procinfo, pid),      PROCINFO_LEN},
        {ELEMENT(cis_procinfo, ppid),     SIZE(cis_procinfo, ppid)},
        {ELEMENT(cis_procinfo, cmd),      SIZE(cis_procinfo, cmd)},
        {ELEMENT(cis_procinfo, priority), 1},
        {ELEMENT(cis_procinfo, utime),    1},
        {ELEMENT(cis_procinfo, stime),    1},
        {ELEMENT(cis_procinfo, rss),      1},
        {ELEMENT(cis_procinfo, vm),       1},
        {ELEMENT(cis_procinfo, pCPU),     1},
};

#define cis_procinfo_desclen (sizeof (cis_procinfo_desc)/sizeof (struct description))

struct description cis_sockinfo_desc[] = {
        {0, OFFSET(cis_sockinfo ,uid),    SOCKINFO_LEN},
        {ELEMENT(cis_sockinfo ,sent),      1},
        {ELEMENT(cis_sockinfo ,rcvd),      1},
};

#define cis_sockinfo_desclen (sizeof (cis_sockinfo_desc)/sizeof (struct description))

struct description cis_netdevinfo_desc[] = {
        {ELEMENT(cis_netdevinfo, name),       NETDEVINFO_LEN},
        {ELEMENT(cis_netdevinfo ,status),     1},
        {ELEMENT(cis_netdevinfo ,rx_bytes),   1},
        {ELEMENT(cis_netdevinfo ,tx_bytes),   1},
        {ELEMENT(cis_netdevinfo ,collisions), 1},
};

#define cis_netdevinfo_desclen (sizeof (cis_netdevinfo_desc)/sizeof (struct description))


#define LAST_1 (1 << (sizeof (int)*8 - 1))

static struct timeval cis_timeout={0,500000};
int cis_errno;
static int cis_result;

static int list_length;
static int update_code;

struct request {
        struct in_addr addr;
        unsigned short uid;
	float interval;
	char *perf_type;
};

int parse_cmd (char *cmds[], int n, char **c)
{
	int i;

	*c += strspn (*c, " \t");

	for (i = 0; i < n; i++)
		if (!strncasecmp (*c, cmds[i], strlen (cmds[i]))) {
			*c += strlen(cmds[i]);
			return (i);
		}
	return -1;
}

static int proc_cmp(const void *l, const void *r)
{
        struct cis_procinfo *left  = (struct cis_procinfo *) l;
        struct cis_procinfo *right = (struct cis_procinfo *) r;

        return (left->pid - right->pid);
}

static int sock_cmp(const void *l, const void *r)
{
        struct cis_sockinfo *left  = (struct cis_sockinfo *) l;
        struct cis_sockinfo *right = (struct cis_sockinfo *) r;
        int res;
        
        if ((res = left->saddr.s_addr - right->saddr.s_addr))
                return (res);
        if ((res = left->type - right->type))
                return (res);
        if ((res = left->sport - right->sport))
                return (res);
        if ((res = left->daddr.s_addr - right->daddr.s_addr))
                return (res);
        if ((res = left->dport - right->dport))
                return (res);

        return (0);
}

static int netdev_cmp(const void *l, const void *r)
{
        struct cis_netdevinfo *left  = (struct cis_netdevinfo *) l;
        struct cis_netdevinfo *right = (struct cis_netdevinfo *) r;

        return (strncmp(left->name, right->name, IFNAMSIZ));
}

bool_t xdr_request(XDR *handle, struct request *req)
{
        if (!xdr_inaddr(handle, &req->addr))
                return (FALSE);
        if (!xdr_u_short(handle, &req->uid))
                return (FALSE);
        if (!xdr_float(handle, &req->interval))
                return (FALSE);

        return (TRUE);
}

static bool_t xdr_hostlist(XDR *handle, char ***hostlistp)
{
	int nhosts = 0;
	char *name , **hostlist = NULL, **newlist;

	do {
		name = NULL;
		if (!xdr_wrapstring(handle, &name))
			goto cleanup;

		newlist = realloc(hostlist, (nhosts+1)*sizeof (char *));
		if (!newlist)
			goto cleanup;

		hostlist = newlist;
		hostlist[nhosts++] = name;

	} while (strlen(name));

	*hostlistp = hostlist;

	return (TRUE);

cleanup:
	if (nhosts) {
		for (; nhosts; nhosts--)
			free(hostlist[nhosts-1]);
		free(hostlist);
	}
	
	return (FALSE);
}

static bool_t xdr_host_req(XDR *handle, struct request *req)
{
	if (!xdr_inaddr(handle, &req->addr))
                return (FALSE);

	if (!xdr_wrapstring(handle, &req->perf_type))
		return (FALSE);

        return (TRUE);
}

static bool_t xdr_host_rep(XDR *handle, struct cis_hostinfo *hi)
{
        if (!xdr_char(handle, &hi->status))
                return (FALSE);

        if (hi->status != HOST_AVAILABLE)
                return (TRUE);

        if (!xdr_inaddr(handle, &hi->addr))
                return (FALSE);

        if (!xdr_hostinfo(handle, hi))
                return (FALSE);

        return (TRUE);
}

static bool_t xdr_hosts_update_req(XDR *handle, struct cis_hostinfo **hi)
{
        int i;
        
        if (!xdr_int(handle, &update_code))
                return (FALSE);

        if (!xdr_int(handle, &list_length))
                return (FALSE);

        for (i = 0; i < list_length; i++, hi++)
                if ((*hi)->addr.s_addr)
                        if (!xdr_inaddr(handle, &(*hi)->addr))
                                return (FALSE);

        return (TRUE);
}

static bool_t xdr_hosts_update_reply(XDR *handle, struct cis_hostinfo **hinfo)
{
        int i;
        struct cis_hostinfo *hi;

        for (i = 0; i < list_length; i++, hinfo++) {
                hi = *hinfo;
                if (!hi->addr.s_addr)
                        continue;

                if (!xdr_char(handle, &hi->status))
                        return (FALSE);

		if (hi->status != HOST_AVAILABLE) {
			hi->CPU_available = 0;
			continue;
		}

                if (update_code & CIS_PROCNUM)
                        if (!xdr_u_short(handle, &hi->procnum))
                                return (FALSE);

                if (update_code & CIS_LOADS)
                        if (!xdr_vector(handle, (char *) hi->loads, 3, sizeof(float), (xdrproc_t) xdr_float))
                                return (FALSE);

                if (update_code & CIS_RAM) {
                        if (!xdr_u_long(handle, &hi->totalram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->sharedram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->bufferram))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->cachedram))
                                return (FALSE);
                }

                if (update_code & CIS_SWAP) {
                        if (!xdr_u_long(handle, &hi->totalswap))
                                return (FALSE);
                        if (!xdr_u_long(handle, &hi->freeswap))
                                return (FALSE);
                }
                
                if (update_code & CIS_CPU)
                        if (!xdr_u_short(handle, &hi->CPU_available))
                                return (FALSE);

                if (update_code & CIS_IDLE)
                        if (!xdr_u_long(handle, &hi->idle))
                                return (FALSE);
        }

        return (TRUE);
}

static bool_t xdr_proc_reply(XDR *handle, struct cis_procinfo **listp)
{
        int length = 0;
        struct cis_procinfo *new_list, *p;

        list_length = 0;
        cis_result  = HOST_AVAILABLE;
        
        if (!xdr_int(handle, &length))
                return (FALSE);

        if (!length)
                return (TRUE);

        if (length < 0) {
                cis_result = length;
                return (TRUE);
        }

        free(*listp);
        *listp = NULL;

        if (!(new_list = calloc(length, sizeof (struct cis_procinfo))))
                return (FALSE);
        *listp = new_list;

        for (p = new_list; p < new_list + length; p++) {

                if (!xdr_process(handle, p))
                        return (FALSE);

                if (!xdr_short(handle, &p->pCPU))
                        return (FALSE);

                p->flags = CONSISTENT;
        }
        qsort(new_list, length, sizeof (struct cis_procinfo), proc_cmp);

        list_length = length;

        return (TRUE);
}

static bool_t xdr_sock_reply(XDR *handle, struct cis_sockinfo **listp)
{
        int length = 0;
        struct cis_sockinfo *new_list, *s;

        free(*listp);
        *listp = NULL;
        list_length = 0;
        cis_result  = HOST_AVAILABLE;

        if (!xdr_int(handle, &length))
                return (FALSE);

        if (!length)
                return (TRUE);

        if (length < 0) {
                cis_result = length;
                return (TRUE);
        }

        new_list = calloc(length, sizeof (struct cis_sockinfo));
        if (!new_list)
                return (FALSE);
        *listp = new_list;

        for (s = *listp; s < *listp + length; s++) {
                if (!xdr_socket_desc(handle, s))
                        return (FALSE);

                if (!xdr_u_int(handle, &s->pid))
                        return (FALSE);
                if (!xdr_u_short(handle, &s->uid))
                        return (FALSE);

                if (!xdr_u_long(handle, &s->sent))
                        return (FALSE);
                if (!xdr_u_long(handle, &s->rcvd))
                        return (FALSE);

                s->flags = TRUE;
        }

        qsort(*listp, length, sizeof (struct cis_sockinfo), sock_cmp);
        list_length = length;
        
        return (TRUE);
}

static bool_t xdr_netdev_reply(XDR *handle, struct cis_netdevinfo **listp)
{
        int length = 0;
        struct cis_netdevinfo *new_list, *dev;

        free(*listp);
        *listp = NULL;
        list_length = 0;
        cis_result  = HOST_AVAILABLE;

        if (!xdr_int(handle, &length))
                return (FALSE);

        if (!length)
                return (TRUE);

        if (length < 0) {
                cis_result = length;
                return (TRUE);
        }

        new_list = calloc(length, sizeof (struct cis_netdevinfo));
        if (!new_list)
                return (FALSE);
        *listp = new_list;

        for (dev = *listp; dev < *listp + length; dev++) {
                if (!xdr_netdev(handle, dev))
                        return (FALSE);

                dev->flags = TRUE;
        }

        qsort(*listp, length, sizeof (struct cis_netdevinfo), netdev_cmp);
        list_length = length;

        return (TRUE);
}

/*
 * Interface calls
 */

void cisTimeout(CLIENT *handle, struct timeval tout)
{
        clnt_control(handle, CLSET_TIMEOUT,(char *) &tout);

        cis_timeout = tout;
}

CLIENT *cisConnect(char *hostname)
{
        return (clnt_create(hostname, CIS_PROG, CIS_VER, "tcp"));
}

void cisDisconnect(CLIENT *handle)
{
        clnt_destroy(handle);
}

char **cisHostList(CLIENT *handle, unsigned char reliability)
{
	char **hostlist;
	int err;
	
	err = clnt_call(handle, HOST_LIST,
			(xdrproc_t) xdr_char, (caddr_t) &reliability,
			(xdrproc_t) xdr_hostlist, (caddr_t) &hostlist,
			cis_timeout);

	if (err != RPC_SUCCESS) {
		cis_errno = err;
		return NULL;
	}
	return hostlist;
}

struct cis_hostinfo *cisHostInfo(CLIENT *handle, char *name, char *perf_type)
{
        struct cis_hostinfo *hinfo;
        int err;
        struct hostent *hent;
	struct request req;

        if (!handle)
                return NULL;
        
        if ((hent = gethostbyname (name)) == NULL) {
                cis_errno = CIS_UNKNOWN;
                return NULL;
        }

        hinfo = calloc(1, sizeof (struct cis_hostinfo));

        if (!hinfo) {
                cis_errno = CIS_NOMEM;
                return NULL;
        }

	req.addr = *(struct in_addr *) hent->h_addr_list[0];
	req.perf_type = perf_type;
	
	err = clnt_call(handle, HOST_INFO,
			(xdrproc_t) xdr_host_req, (caddr_t) &req,
                        (xdrproc_t) xdr_host_rep, (caddr_t) hinfo,
                        cis_timeout);

	hinfo->addr = req.addr;
        
        if (err != RPC_SUCCESS) {
                cis_errno = err;
                free(hinfo);
                return NULL;
        }

        return hinfo;
}

int cisUpdateHostsInfo(CLIENT *handle, int code, struct cis_hostinfo **hosttab, int nhosts)
{
        int err;

        if (!handle)
                return -1;

        update_code = code;
        list_length = nhosts;
        
        err = clnt_call(handle, HOST_INFO_UPDATE,
                        (xdrproc_t) xdr_hosts_update_req, (caddr_t) hosttab,
                        (xdrproc_t) xdr_hosts_update_reply, (caddr_t) hosttab,
                        cis_timeout);

        if (err != RPC_SUCCESS) {
                cis_errno = err;
                return -1;
        }
        return 0;
}


int cisUpdateHostInfo (CLIENT *handle, int code, struct cis_hostinfo *hinfo)
{
        return cisUpdateHostsInfo (handle, code, &hinfo, 1);
}


int cisProcessList(CLIENT *handle, struct cis_procinfo **list, struct in_addr addr, unsigned short uid)
{
        struct request req = {addr, uid, 0};
        int err;
        *list = NULL;

        err = clnt_call(handle, PROCESS_LIST,
                        (xdrproc_t) xdr_request,   (caddr_t) &req,
                        (xdrproc_t) xdr_proc_reply, (caddr_t) list,
                        cis_timeout);
        if (err)
                return -err;

        if (cis_result)
                return cis_result;

        return list_length;
}

int cisSocketList(CLIENT *handle, struct cis_sockinfo **list, struct in_addr addr, unsigned short uid, float interval)
{
        struct request req = {addr, uid, interval};
        int err;
        *list = NULL;

        err = clnt_call(handle, SOCKET_LIST,
                        (xdrproc_t) xdr_request,   (caddr_t) &req,
                        (xdrproc_t) xdr_sock_reply, (caddr_t) list,
                        cis_timeout);
        if (err)
                return -err;

        if (cis_result)
                return cis_result;

        return list_length;
}

int cisNetdevList(CLIENT *handle, struct cis_netdevinfo **list, struct in_addr addr, float interval)
{
        struct request req = {addr, 0, interval};
        int err;
        *list = NULL;

        err = clnt_call(handle, NETDEV_LIST,
                        (xdrproc_t) xdr_request,   (caddr_t) &req,
                        (xdrproc_t) xdr_netdev_reply, (caddr_t) list,
                        cis_timeout);
        if (err)
                return -err;

        if (cis_result)
                return cis_result;

        return list_length;
}

/*
 * Trace file manipulation functions.
 */
// internal routines.

static int read_record_header(FILE *file, struct record_info *rinfo)
{
        struct record_info r;
        
        if ((r.offset = ftell(file)) < 0)
                return -1;
        
        if (!fread(&r.time,   sizeof (struct timeval), 1, file) ||
            !fread(&r.type,   sizeof (char),           1, file) ||
            !fread(&r.host,   sizeof (struct in_addr), 1, file) ||
            !fread(&r.length, sizeof (short),          1, file)) {
                fseek(file, r.offset, SEEK_SET);
                return -1;
        }

        rinfo->time   = r.time;
        rinfo->type   = r.type;
        rinfo->host   = r.host;
        rinfo->length = r.length;
        rinfo->offset = r.offset;

        return 0;
}

int struct_size[] = {
        0,
        sizeof (struct cis_hostinfo),
        0,
        sizeof (struct cis_procinfo),
        0,
        sizeof (struct cis_sockinfo),
        0,
        sizeof (struct cis_netdevinfo),
        0,
};

int save_size[] = {
        0,
        sizeof (struct cis_hostinfo),
        0,
        PROCINFO_LEN,
        0,
        SOCKINFO_LEN,
        0,
        NETDEVINFO_LEN,
        0,
};

static int read_record_data(FILE *file, struct record_info *rinfo)
{
        char *buf = NULL, *c = NULL;
        int type = rinfo->type;
        int block = rinfo->length;
        int num   = 1;
        int fragm = rinfo->length;
        
        if (!rinfo->length)
                goto end;

        if (struct_size[type]) {
                fragm = save_size[type];
                block = struct_size[type];
                num   = rinfo->length/fragm;
        }

        buf = calloc(num,block);
        if (!buf)
                return -1;

        for (c = buf; num; num--, c += block)
                if (fread(c, 1, fragm, file) < fragm) {
                        free(buf);
                        return -1;
                }

     end:

        rinfo->data   = buf;
        rinfo->length = c - buf;
        return 0;
}

static inline int objcmp(char *a, char *b, int len)
{
        for (a += len-1, b += len-1; len; len--, a--, b--)
                if (*a != *b)
                        return len;
        return 0;
}

static inline int changed(void *oldp, void *newp, struct description *desc, int desclen)
{
        struct description *d;

        if ((!oldp || !newp) && (oldp || newp))
                return TRUE;

        for (d = desc+1; d < desc+desclen; d++)
                if (objcmp((char *) oldp + d->offset,
                           (char *) newp + d->offset,
                           d->length))
                        return TRUE;
        return FALSE;
}


/*
 * The implementation is done for Linux endian. Should be rewriten if ported.
 */
static inline int add_changes(char **buffp, int *bufflen, void *oldp, void *newp, struct description *desc, int desclen)
{
        int len = *bufflen;
        char *buff = *buffp;
        char *tmp;
        int flags, bits;
        int flagsp, nflags, nbits;
        int fragment;

        struct description *d;

        if (!oldp) { /* New object, save whole object */
                if (!(tmp = realloc(buff, len + desc->fragment + 1)))
                        goto error;
                buff = tmp;
                *(buff+len) = 1; /* code */
                memcpy(buff+len+1, (char *) newp, desc->fragment);
                len += desc->fragment + 1;
                goto end;
        }
                
        if (!newp) { /* Destroyed object, save its identifier */
                if (!(tmp = realloc(buff, len + desc->length + 1)))
                        goto error;
                buff = tmp;
                *(buff+len) = 2; /* code */
                memcpy(buff+len+1, (char *) oldp + desc->offset, desc->length);
                len += desc->length + 1;
                goto end;
        }

        flagsp = len;
        flags  = 0;
        nflags = 2;
        /* Store the identifier */
        if (!(tmp = realloc(buff, len + desc->length + 1)))
                goto error;
        buff = tmp;
        memcpy(buff+len+1, (char *) oldp + desc->offset, desc->length);
        len += desc->length+1;

        for (d = desc+1; d < desc+desclen; d++) {

                fragment = objcmp((char *) oldp + d->offset,
                                  (char *) newp + d->offset,
                                  d->length);
                nbits = 1;
                if (!fragment)
                        bits = 0;
                else if (fragment <= d->fragment)
                        bits = 1, nbits = 2, fragment = d->fragment;
                else if (fragment <= 2*d->fragment && 2*d->fragment < d->length)
                        bits = 3, nbits = 3, fragment = 2*d->fragment;
                else
                        bits = 7, nbits = 3, fragment = d->length;
                        
                if (nflags + nbits > 8) {
                        if (!(tmp = realloc(buff, len+1)))
                                goto error;
                        buff = tmp;

                        flags >>= 8 - nflags;        /* Allocate space for bits */
                        flags |= bits << nflags;     /* Add bits to the flags */
                        buff[flagsp] = flags;        /* Store lower byte of flags */
                        nflags += nbits - 8;         /* Compute length of remainder */
                        flags >>= nflags;
                        flagsp = len++;              /* Get the pointer to the next flags field */
                } else {
                        flags >>= nbits;             /* Allocate space for bits */
                        flags |= bits << (8 - nbits);/* Add bits */
                        nflags += nbits;
                }

                if (!bits) /* no change */
                        continue;

                if (!(tmp = realloc(buff, len+fragment)))
                        goto error;
                buff = tmp;
                memcpy(buff + len, (char *) newp + d->offset, fragment);
                len += fragment;
        }

        if (nflags)
                buff[flagsp] = flags >> (8-nflags);

    end:
        *buffp   = buff;
        *bufflen = len;

        return TRUE;
    error:
        return FALSE;
}

void apply_changes(void *str, char **changes, struct description *desc, int desclen)
{
        char *c = *changes + 1;
        unsigned short flags = ((short)1 << 15) | **changes;
        struct description *d;

        if (flags & 1) {
                memcpy((char *) str, c, desc->fragment);
                c += desc->fragment;
                goto end;
        }
        if (flags & 2) {
                memcpy((char *) str + desc->offset, c, desc->length);
                c += desc->length;
                goto end;
        }

        c += desc->length; /* Skip the identifier, str should point to */
                           /* correct object */
        flags >>= 2;       /* Skip new and terminated flag */

        for (d = desc+1; d < desc+desclen; d++, flags >>= 1) {

                if (!(flags >> 8))
                        flags = (1 << 15) | *(c++);

                if (!(flags & 1)) /* no change */
                        continue;

                flags >>= 1;
                if (!(flags >> 8))
                        flags = (1 << 15) | *(c++);

                if (!(flags & 1)) {
                        memcpy((char *) str + d->offset, c, d->fragment);
                        c += d->fragment;
                        continue;
                }

                flags >>= 1;
                if (!(flags >> 8))
                        flags = (1 << 15) | *(c++);
                        
                if (!(flags & 1)) {
                        memcpy((char *) str + d->offset, c, 2*d->fragment);
                        c += 2*d->fragment;
                        continue;
                }

                memcpy((char *) str + d->offset, c, d->length);
                c += d->length;
        }

    end:
        *changes = c;
}

/*
 * Library routines.
 */
int cisSaveHeader(FILE *file)
{
        struct cis_header header = {CIS_RECORD_VERSION};
        
        ftruncate(fileno(file),0);
        
        return (fwrite(&header, sizeof (struct cis_header), 1, file));
}

int cisReadHeader(FILE *file, struct cis_header *header)
{
        if (fseek(file, 0L, SEEK_SET) < 0)
                return -1;

        return (fread(header, sizeof (struct cis_header), 1, file));
}

int cisHostinfoPrepareChanges(struct cis_hostinfo *oldhi, struct cis_hostinfo *newhi, char **changes)
{
        char *buff = NULL;
        int len = 0;
        int desclen = sizeof (cis_hostinfo_desc)/sizeof (struct description);

        if (!oldhi && !newhi) {
                *changes = NULL;
                return 0;
        }

        if (changed(oldhi, newhi, cis_hostinfo_desc, cis_hostinfo_desclen))
                if (!add_changes(&buff, &len, oldhi, newhi, cis_hostinfo_desc, desclen))
                        goto error;

        *changes = buff;
        
        return len;

    error:
        free(buff);
        return -1;
}

int cisHostinfoApplyChanges(struct cis_hostinfo *hi, char *changes, int chlen, struct cis_hostinfo **newhi)
{
        *newhi = calloc(1, sizeof (struct cis_hostinfo));

        if (!*newhi)
                return -1;

        **newhi = *hi;
        
        if (!changes)
                return 0;

        if (*changes & 2) {
                (*newhi)->status = HOST_NOT_AVAILABLE;
                return 1;
        }

        apply_changes(*newhi, &changes, cis_hostinfo_desc, cis_hostinfo_desclen);

        return 1;
}

int cisProcPrepareChanges(struct cis_procinfo *oldtab, int nold, struct cis_procinfo *newtab, int nnew, char **changes)
{
        struct cis_procinfo *p, *q;
        char *buff = NULL;
        int len = 0;

        qsort(oldtab, nold, sizeof (struct cis_procinfo), proc_cmp);
        qsort(newtab, nnew, sizeof (struct cis_procinfo), proc_cmp);

        p = oldtab;
        q = newtab;

        while (p < oldtab+nold || q < newtab+nnew) {

                if (q < newtab+nnew && (p == oldtab+nold || p->pid > q->pid)) {
                        if (!add_changes(&buff, &len, NULL, q,      /* new process */
                                         cis_procinfo_desc, cis_procinfo_desclen))
                                goto error;
                        q++;
                        continue;
                }
                if (p < oldtab+nold && (q == newtab+nnew || p->pid < q->pid)) {
                        if (!add_changes(&buff, &len, p, NULL,    /* terminated process */
                                         cis_procinfo_desc, cis_procinfo_desclen))
                                goto error;
                        p++;
                        continue;
                }

                if (changed(p, q, cis_procinfo_desc, cis_procinfo_desclen))
                        add_changes(&buff, &len, p, q,
                                    cis_procinfo_desc, cis_procinfo_desclen);
                
                p++;
                q++;
        }
        *changes = buff;
        
        return len;

    error:
        free(buff);
        return -1;
}

int cisProcApplyChanges(struct cis_procinfo *oldtab, int nold, char *changes, int chlen, struct cis_procinfo **newtab)
{
        struct cis_procinfo *p, *q, *tab = NULL;
        char *c;
        int len = 0;
        int pid;
        int flags;

        *newtab = NULL;

        qsort(oldtab, nold, sizeof (struct cis_procinfo), proc_cmp);

        p = oldtab;
        q = NULL;

        c = changes;
        while (c < changes + chlen) {
                flags = *c;
                if (flags & 1)
                        memcpy(&pid, c+1+cis_procinfo_desc->offset, sizeof (int));
                else
                        memcpy(&pid, c+1, sizeof (int));
                
                /*
                 * Copy all processes up to new/updated one. If the process is terminated
                 * skip it.
                 */
                while (p < oldtab + nold && p->pid < pid) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_procinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        q = tab+len;
                        *q = *p;
                        len++, p++;
                }
                if (!(flags & 1) && p == oldtab+nold)
                        goto error;

                if (flags & 2) {
                        if (p->pid != pid)
                                goto error;
                        c += sizeof(int)+1;
                        p++;
                        continue;
                }

                /*
                 * If the pid was changed, copy next process.
                 */
                if (!q || q->pid != pid) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_procinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        q = tab+len;
                        len++;
                        if (flags & 1) {
                                memset(q, 0, sizeof (struct cis_procinfo));
                                q->pid = pid;
                        }
                        else if (p < oldtab+nold)
                                memcpy(q, p++, sizeof (struct cis_procinfo));
                        else
                                goto error;
                }

                if (q->pid != pid)
                        goto error;

                apply_changes(q, &c, cis_procinfo_desc, cis_procinfo_desclen);
        }
        /*
         * Copy the rest of oldtab.
         */
        for (;p < oldtab + nold; p++) {
                tab = realloc(*newtab, (len+1)*sizeof (struct cis_procinfo));
                if (!tab)
                        goto error;
                *newtab = tab;
                q = tab+len;
                *q = *p;
                len++;
        }

        return len;

    error:
        free(tab);
        *newtab = NULL;
        return -1;
}

int cisSockPrepareChanges(struct cis_sockinfo *oldtab, int nold, struct cis_sockinfo *newtab, int nnew, char **changes)
{
        struct cis_sockinfo *s, *t;
        char *buff = NULL;
        int len = 0;

        qsort(oldtab, nold, sizeof (struct cis_sockinfo), sock_cmp);
        qsort(newtab, nnew, sizeof (struct cis_sockinfo), sock_cmp);

        s = oldtab;
        t = newtab;

        while (s < oldtab+nold || t < newtab+nnew) {

                if (t < newtab+nnew &&
                    (s == oldtab+nold || sock_cmp(s, t) > 0)) {
                        if (!add_changes(&buff, &len, NULL, t,      /* new socket */
                                         cis_sockinfo_desc, cis_sockinfo_desclen))
                                goto error;
                        t++;
                        continue;
                }
                if (s < oldtab+nold && (t == newtab+nnew || sock_cmp(s, t) < 0)) {
                        if (!add_changes(&buff, &len, s, NULL,    /* closed socket */
                                         cis_sockinfo_desc, cis_sockinfo_desclen))
                                goto error;
                        s++;
                        continue;
                }

                if (changed(s, t, cis_sockinfo_desc, cis_sockinfo_desclen))
                        add_changes(&buff, &len, s, t,
                                    cis_sockinfo_desc, cis_sockinfo_desclen);
                
                s++;
                t++;
        }
        *changes = buff;
        
        return len;

    error:
        free(buff);
        return -1;
}

int cisSockApplyChanges(struct cis_sockinfo *oldtab, int nold, char *changes, int chlen, struct cis_sockinfo **newtab)
{
        struct cis_sockinfo *s, *t, *tab = NULL, tmp;
        char *c;
        int len = 0;
        int flags;

        *newtab = NULL;

        qsort(oldtab, nold, sizeof (struct cis_sockinfo), sock_cmp);

        s = oldtab;
        t = NULL;

        c = changes;
        while (c < changes + chlen) {
                flags = *c;
                if (flags & 1)
                        memcpy(&tmp.saddr, c+1+cis_sockinfo_desc->offset, cis_sockinfo_desc->length);
                else
                        memcpy(&tmp.saddr, c+1, cis_sockinfo_desc->length);

                /*
                 * Copy all sockets up to new/updated one. If the socket is closed
                 * skip it.
                 */
                while (s < oldtab + nold && sock_cmp(s, &tmp) < 0) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_sockinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        t = tab+len;
                        *t = *s;
                        len++, s++;
                }
                if (!(flags & 1) && s == oldtab+nold)
                        goto error;

                if (flags & 2) {
                        if (sock_cmp(s, &tmp) != 0)
                                goto error;
                        c += cis_sockinfo_desc->length+1;
                        s++;
                        continue;
                }

                if (!t || sock_cmp(t, &tmp) != 0) {
                        /* Copy next socket.  */
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_sockinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        t = tab+len;
                        len++;
                        if (flags & 1) {
                                memset(t, 0, sizeof (struct cis_sockinfo));
                                memcpy(&t->saddr, &tmp.saddr, cis_sockinfo_desc->length);
                        }
                        else if (s < oldtab+nold)
                                memcpy(t, s++, sizeof (struct cis_sockinfo));
                        else
                                goto error;
                }

                if (sock_cmp(t, &tmp) != 0)
                        goto error;

                apply_changes(t, &c, cis_sockinfo_desc, cis_sockinfo_desclen);
        }
        /*
         * Copy the rest of oldtab.
         */
        for (;s < oldtab + nold; s++) {
                tab = realloc(*newtab, (len+1)*sizeof (struct cis_sockinfo));
                if (!tab)
                        goto error;
                *newtab = tab;
                t = tab+len;
                *t = *s;
                len++;
        }

        return len;

    error:
        free(tab);
        *newtab = NULL;
        return -1;
}

int cisNetdevPrepareChanges(struct cis_netdevinfo *oldtab, int nold, struct cis_netdevinfo *newtab, int nnew, char **changes)
{
        struct cis_netdevinfo *nd, *od;
        char *buff = NULL;
        int len = 0;

        qsort(oldtab, nold, sizeof (struct cis_netdevinfo), netdev_cmp);
        qsort(newtab, nnew, sizeof (struct cis_netdevinfo), netdev_cmp);

        od = oldtab;
        nd = newtab;

        while (od < oldtab+nold || nd < newtab+nnew) {

                if (nd < newtab+nnew && (od == oldtab+nold || netdev_cmp(od->name, nd->name) > 0)) {
                        if (!add_changes(&buff, &len, NULL, nd,      /* new device */
                                         cis_netdevinfo_desc, cis_netdevinfo_desclen))
                                goto error;
                        nd++;
                        continue;
                }
                if (od < oldtab+nold && (nd == newtab+nnew || netdev_cmp(od->name, nd->name) < 0)) {
                        if (!add_changes(&buff, &len, od, NULL,      /* removed device */
                                         cis_netdevinfo_desc, cis_netdevinfo_desclen))
                                goto error;
                        od++;
                        continue;
                }

                if (changed(od, nd, cis_netdevinfo_desc, cis_netdevinfo_desclen))
                        add_changes(&buff, &len, od, nd,
                                    cis_netdevinfo_desc, cis_netdevinfo_desclen);

                od++;
                nd++;
        }
        *changes = buff;

        return len;

    error:
        free(buff);
        return -1;
}

int cisNetdevApplyChanges(struct cis_netdevinfo *oldtab, int nold, char *changes, int chlen, struct cis_netdevinfo **newtab)
{
        struct cis_netdevinfo *od, *nd, *tab = NULL;
        char *c, name[IFNAMSIZ];
        int len = 0;
        int flags;

        *newtab = NULL;

        qsort(oldtab, nold, sizeof (struct cis_netdevinfo), netdev_cmp);

        od = oldtab;
        nd = NULL;

        c = changes;
        while (c < changes + chlen) {
                flags = *c;
                if (flags & 1)
                        memcpy(name, c+1+cis_netdevinfo_desc->offset, IFNAMSIZ);
                else
                        memcpy(name, c+1, IFNAMSIZ);

                /*
                 * Copy all devices up to new/updated one. If the device is removed
                 * skip it.
                 */
                while (od < oldtab + nold && netdev_cmp(od->name, name) <0) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_netdevinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        nd = tab+len;
                        *nd = *od;
                        len++, od++;
                }
                if (!(flags & 1) && od == oldtab+nold)
                        goto error;

                if (flags & 2) {
                        if (netdev_cmp(od->name, name))
                                goto error;
                        c += IFNAMSIZ+1;
                        od++;
                        continue;
                }

                /*
                 * If the name was changed, copy next device.
                 */
                if (!nd || netdev_cmp(nd->name, name)) {
                        tab = realloc(*newtab, (len+1)*sizeof (struct cis_netdevinfo));
                        if (!tab)
                                goto error;
                        *newtab = tab;
                        nd = tab+len;
                        len++;
                        if (flags & 1) {
                                memset(nd, 0, sizeof (struct cis_netdevinfo));
                                strncpy(nd->name, name, IFNAMSIZ);
                        }
                        else if (od < oldtab+nold)
                                memcpy(nd, od++, sizeof (struct cis_netdevinfo));
                        else
                                goto error;
                }

                if (netdev_cmp(nd->name, name))
                        goto error;

                apply_changes(nd, &c, cis_netdevinfo_desc, cis_netdevinfo_desclen);
        }
        /*
         * Copy the rest of oldtab.
         */
        for (;od < oldtab + nold; od++) {
                tab = realloc(*newtab, (len+1)*sizeof (struct cis_netdevinfo));
                if (!tab)
                        goto error;
                *newtab = tab;
                nd = tab+len;
                *nd = *od;
                len++;
        }

        return len;

    error:
        free(tab);
        *newtab = NULL;
        return -1;
}

int cisSaveRecord(FILE *file, struct timeval time, int type, struct in_addr host, char *data, int block, int num, int save)
{
        short length = num*save;
        long offset;

        if ((offset = ftell(file)) < 0)
                return -1;

        if (!data)
                num = 0;

        if (!fwrite(&time,   sizeof (struct timeval), 1, file) ||
            !fwrite(&type,   sizeof (char),           1, file) ||
            !fwrite(&host,   sizeof (struct in_addr), 1, file) ||
            !fwrite(&length, sizeof (short),          1, file))
                goto error;

        for (; num; num--, data += block)
                if (fwrite(data, 1, save, file) < save)
                        goto error;

        return 0;

    error:
        ftruncate(fileno(file), offset);
        return -1;
}

int cisRecordInfo(FILE *file, struct record_info *rinfo)
{
        int ret;

        ret = read_record_header(file, rinfo);
        if (ret < 0)
                return ret;

        fseek(file, rinfo->offset, SEEK_SET);
        
        return ret;
}
        
int cisReadRecord(FILE *file, struct record_info *rinfo)
{
        int ret;

        ret = read_record_header(file, rinfo);
        if (ret < 0)
                return ret;

        return (read_record_data(file, rinfo));
}

int cisParseFile(FILE *file, struct record_info **rec, void callback(int))
{
        struct record_info *array = NULL, *new_array, tmp;
        long offset, total_length;
        int n = 0, perc = 0, oldperc = 0;
	int ret;
        struct stat fst;

        if (fstat (fileno(file), &fst) == -1)
                return -1;

        total_length = fst.st_size;

        while (!feof(file)) {

                offset = ftell(file);

                ret = cisRecordInfo(file, &tmp);
                if (ret < 0)
                        break;

                if (fseek(file, tmp.length + CIS_RECHDRLEN, SEEK_CUR) == -1)
                        break;

                new_array = realloc(array, (n+1) * sizeof (struct record_info));
                if (!new_array) {
                        errno = ENOMEM;
                        goto error;
                }

                array = new_array;
                memcpy(&array[n], &tmp, sizeof (struct record_info));
                n++;

		if (callback) {
			perc = 100 - ((total_length - offset)*100)/total_length;
			if (perc != oldperc)
				callback(oldperc = perc);
		}
        } 

        *rec = array;
        return n;

    error:
        free(array);
        return -1;
}

int cisSaveHostinfo(FILE *file, struct in_addr host, struct cis_hostinfo *data)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_HOSTINFO,
                             host,
                             (char *) data,
                             sizeof (struct cis_hostinfo),
                             1,
                             sizeof (struct cis_hostinfo));
}

int cisSaveHostinfoChanges(FILE *file, struct in_addr host, char *changes, int length)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_HOSTINFO_CHANGES,
                             host,
                             changes,
                             length,
                             1,
                             length);
}

int cisSaveProcinfo(FILE *file, struct in_addr host, struct cis_procinfo *data, int number)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_PROCINFO,
                             host,
                             (char *) data,
                             sizeof (struct cis_procinfo),
                             number,
                             cis_procinfo_desc[0].fragment);
}

int cisSaveProcChanges(FILE *file, struct in_addr host, char *changes, int length)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_PROC_CHANGES,
                             host,
                             changes,
                             length,
                             1,
                             length);
}

int cisSaveSockinfo(FILE *file, struct in_addr host, struct cis_sockinfo *data, int number)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_SOCKINFO,
                             host,
                             (char *) data,
                             sizeof (struct cis_sockinfo),
                             number,
                             cis_sockinfo_desc[0].fragment);
}

int cisSaveSockChanges(FILE *file, struct in_addr host, char *changes, int length)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_SOCK_CHANGES,
                             host,
                             changes,
                             length,
                             1,
                             length);
}

int cisSaveNetdevinfo(FILE *file, struct in_addr host, struct cis_netdevinfo *data, int number)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_NETDEVINFO,
                             host,
                             (char *) data,
                             sizeof (struct cis_netdevinfo),
                             number,
                             cis_netdevinfo_desc[0].fragment);
}

int cisSaveNetdevChanges(FILE *file, struct in_addr host, char *changes, int length)
{
        struct timeval time;
        gettimeofday(&time, NULL);

        return cisSaveRecord(file,
                             time,
                             CIS_NETDEV_CHANGES,
                             host,
                             changes,
                             length,
                             1,
                             length);
}

int cisApplyChanges(int type, void *oldtab, int nold, char *changes, int chlen, void **newtab)
{
        switch (type) {

        case CIS_HOSTINFO_CHANGES:
                return (cisHostinfoApplyChanges((struct cis_hostinfo *) oldtab,
                                                changes, chlen,
                                                (struct cis_hostinfo **) newtab));

        case CIS_PROC_CHANGES:
                return (cisProcApplyChanges((struct cis_procinfo *) oldtab, nold,
                                            changes, chlen,
                                            (struct cis_procinfo **) newtab));

        case CIS_SOCK_CHANGES:
                return (cisSockApplyChanges((struct cis_sockinfo *) oldtab, nold,
                                            changes, chlen,
                                            (struct cis_sockinfo **) newtab));

        case CIS_NETDEV_CHANGES:
                return (cisNetdevApplyChanges((struct cis_netdevinfo *) oldtab, nold,
                                              changes, chlen,
                                              (struct cis_netdevinfo **) newtab));
        default:
                return -1;
        }
}