/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Kernel module for monitoring of INET sockets with open/close notification
 * via netlink device.
 */

#include <linux/config.h>
#include <linux/module.h>

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <asm/uaccess.h>

#include <net/inet_common.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <linux/proc_fs.h>

#include "sockmon.h"

extern struct net_proto_family inet_family_ops;

struct proto_ops old_inet_stream_ops;
struct proto_ops old_inet_dgram_ops;
struct net_proto_family old_inet_family_ops;

static struct sock *smsk;

struct sm_type {
        short type;
        struct proto *prot;
};

static struct sm_type sm_types[] = {
        { SOCK_STREAM, &tcp_prot },
        { SOCK_DGRAM,  &udp_prot },
};

int read_socklist(char *buffer, char **start, off_t offset, int length, int dummy)
{
        struct sock *sk, *next;
        int len = 0;
        off_t pos = 0;
        off_t begin;
        struct sm_type *st;
        struct cis_sockinfo msg;

        SOCKHASH_LOCK();

        for (st = sm_types;
             st < sm_types + (sizeof (sm_types)/ sizeof (sm_types[0]));
             st++) {

                sk = st->prot->sklist_next;
                while (sk != (struct sock *) st->prot) {

                        if (st->type == SOCK_STREAM && sk->state != TCP_ESTABLISHED)
                                goto next;
                        
                        pos += SOCKINFO_LEN;
                        if (pos < offset)
                                goto next;

                        msg.type         = st->type;

                        msg.saddr.s_addr = sk->rcv_saddr;
                        msg.sport        = ntohs(sk->sport);
                        msg.daddr.s_addr = sk->daddr;
                        msg.dport        = ntohs(sk->dport);
                        msg.pid          = sk->owner_pid;
                        msg.uid          = sk->owner_uid;
                        msg.sent         = sk->bytes_sent;
                        msg.rcvd         = sk->bytes_received;

                        memcpy (buffer + len, &msg, SOCKINFO_LEN);
                        len += SOCKINFO_LEN;
                        if(len >= length)
                                goto out;
                    next:
                        next = sk->sklist_next;
                        sk = next;
                }
        }

    out:
        SOCKHASH_UNLOCK();

        begin = len - (pos - offset);
        *start = buffer + begin;
        len -= begin;
        if(len > length)
                len = length;
        if (len < 0)
                len = 0;
        return len;
}

static struct proc_dir_entry proc_root_socklist = {
        0,                               /* Inode number      */
        8, "socklist",                   /* The name of file with length */
        S_IFREG | S_IRUGO,               /* Acess permissions */
        1, 0, 0,                         /* Number of links, owner, group */
        0,                               /* The size of the file reported by ls. */
        NULL,                            /* Operations - use default */
        read_socklist                    /* The read function */
        /* nothing more */
};

static int sockmon_msg(struct socket *sock, unsigned char action)
{
        struct sock *sk = sock->sk;
        struct cis_sockinfo *msg;
        size_t len = SOCKINFO_LEN+1;

        struct sk_buff *outskb = alloc_skb(len, GFP_ATOMIC);

        if (outskb) {
                skb_put(outskb, len);
                msg = (struct cis_sockinfo *) outskb->data;

                msg->flags        = action;
                msg->type         = sock->type;
                msg->saddr.s_addr = sk->rcv_saddr;
                msg->sport        = ntohs(sk->sport);
                msg->daddr.s_addr = sk->daddr;
                msg->dport        = ntohs(sk->dport);
                msg->pid          = sk->owner_pid;
                msg->uid          = sk->owner_uid;
                msg->sent         = sk->bytes_sent;
                msg->rcvd         = sk->bytes_received;

                netlink_broadcast(smsk, outskb, 0, ~0, GFP_KERNEL);
        }
        else
                return (-1);

	return (0);
}

/*
 * New inet create.
 */
int new_inet_create(struct socket *sock, int protocol)
{
        int err;
        struct sock *sk;
        
        err = old_inet_family_ops.create (sock, protocol);

        if (err == 0) {
                sk = sock->sk;

                sk->owner_pid      = current->pid;
                sk->owner_uid      = current->uid;
                sk->bytes_sent     = 0;
                sk->bytes_received = 0;
        }
        
        return err;
}

/*
 * Stream operations wrappers
 */

int new_stream_release(struct socket *sock, struct socket *peersock)
{
        sockmon_msg (sock, SM_CLOSE);
        return (old_inet_stream_ops.release) (sock, peersock);
}

int new_stream_bind(struct socket *sock, struct sockaddr *uaddr, int addr_len)
{
        struct sock *sk = sock->sk;
        int err;

        err = old_inet_stream_ops.bind (sock, uaddr, addr_len);

        if (sk) {
                sk->owner_pid      = current->pid;
                sk->owner_uid      = current->uid;
                sk->bytes_sent     = 0;
                sk->bytes_received = 0;
        }
        return err;
}

int new_stream_connect(struct socket *sock, struct sockaddr * uaddr,
			int addr_len, int flags)
{
        int err = old_inet_stream_ops.connect (sock, uaddr, addr_len, flags);

        if (err == 0)
                sockmon_msg (sock, SM_CONNECT);

        return err;
}

int new_stream_accept(struct socket *sock, struct socket *newsock, int flags)
{
        int err = old_inet_stream_ops.accept (sock, newsock, flags);
        struct sock *sk;
        
        if (err == 0) {
                sk = newsock->sk;

                sk->owner_pid      = current->pid;
                sk->owner_uid      = current->uid;
                sk->bytes_sent     = 0;
                sk->bytes_received = 0;
                
                sockmon_msg (newsock, SM_CONNECT);
        }

        return err;
}

int new_stream_sendmsg(struct socket *sock, struct msghdr *msg, int size,
                       struct scm_cookie *scm)
{
        struct sock *sk = sock->sk;
        int err;

        err = old_inet_stream_ops.sendmsg (sock, msg, size, scm);

        if (sk && err > 0) {
                if (sk->owner_pid != current->pid) {
                        sk->owner_pid = current->pid;
                        sk->owner_uid = current->uid;
                }
                sk->bytes_sent += err;
        }
        return err;
}

int new_stream_recvmsg(struct socket *sock, struct msghdr *msg, int size,
                       int flags, struct scm_cookie *scm)
{
        struct sock *sk = sock->sk;
        int err;

        err = old_inet_stream_ops.recvmsg (sock, msg, size, flags, scm);

        if (sk && err > 0) {
                if (sk->owner_pid != current->pid) {
                        sk->owner_pid = current->pid;
                        sk->owner_uid = current->uid;
                }
                sk->bytes_received += err;
        }
        return err;
}
/*
 * Datagram operations wrappers.
 * Yes. Some functions are duplicated. But what if the original handler will
 * not be the same in the future...
 */

int new_dgram_release(struct socket *sock, struct socket *peersock)
{
        sockmon_msg (sock, SM_CLOSE);
        return (old_inet_dgram_ops.release) (sock, peersock);
}

int new_dgram_bind(struct socket *sock, struct sockaddr *uaddr, int addr_len)
{
        struct sock *sk = sock->sk;
        int err;

        err = old_inet_dgram_ops.bind (sock, uaddr, addr_len);

        if (sk) {
                sk->owner_pid      = current->pid;
                sk->owner_uid      = current->uid;
                sk->bytes_sent     = 0;
                sk->bytes_received = 0;
        }
        return err;
}

int new_dgram_connect(struct socket *sock, struct sockaddr * uaddr,
			int addr_len, int flags)
{
        struct sock *sk = sock->sk;
        int err = old_inet_dgram_ops.connect (sock, uaddr, addr_len, flags);

        if (sk && err == 0) {
                sk->owner_pid      = current->pid;
                sk->owner_uid      = current->uid;
                sk->bytes_sent     = 0;
                sk->bytes_received = 0;
                
                sockmon_msg (sock, SM_CONNECT);
        }

        return err;
}

int new_dgram_sendmsg(struct socket *sock, struct msghdr *msg, int size,
                       struct scm_cookie *scm)
{
        struct sock *sk = sock->sk;
        int err;

        err = old_inet_dgram_ops.sendmsg (sock, msg, size, scm);
        
        if (sk && err > 0) {
                if (sk->owner_pid != current->pid) {
                        sk->owner_pid = current->pid;
                        sk->owner_uid = current->uid;
                }
                sk->bytes_sent += err;
        }
        return err;
}

int new_dgram_recvmsg(struct socket *sock, struct msghdr *msg, int size,
                       int flags, struct scm_cookie *scm)
{
        struct sock *sk = sock->sk;
        int err;

        err = old_inet_dgram_ops.recvmsg (sock, msg, size, flags, scm);

        if (sk && err > 0) {
                if (sk->owner_pid != current->pid) {
                        sk->owner_pid = current->pid;
                        sk->owner_uid = current->uid;
                }
                sk->bytes_received += err;
        }
        return err;
}

int sockmon_init(void)
{
        int err;

        smsk = netlink_kernel_create(NETLINK_SOCKMON, NULL);
        if (smsk == NULL) {
                printk("sockmon_init: cannot initialize netlink\n");
                return -ENODEV;
        }
        
        err = proc_register(&proc_root, &proc_root_socklist);
        if (err) {
                printk("sockmon_init: cannot register socklist\n");
                return err;
        }
        
        old_inet_stream_ops = inet_stream_ops;
        old_inet_dgram_ops  = inet_dgram_ops;
        old_inet_family_ops = inet_family_ops;
	
        inet_stream_ops.release = new_stream_release;
        inet_stream_ops.bind    = new_stream_bind;
        inet_stream_ops.connect = new_stream_connect;
        inet_stream_ops.accept  = new_stream_accept;
        inet_stream_ops.sendmsg = new_stream_sendmsg;
        inet_stream_ops.recvmsg = new_stream_recvmsg;

        inet_dgram_ops.release = new_dgram_release;
        inet_dgram_ops.bind    = new_dgram_bind;
        inet_dgram_ops.connect = new_dgram_connect;
        inet_dgram_ops.sendmsg = new_dgram_sendmsg;
        inet_dgram_ops.recvmsg = new_dgram_recvmsg;

        inet_family_ops.create = new_inet_create;

//        printk (KERN_INFO "Socket monitoring enabled.\n");

        return 0;
}

int init_module(void)
{
        return (sockmon_init ());
}

void cleanup_module(void)
{
        inet_stream_ops = old_inet_stream_ops;
        inet_dgram_ops  = old_inet_dgram_ops;
        inet_family_ops = old_inet_family_ops;

        proc_unregister(&proc_root, proc_root_socklist.low_ino);
        sock_release (smsk->socket);

//	printk (KERN_INFO "Socket monitoring disabled.\n");
}
