/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 */

#ifndef _cis_clnt_h
#define _cis_clnt_h

#include "cis.h"

struct description {
        short offset;
        short length;
        short fragment;
};

extern struct description host_sysinfo_desc[];
extern struct description cis_procinfo_desc[];
extern struct description cis_sockinfo_desc[];

enum {
        CIS_HOSTINFO = 1,
        CIS_HOSTINFO_CHANGES,
        CIS_PROCINFO,
        CIS_PROC_CHANGES,
        CIS_SOCKINFO,
        CIS_SOCK_CHANGES,
        CIS_NETDEVINFO,
        CIS_NETDEV_CHANGES,
};


enum {
        CIS_NOMEM = 200,
        CIS_UNKNOWN,
};

extern int cis_errno;

#define CIS_RECHDRLEN (sizeof (struct timeval)+ sizeof (char)+\
                       sizeof (struct in_addr) + sizeof (short))

extern char *rel_types[];
int parse_cmd (char *cmds[], int n, char **c);

/* Client -> CIS server RPC calls */

/* Function connects to specific CIS server and returns RPC CLIENT handle */
CLIENT *cisConnect (char *hostname);

/* Close RPC handle */
void cisDisconnect (CLIENT *handle);

/* Set RPC timeout */
void cisTimeout    (CLIENT *handle,
                    struct timeval tout);

/*
 * Get the list of hosts monitored by CIS server. Return code is number of
 * hosts.
 */
char **cisHostList(CLIENT *handle, unsigned char reliability);

/*
 * Get the info about host. Allocated structure cis_hostinfo is returned
 * If the host is not monitored or an error occured, NULL is returned and
 * cis_errno contains RPC error code.
 */
struct cis_hostinfo *cisHostInfo(CLIENT *handle, char *name, char *perf_type);

/*
 * Update host information. Code identifies fields that should be updated.
 * Status of host is updated automatically. If code is equal to zero, only
 * status is updated. If it's -1, all fields will be updated. hosttab must
 * contain array of pointers to structures returned by cisHostInfo() call.
 * Return code is zero if call was successful or -1 if some error occured.
 * In that case cis_errno contains RPC error code.
 */
int cisUpdateHostsInfo(CLIENT *handle,
                       int code,
                       struct cis_hostinfo **hosttab,
                       int nhosts);

/*
 * Update info for only one host.
 */
int cisUpdateHostInfo (CLIENT *handle,
                       int code,
                       struct cis_hostinfo *hinfo);

/*
 * Get a list of processes. If uid is specified, only processes of given
 * user are returned. Array is allocated dynamicaly and return code contains
 * its length. If some error has occured, return code is less than zero and
 * contains RPC error code (rpc/clnt.h). The cis_procinfo structure is defined
 * in cis.h.
 */
int cisProcessList (CLIENT *handle,
                    struct cis_procinfo **list,
                    struct in_addr addr,
                    unsigned short uid);

/*
 * Get a list of sockets. It's similar to cisProcessList() function.
 * Send and receive rates (in bytes per second) are counted for specified
 * interval (in seconds). If it is not specified or it is longer than CIS
 * server's archive, the length of archive is used instead.
 */
int cisSocketList  (CLIENT *handle,
                    struct cis_sockinfo **list,
                    struct in_addr addr,
                    unsigned short uid,
                    float interval);

/*
 * Get a list of network devices. It's similar to cisSocketList() function.
 */
int cisNetdevList  (CLIENT *handle,
                    struct cis_netdevinfo **list,
                    struct in_addr addr,
                    float interval);

/* end of client calls */

/* CIS record files processing */

#define CIS_RECORD_VERSION  2

struct cis_header {
        int version;
};

struct record_info {
        struct timeval time;
        struct in_addr host;
        char type;
        short length;
        char *data;
        long offset;
        void *host_ptr;
};

int cisHostinfoPrepareChanges(struct cis_hostinfo *oldhi, struct cis_hostinfo *newhi, char **changes);
int cisHostinfoApplyChanges(struct cis_hostinfo *oldhi, char *changes, int chlen, struct cis_hostinfo **newhi);

int cisProcPrepareChanges(struct cis_procinfo *oldtab, int nold, struct cis_procinfo *newtab, int nnew, char **changes);
int cisProcApplyChanges(struct cis_procinfo *oldtab, int nold, char *changes, int chlen, struct cis_procinfo **newtab);

int cisSockPrepareChanges(struct cis_sockinfo *oldtab, int nold, struct cis_sockinfo *newtab, int nnew, char **changes);
int cisSockApplyChanges(struct cis_sockinfo *oldtab, int nold, char *changes, int chlen, struct cis_sockinfo **newtab);

int cisNetdevPrepareChanges(struct cis_netdevinfo *oldtab, int nold, struct cis_netdevinfo *newtab, int nnew, char **changes);
int cisNetdevApplyChanges(struct cis_netdevinfo *oldtab, int nold, char *changes, int chlen, struct cis_netdevinfo **newtab);

int cisApplyChanges(int type, void *oldtab, int nold, char *changes, int chlen, void **newtab);

int cisSaveHeader(FILE *file);
int cisReadHeader(FILE *file, struct cis_header *header);

int cisSaveRecord(FILE *file, struct timeval time, int type, struct in_addr host, char *data, int block, int num, int to_save);
int cisReadRecord(FILE *file, struct record_info *rinfo);

int cisSaveHostinfo        (FILE *file, struct in_addr host, struct cis_hostinfo *data);
int cisSaveHostinfoChanges (FILE *file, struct in_addr host, char *changes, int length);

int cisSaveProcinfo    (FILE *file, struct in_addr host, struct cis_procinfo *data, int number);
int cisSaveProcChanges (FILE *file, struct in_addr host, char *changes, int length);

int cisSaveSockinfo    (FILE *file, struct in_addr host, struct cis_sockinfo *data, int number);
int cisSaveSockChanges (FILE *file, struct in_addr host, char *changes, int length);

int cisSaveNetdevinfo    (FILE *file, struct in_addr host, struct cis_netdevinfo *data, int number);
int cisSaveNetdevChanges (FILE *file, struct in_addr host, char *changes, int length);

int cisRecordInfo (FILE *file, struct record_info *rinfo);
int cisParseFile  (FILE *file, struct record_info **rec, void calback(int));

/* end of record processing functions */
#endif
