/*
 * Cluster Information Service - a monitoring system for Linux clusters
 * Copyright (C) 2000 Institute of Informatics, Slovak Academy of Sciences.
 * Written by Jan Astalos (astalos.ui@savba.sk)
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 */

#ifndef _cis_h
#define _cis_h

#if !defined(__KERNEL__)

#undef FALSE
#undef TRUE

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rpc/auth.h>

#include <net/if.h>
#include <sys/utsname.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <rpc/auth.h>

/*
 * RPC numbers.
 */
#define CIS_PROG           0x22233300
#define CIS_VER            1

/*
 * CIS server dispatch routines.
 */
/* Client calls */
enum {
        HOST_LIST = 1,
        HOST_INFO,
        HOST_INFO_UPDATE,
        PROCESS_LIST,
        SOCKET_LIST,
        NETDEV_LIST,
};

/* Monitor calls */
enum {
        PROCMON_INIT = 0x10,
        PROCMON_EXIT,
        PROCMON_FULL,
        PROCMON_NEW,
        PROCMON_CHANGES,
        PROCMON_CHANGES_ASYNC,
        PROCMON_SYSINFO,
};

enum {
        SOCKMON_INIT = 0x20,
        SOCKMON_EXIT,
        SOCKMON_FULL,
        SOCKMON_NEW,
        SOCKMON_CHANGES,
};

enum {
        NETMON_INIT = 0x30,
        NETMON_EXIT,
        NETMON_FULL,
        NETMON_NEW,
        NETMON_CHANGES,
};

/* Monitor types (0 is unused) */
enum {
        UNUSED,
        PROCMON,           /* Process monitor */
        SOCKMON,           /* Socket monitor */
        NETMON,            /* Network interface monitor (not implemented) */
        CIS_MONITOR_TYPES,
};

#define monitor_type(a)  ((0x70 & a) >> 4)
#define init_msg(a)      (!(0xf & a))

/*
 * Change codes.
 */
enum {
        PROC_NEW = 1,
        PROC_TERMINATED,
        PROC_UTIME,
        PROC_STIME,
        PROC_PRIORITY,
        PROC_RSS,
        PROC_VM,
        PROC_CMD,
        PROC_PPID,
        PROC_PCPU,
        
        SOCK_NEW,
        SOCK_CLOSED,
        SOCK_PID,
        SOCK_UID,
        SOCK_SENT,
        SOCK_RCVD,

        NETDEV_STARTED,
        NETDEV_STOPPED,
        NETDEV_SENT,
        NETDEV_RCVD,
        NETDEV_COLLISIONS,
};


/* Host states */
enum {
        HOST_AVAILABLE     = 0,
        HOST_NOT_AVAILABLE = -100,
        HOST_UNKNOWN       = -101,
};

#define CIS_PROCNUM  1
#define CIS_LOADS    (1<<1)
#define CIS_RAM      (1<<2)
#define CIS_SWAP     (1<<3)
#define CIS_CPU      (1<<4)
#define CIS_IDLE     (1<<5)
#define CIS_ALL      (-1)

#define CONSISTENT   1
#define TMP_FLAG     (1<<5)

/* Misc. */
#define MAX_ADDR_NUM       16 /* Should be enough in 'normal' circumstances */
#define MIN_USER_UID       500

#endif /* !__KERNEL__ && !_MONITOR_*/

#define CMD_LENGTH         16

#define OFFSET(str, el)  (int) &(((struct str *) 0)->el)

#define MON_DATALEN        (2040)

struct monitor_msg {
        unsigned int       time;
        unsigned char      tag;
        unsigned char      seqnum;
        unsigned char      flags;
        char               data[MON_DATALEN];
};

#define MSG_LAST           1<<1
#define MSG_FIRST          (1<<2)

struct monitor_reply {
        int                code;
        float              interval;
};

#define MON_BUFFLEN  (sizeof (struct monitor_msg))

enum {
        CMD_COMMENT,
        CMD_PERFTYPES,
        CMD_PERFVALUES,
	CMD_RELIABILITY,
	CMD_ACCEPTNEWHOST,
	CMD_TABLEN,
};

enum {
	REL_MINIMAL,
	REL_LOW,
	REL_AVERAGE,
	REL_HIGH,
	REL_MAXIMAL,
	REL_TABLEN,
};

struct cis_uname
{
        char *sysname;
        char *nodename;
        char *release;
        char *version;
        char *machine;
};

struct cis_hostinfo
{
        /* Dynamic information */
        unsigned short procnum;
        unsigned long  totalswap;
        unsigned long  freeswap;
        unsigned long  freeram;
        unsigned long  sharedram;
        unsigned long  bufferram;
        unsigned long  cachedram;
        float          loads[3];
        unsigned long  idle;       /* idle time from host startup in jiffies */
                                   /* 1 jiffie = 1/ct_per_sec [s] */

        unsigned short CPU_available;

        /* Static information */
        char           status;
	unsigned char  reliability;
        unsigned short ct_per_sec; /* clock ticks per second */
        struct in_addr addr;
	unsigned long  totalram;
	float          performance;
};

#define HOSTINFO_LEN (OFFSET(cis_hostinfo, CPU_available))

struct cis_procinfo
{
        unsigned int   pid;
        unsigned long  start_time;            /* start time (GMT)          [s] */
        unsigned int   uid;
        unsigned int   ppid;

        char           cmd[CMD_LENGTH];

        unsigned long  rss;
        unsigned long  vm;

        long           priority;
        
        unsigned long  stime;                 /* time spent in system mode [s] */
        unsigned long  utime;                 /* time spent in user mode   [s] */
        
        short          pCPU;                  /* real CPU time consumption */
                                              /* 9999 -> 99.99% */

        /* Internal CIS fields. Not filled in client calls */
        unsigned char  flags;
        struct cis_procinfo *prev, *next;
        struct cis_procinfo *children;
        void *priv;
};
#define PROCINFO_LEN  (OFFSET(cis_procinfo, flags))

struct cis_sockinfo
{
        struct in_addr saddr;
        struct in_addr daddr;

        unsigned short sport;
        unsigned short dport;

        short          type;

        unsigned short uid;
        unsigned int   pid;
        
        unsigned long sent;
        unsigned long rcvd;

        /* Internal CIS fields. Not filled in client calls */
        unsigned char flags;
        struct cis_sockinfo *prev, *next;
        void *priv;
};
#define SOCKINFO_LEN     (OFFSET(cis_sockinfo, flags))
#define SOCKINFO_HDRLEN  (OFFSET(cis_sockinfo, uid))

struct cis_netdevinfo
{
        char name[IFNAMSIZ];

        unsigned char id;
        unsigned char status;

        unsigned long rx_bytes;
        unsigned long tx_bytes;

        unsigned long collisions;

        /* Internal CIS fields. Not filled in client calls */
        unsigned char flags;
        struct cis_netdevinfo *prev, *next;
        void *priv;
};

#define NETDEVINFO_LEN  (OFFSET(cis_netdevinfo, flags))


#define usectotval(A,B) { A.tv_sec  = (B) / 1000000L; \
        A.tv_usec = (B) % 1000000L; }

#define tvaltousec(A)   A.tv_sec * 1000000L + A.tv_usec

#define sectotval(A,B) { A.tv_sec  = (int) (B) ; \
        A.tv_usec = (int) (((B) - (int) (B)) * 1000000L); }

#define tvaltosec(A)    (A.tv_sec + (A.tv_usec / 1e6))

#define tvaludiff(A,B)    ((A.tv_sec  - B.tv_sec) * 1000000L + \
        A.tv_usec - B.tv_usec)

#define list_insert(list, item)                        \
        {                                              \
                if (list)                              \
                        list->prev = item;             \
                item->next = list;                     \
                item->prev = NULL;                     \
                list = item;                           \
        }

#define list_remove(list, item)                        \
        {                                              \
                if (item->prev)                        \
                        item->prev->next = item->next; \
                else                                   \
                        list = item->next;             \
                                                       \
                if (item->next)                        \
                        item->next->prev = item->prev; \
        }

#endif
